package io.k8s.api.rbac.v1

import dev.hnaderi.k8s.utils._

/** PolicyRule holds information that describes a policy rule, but does not contain information about who the rule applies to or which namespace the rule applies to. */
final case class PolicyRule(
  verbs : Seq[String],
  apiGroups : Option[Seq[String]] = None,
  resources : Option[Seq[String]] = None,
  resourceNames : Option[Seq[String]] = None,
  nonResourceURLs : Option[Seq[String]] = None
) {

  /** Returns a new data with verbs set to new value */
  def withVerbs(value: Seq[String]) : PolicyRule = copy(verbs = value)
  /** Appends new values to verbs */
  def addVerbs(newValues: String*) : PolicyRule = copy(verbs = verbs ++ newValues)
  /** transforms verbs to result of function */
  def mapVerbs(f: Seq[String] => Seq[String]) : PolicyRule = copy(verbs = f(verbs))

  /** Returns a new data with apiGroups set to new value */
  def withApiGroups(value: Seq[String]) : PolicyRule = copy(apiGroups = Some(value))
  /** Appends new values to apiGroups */
  def addApiGroups(newValues: String*) : PolicyRule = copy(apiGroups = Some(apiGroups.fold(newValues)(_ ++ newValues)))
  /** if apiGroups has a value, transforms to the result of function*/
  def mapApiGroups(f: Seq[String] => Seq[String]) : PolicyRule = copy(apiGroups = apiGroups.map(f))

  /** Returns a new data with resources set to new value */
  def withResources(value: Seq[String]) : PolicyRule = copy(resources = Some(value))
  /** Appends new values to resources */
  def addResources(newValues: String*) : PolicyRule = copy(resources = Some(resources.fold(newValues)(_ ++ newValues)))
  /** if resources has a value, transforms to the result of function*/
  def mapResources(f: Seq[String] => Seq[String]) : PolicyRule = copy(resources = resources.map(f))

  /** Returns a new data with resourceNames set to new value */
  def withResourceNames(value: Seq[String]) : PolicyRule = copy(resourceNames = Some(value))
  /** Appends new values to resourceNames */
  def addResourceNames(newValues: String*) : PolicyRule = copy(resourceNames = Some(resourceNames.fold(newValues)(_ ++ newValues)))
  /** if resourceNames has a value, transforms to the result of function*/
  def mapResourceNames(f: Seq[String] => Seq[String]) : PolicyRule = copy(resourceNames = resourceNames.map(f))

  /** Returns a new data with nonResourceURLs set to new value */
  def withNonResourceURLs(value: Seq[String]) : PolicyRule = copy(nonResourceURLs = Some(value))
  /** Appends new values to nonResourceURLs */
  def addNonResourceURLs(newValues: String*) : PolicyRule = copy(nonResourceURLs = Some(nonResourceURLs.fold(newValues)(_ ++ newValues)))
  /** if nonResourceURLs has a value, transforms to the result of function*/
  def mapNonResourceURLs(f: Seq[String] => Seq[String]) : PolicyRule = copy(nonResourceURLs = nonResourceURLs.map(f))
}

object PolicyRule {

    implicit val encoder : Encoder[io.k8s.api.rbac.v1.PolicyRule] = new Encoder[io.k8s.api.rbac.v1.PolicyRule] {
        def apply[T : Builder](o: io.k8s.api.rbac.v1.PolicyRule) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("verbs", o.verbs)
            .write("apiGroups", o.apiGroups)
            .write("resources", o.resources)
            .write("resourceNames", o.resourceNames)
            .write("nonResourceURLs", o.nonResourceURLs)
            .build
        }
    }

    implicit val decoder: Decoder[PolicyRule] = new Decoder[PolicyRule] {
      def apply[T : Reader](t: T): Either[String, PolicyRule] = for {
          obj <- ObjectReader(t)
          verbs <- obj.read[Seq[String]]("verbs")
          apiGroups <- obj.readOpt[Seq[String]]("apiGroups")
          resources <- obj.readOpt[Seq[String]]("resources")
          resourceNames <- obj.readOpt[Seq[String]]("resourceNames")
          nonResourceURLs <- obj.readOpt[Seq[String]]("nonResourceURLs")
      } yield PolicyRule (
          verbs = verbs,
          apiGroups = apiGroups,
          resources = resources,
          resourceNames = resourceNames,
          nonResourceURLs = nonResourceURLs
        )
    }
}

