package io.k8s.api.admissionregistration.v1

import dev.hnaderi.k8s.utils._

/** RuleWithOperations is a tuple of Operations and Resources. It is recommended to make sure that all the tuple expansions are valid. */
final case class RuleWithOperations(
  operations : Option[Seq[String]] = None,
  apiVersions : Option[Seq[String]] = None,
  scope : Option[String] = None,
  apiGroups : Option[Seq[String]] = None,
  resources : Option[Seq[String]] = None
) {

  /** Returns a new data with operations set to new value */
  def withOperations(value: Seq[String]) : RuleWithOperations = copy(operations = Some(value))
  /** Appends new values to operations */
  def addOperations(newValues: String*) : RuleWithOperations = copy(operations = Some(operations.fold(newValues)(_ ++ newValues)))
  /** if operations has a value, transforms to the result of function*/
  def mapOperations(f: Seq[String] => Seq[String]) : RuleWithOperations = copy(operations = operations.map(f))

  /** Returns a new data with apiVersions set to new value */
  def withApiVersions(value: Seq[String]) : RuleWithOperations = copy(apiVersions = Some(value))
  /** Appends new values to apiVersions */
  def addApiVersions(newValues: String*) : RuleWithOperations = copy(apiVersions = Some(apiVersions.fold(newValues)(_ ++ newValues)))
  /** if apiVersions has a value, transforms to the result of function*/
  def mapApiVersions(f: Seq[String] => Seq[String]) : RuleWithOperations = copy(apiVersions = apiVersions.map(f))

  /** Returns a new data with scope set to new value */
  def withScope(value: String) : RuleWithOperations = copy(scope = Some(value))
  /** if scope has a value, transforms to the result of function*/
  def mapScope(f: String => String) : RuleWithOperations = copy(scope = scope.map(f))

  /** Returns a new data with apiGroups set to new value */
  def withApiGroups(value: Seq[String]) : RuleWithOperations = copy(apiGroups = Some(value))
  /** Appends new values to apiGroups */
  def addApiGroups(newValues: String*) : RuleWithOperations = copy(apiGroups = Some(apiGroups.fold(newValues)(_ ++ newValues)))
  /** if apiGroups has a value, transforms to the result of function*/
  def mapApiGroups(f: Seq[String] => Seq[String]) : RuleWithOperations = copy(apiGroups = apiGroups.map(f))

  /** Returns a new data with resources set to new value */
  def withResources(value: Seq[String]) : RuleWithOperations = copy(resources = Some(value))
  /** Appends new values to resources */
  def addResources(newValues: String*) : RuleWithOperations = copy(resources = Some(resources.fold(newValues)(_ ++ newValues)))
  /** if resources has a value, transforms to the result of function*/
  def mapResources(f: Seq[String] => Seq[String]) : RuleWithOperations = copy(resources = resources.map(f))
}

object RuleWithOperations {

    implicit val encoder : Encoder[io.k8s.api.admissionregistration.v1.RuleWithOperations] = new Encoder[io.k8s.api.admissionregistration.v1.RuleWithOperations] {
        def apply[T : Builder](o: io.k8s.api.admissionregistration.v1.RuleWithOperations) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("operations", o.operations)
            .write("apiVersions", o.apiVersions)
            .write("scope", o.scope)
            .write("apiGroups", o.apiGroups)
            .write("resources", o.resources)
            .build
        }
    }

    implicit val decoder: Decoder[RuleWithOperations] = new Decoder[RuleWithOperations] {
      def apply[T : Reader](t: T): Either[String, RuleWithOperations] = for {
          obj <- ObjectReader(t)
          operations <- obj.readOpt[Seq[String]]("operations")
          apiVersions <- obj.readOpt[Seq[String]]("apiVersions")
          scope <- obj.readOpt[String]("scope")
          apiGroups <- obj.readOpt[Seq[String]]("apiGroups")
          resources <- obj.readOpt[Seq[String]]("resources")
      } yield RuleWithOperations (
          operations = operations,
          apiVersions = apiVersions,
          scope = scope,
          apiGroups = apiGroups,
          resources = resources
        )
    }
}

