package io.k8s.api.authorization.v1

import dev.hnaderi.k8s.utils._

/** ResourceRule is the list of actions the subject is allowed to perform on resources. The list ordering isn't significant, may contain duplicates, and possibly be incomplete. */
final case class ResourceRule(
  verbs : Seq[String],
  apiGroups : Option[Seq[String]] = None,
  resourceNames : Option[Seq[String]] = None,
  resources : Option[Seq[String]] = None
) {

  /** Returns a new data with verbs set to new value */
  def withVerbs(value: Seq[String]) : ResourceRule = copy(verbs = value)
  /** Appends new values to verbs */
  def addVerbs(newValues: String*) : ResourceRule = copy(verbs = verbs ++ newValues)
  /** transforms verbs to result of function */
  def mapVerbs(f: Seq[String] => Seq[String]) : ResourceRule = copy(verbs = f(verbs))

  /** Returns a new data with apiGroups set to new value */
  def withApiGroups(value: Seq[String]) : ResourceRule = copy(apiGroups = Some(value))
  /** Appends new values to apiGroups */
  def addApiGroups(newValues: String*) : ResourceRule = copy(apiGroups = Some(apiGroups.fold(newValues)(_ ++ newValues)))
  /** if apiGroups has a value, transforms to the result of function*/
  def mapApiGroups(f: Seq[String] => Seq[String]) : ResourceRule = copy(apiGroups = apiGroups.map(f))

  /** Returns a new data with resourceNames set to new value */
  def withResourceNames(value: Seq[String]) : ResourceRule = copy(resourceNames = Some(value))
  /** Appends new values to resourceNames */
  def addResourceNames(newValues: String*) : ResourceRule = copy(resourceNames = Some(resourceNames.fold(newValues)(_ ++ newValues)))
  /** if resourceNames has a value, transforms to the result of function*/
  def mapResourceNames(f: Seq[String] => Seq[String]) : ResourceRule = copy(resourceNames = resourceNames.map(f))

  /** Returns a new data with resources set to new value */
  def withResources(value: Seq[String]) : ResourceRule = copy(resources = Some(value))
  /** Appends new values to resources */
  def addResources(newValues: String*) : ResourceRule = copy(resources = Some(resources.fold(newValues)(_ ++ newValues)))
  /** if resources has a value, transforms to the result of function*/
  def mapResources(f: Seq[String] => Seq[String]) : ResourceRule = copy(resources = resources.map(f))
}

object ResourceRule {

    implicit val encoder : Encoder[io.k8s.api.authorization.v1.ResourceRule] = new Encoder[io.k8s.api.authorization.v1.ResourceRule] {
        def apply[T : Builder](o: io.k8s.api.authorization.v1.ResourceRule) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("verbs", o.verbs)
            .write("apiGroups", o.apiGroups)
            .write("resourceNames", o.resourceNames)
            .write("resources", o.resources)
            .build
        }
    }

    implicit val decoder: Decoder[ResourceRule] = new Decoder[ResourceRule] {
      def apply[T : Reader](t: T): Either[String, ResourceRule] = for {
          obj <- ObjectReader(t)
          verbs <- obj.read[Seq[String]]("verbs")
          apiGroups <- obj.readOpt[Seq[String]]("apiGroups")
          resourceNames <- obj.readOpt[Seq[String]]("resourceNames")
          resources <- obj.readOpt[Seq[String]]("resources")
      } yield ResourceRule (
          verbs = verbs,
          apiGroups = apiGroups,
          resourceNames = resourceNames,
          resources = resources
        )
    }
}

