package io.k8s.api.core.v1

import dev.hnaderi.k8s._
import dev.hnaderi.k8s.utils._

/** ConfigMap holds configuration data for pods to consume. */
final case class ConfigMap(
  data : Option[Map[String, String]] = None,
  immutable : Option[Boolean] = None,
  binaryData : Option[Map[String, String]] = None,
  metadata : Option[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta] = None
) extends KObject {
  protected val _resourceKind = ResourceKind("", "ConfigMap", "v1")


  /** Returns a new data with data set to new value */
  def withData(value: Map[String, String]) : ConfigMap = copy(data = Some(value))
  /** Adds new values to data */
  def addData(newValues: (String, String)*) : ConfigMap = copy(data = Some(data.fold(newValues.toMap)(_ ++ newValues)))
  /** if data has a value, transforms to the result of function*/
  def mapData(f: Map[String, String] => Map[String, String]) : ConfigMap = copy(data = data.map(f))

  /** Returns a new data with immutable set to new value */
  def withImmutable(value: Boolean) : ConfigMap = copy(immutable = Some(value))
  /** if immutable has a value, transforms to the result of function*/
  def mapImmutable(f: Boolean => Boolean) : ConfigMap = copy(immutable = immutable.map(f))

  /** Returns a new data with binaryData set to new value */
  def withBinaryData(value: Map[String, String]) : ConfigMap = copy(binaryData = Some(value))
  /** Adds new values to binaryData */
  def addBinaryData(newValues: (String, String)*) : ConfigMap = copy(binaryData = Some(binaryData.fold(newValues.toMap)(_ ++ newValues)))
  /** if binaryData has a value, transforms to the result of function*/
  def mapBinaryData(f: Map[String, String] => Map[String, String]) : ConfigMap = copy(binaryData = binaryData.map(f))

  /** Returns a new data with metadata set to new value */
  def withMetadata(value: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : ConfigMap = copy(metadata = Some(value))
  /** if metadata has a value, transforms to the result of function*/
  def mapMetadata(f: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta => io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : ConfigMap = copy(metadata = metadata.map(f))

  override def foldTo[T : Builder] : T = ConfigMap.encoder.apply(this)
}

object ConfigMap {

    implicit val encoder : Encoder[io.k8s.api.core.v1.ConfigMap] = new Encoder[io.k8s.api.core.v1.ConfigMap] {
        def apply[T : Builder](o: io.k8s.api.core.v1.ConfigMap) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("data", o.data)
            .write("immutable", o.immutable)
            .write("binaryData", o.binaryData)
            .write("metadata", o.metadata)
            .write("kind", o.kind)
            .write("apiVersion", o.apiVersion)
            .build
        }
    }

    implicit val decoder: Decoder[ConfigMap] = new Decoder[ConfigMap] {
      def apply[T : Reader](t: T): Either[String, ConfigMap] = for {
          obj <- ObjectReader(t)
          data <- obj.readOpt[Map[String, String]]("data")
          immutable <- obj.readOpt[Boolean]("immutable")
          binaryData <- obj.readOpt[Map[String, String]]("binaryData")
          metadata <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta]("metadata")
      } yield ConfigMap (
          data = data,
          immutable = immutable,
          binaryData = binaryData,
          metadata = metadata
        )
    }
}

