package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** NodeStatus is information about the current status of a node. */
final case class NodeStatus(
  volumesInUse : Option[Seq[String]] = None,
  conditions : Option[Seq[io.k8s.api.core.v1.NodeCondition]] = None,
  nodeInfo : Option[io.k8s.api.core.v1.NodeSystemInfo] = None,
  volumesAttached : Option[Seq[io.k8s.api.core.v1.AttachedVolume]] = None,
  phase : Option[String] = None,
  config : Option[io.k8s.api.core.v1.NodeConfigStatus] = None,
  allocatable : Option[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]] = None,
  images : Option[Seq[io.k8s.api.core.v1.ContainerImage]] = None,
  daemonEndpoints : Option[io.k8s.api.core.v1.NodeDaemonEndpoints] = None,
  addresses : Option[Seq[io.k8s.api.core.v1.NodeAddress]] = None,
  capacity : Option[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]] = None
) {

  /** Returns a new data with volumesInUse set to new value */
  def withVolumesInUse(value: Seq[String]) : NodeStatus = copy(volumesInUse = Some(value))
  /** Appends new values to volumesInUse */
  def addVolumesInUse(newValues: String*) : NodeStatus = copy(volumesInUse = Some(volumesInUse.fold(newValues)(_ ++ newValues)))
  /** if volumesInUse has a value, transforms to the result of function*/
  def mapVolumesInUse(f: Seq[String] => Seq[String]) : NodeStatus = copy(volumesInUse = volumesInUse.map(f))

  /** Returns a new data with conditions set to new value */
  def withConditions(value: Seq[io.k8s.api.core.v1.NodeCondition]) : NodeStatus = copy(conditions = Some(value))
  /** Appends new values to conditions */
  def addConditions(newValues: io.k8s.api.core.v1.NodeCondition*) : NodeStatus = copy(conditions = Some(conditions.fold(newValues)(_ ++ newValues)))
  /** if conditions has a value, transforms to the result of function*/
  def mapConditions(f: Seq[io.k8s.api.core.v1.NodeCondition] => Seq[io.k8s.api.core.v1.NodeCondition]) : NodeStatus = copy(conditions = conditions.map(f))

  /** Returns a new data with nodeInfo set to new value */
  def withNodeInfo(value: io.k8s.api.core.v1.NodeSystemInfo) : NodeStatus = copy(nodeInfo = Some(value))
  /** if nodeInfo has a value, transforms to the result of function*/
  def mapNodeInfo(f: io.k8s.api.core.v1.NodeSystemInfo => io.k8s.api.core.v1.NodeSystemInfo) : NodeStatus = copy(nodeInfo = nodeInfo.map(f))

  /** Returns a new data with volumesAttached set to new value */
  def withVolumesAttached(value: Seq[io.k8s.api.core.v1.AttachedVolume]) : NodeStatus = copy(volumesAttached = Some(value))
  /** Appends new values to volumesAttached */
  def addVolumesAttached(newValues: io.k8s.api.core.v1.AttachedVolume*) : NodeStatus = copy(volumesAttached = Some(volumesAttached.fold(newValues)(_ ++ newValues)))
  /** if volumesAttached has a value, transforms to the result of function*/
  def mapVolumesAttached(f: Seq[io.k8s.api.core.v1.AttachedVolume] => Seq[io.k8s.api.core.v1.AttachedVolume]) : NodeStatus = copy(volumesAttached = volumesAttached.map(f))

  /** Returns a new data with phase set to new value */
  def withPhase(value: String) : NodeStatus = copy(phase = Some(value))
  /** if phase has a value, transforms to the result of function*/
  def mapPhase(f: String => String) : NodeStatus = copy(phase = phase.map(f))

  /** Returns a new data with config set to new value */
  def withConfig(value: io.k8s.api.core.v1.NodeConfigStatus) : NodeStatus = copy(config = Some(value))
  /** if config has a value, transforms to the result of function*/
  def mapConfig(f: io.k8s.api.core.v1.NodeConfigStatus => io.k8s.api.core.v1.NodeConfigStatus) : NodeStatus = copy(config = config.map(f))

  /** Returns a new data with allocatable set to new value */
  def withAllocatable(value: Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]) : NodeStatus = copy(allocatable = Some(value))
  /** Adds new values to allocatable */
  def addAllocatable(newValues: (String, io.k8s.apimachinery.pkg.api.resource.Quantity)*) : NodeStatus = copy(allocatable = Some(allocatable.fold(newValues.toMap)(_ ++ newValues)))
  /** if allocatable has a value, transforms to the result of function*/
  def mapAllocatable(f: Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity] => Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]) : NodeStatus = copy(allocatable = allocatable.map(f))

  /** Returns a new data with images set to new value */
  def withImages(value: Seq[io.k8s.api.core.v1.ContainerImage]) : NodeStatus = copy(images = Some(value))
  /** Appends new values to images */
  def addImages(newValues: io.k8s.api.core.v1.ContainerImage*) : NodeStatus = copy(images = Some(images.fold(newValues)(_ ++ newValues)))
  /** if images has a value, transforms to the result of function*/
  def mapImages(f: Seq[io.k8s.api.core.v1.ContainerImage] => Seq[io.k8s.api.core.v1.ContainerImage]) : NodeStatus = copy(images = images.map(f))

  /** Returns a new data with daemonEndpoints set to new value */
  def withDaemonEndpoints(value: io.k8s.api.core.v1.NodeDaemonEndpoints) : NodeStatus = copy(daemonEndpoints = Some(value))
  /** if daemonEndpoints has a value, transforms to the result of function*/
  def mapDaemonEndpoints(f: io.k8s.api.core.v1.NodeDaemonEndpoints => io.k8s.api.core.v1.NodeDaemonEndpoints) : NodeStatus = copy(daemonEndpoints = daemonEndpoints.map(f))

  /** Returns a new data with addresses set to new value */
  def withAddresses(value: Seq[io.k8s.api.core.v1.NodeAddress]) : NodeStatus = copy(addresses = Some(value))
  /** Appends new values to addresses */
  def addAddresses(newValues: io.k8s.api.core.v1.NodeAddress*) : NodeStatus = copy(addresses = Some(addresses.fold(newValues)(_ ++ newValues)))
  /** if addresses has a value, transforms to the result of function*/
  def mapAddresses(f: Seq[io.k8s.api.core.v1.NodeAddress] => Seq[io.k8s.api.core.v1.NodeAddress]) : NodeStatus = copy(addresses = addresses.map(f))

  /** Returns a new data with capacity set to new value */
  def withCapacity(value: Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]) : NodeStatus = copy(capacity = Some(value))
  /** Adds new values to capacity */
  def addCapacity(newValues: (String, io.k8s.apimachinery.pkg.api.resource.Quantity)*) : NodeStatus = copy(capacity = Some(capacity.fold(newValues.toMap)(_ ++ newValues)))
  /** if capacity has a value, transforms to the result of function*/
  def mapCapacity(f: Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity] => Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]) : NodeStatus = copy(capacity = capacity.map(f))
}

object NodeStatus {

    implicit val encoder : Encoder[io.k8s.api.core.v1.NodeStatus] = new Encoder[io.k8s.api.core.v1.NodeStatus] {
        def apply[T : Builder](o: io.k8s.api.core.v1.NodeStatus) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("volumesInUse", o.volumesInUse)
            .write("conditions", o.conditions)
            .write("nodeInfo", o.nodeInfo)
            .write("volumesAttached", o.volumesAttached)
            .write("phase", o.phase)
            .write("config", o.config)
            .write("allocatable", o.allocatable)
            .write("images", o.images)
            .write("daemonEndpoints", o.daemonEndpoints)
            .write("addresses", o.addresses)
            .write("capacity", o.capacity)
            .build
        }
    }

    implicit val decoder: Decoder[NodeStatus] = new Decoder[NodeStatus] {
      def apply[T : Reader](t: T): Either[String, NodeStatus] = for {
          obj <- ObjectReader(t)
          volumesInUse <- obj.readOpt[Seq[String]]("volumesInUse")
          conditions <- obj.readOpt[Seq[io.k8s.api.core.v1.NodeCondition]]("conditions")
          nodeInfo <- obj.readOpt[io.k8s.api.core.v1.NodeSystemInfo]("nodeInfo")
          volumesAttached <- obj.readOpt[Seq[io.k8s.api.core.v1.AttachedVolume]]("volumesAttached")
          phase <- obj.readOpt[String]("phase")
          config <- obj.readOpt[io.k8s.api.core.v1.NodeConfigStatus]("config")
          allocatable <- obj.readOpt[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]]("allocatable")
          images <- obj.readOpt[Seq[io.k8s.api.core.v1.ContainerImage]]("images")
          daemonEndpoints <- obj.readOpt[io.k8s.api.core.v1.NodeDaemonEndpoints]("daemonEndpoints")
          addresses <- obj.readOpt[Seq[io.k8s.api.core.v1.NodeAddress]]("addresses")
          capacity <- obj.readOpt[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]]("capacity")
      } yield NodeStatus (
          volumesInUse = volumesInUse,
          conditions = conditions,
          nodeInfo = nodeInfo,
          volumesAttached = volumesAttached,
          phase = phase,
          config = config,
          allocatable = allocatable,
          images = images,
          daemonEndpoints = daemonEndpoints,
          addresses = addresses,
          capacity = capacity
        )
    }
}

