package io.k8s.api.storage.v1

import dev.hnaderi.k8s.utils._

/** TokenRequest contains parameters of a service account token. */
final case class TokenRequest(
  audience : String,
  expirationSeconds : Option[Int] = None
) {

  /** Returns a new data with audience set to new value */
  def withAudience(value: String) : TokenRequest = copy(audience = value)
  /** transforms audience to result of function */
  def mapAudience(f: String => String) : TokenRequest = copy(audience = f(audience))

  /** Returns a new data with expirationSeconds set to new value */
  def withExpirationSeconds(value: Int) : TokenRequest = copy(expirationSeconds = Some(value))
  /** if expirationSeconds has a value, transforms to the result of function*/
  def mapExpirationSeconds(f: Int => Int) : TokenRequest = copy(expirationSeconds = expirationSeconds.map(f))
}

object TokenRequest {

    implicit val encoder : Encoder[io.k8s.api.storage.v1.TokenRequest] = new Encoder[io.k8s.api.storage.v1.TokenRequest] {
        def apply[T : Builder](o: io.k8s.api.storage.v1.TokenRequest) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("audience", o.audience)
            .write("expirationSeconds", o.expirationSeconds)
            .build
        }
    }

    implicit val decoder: Decoder[TokenRequest] = new Decoder[TokenRequest] {
      def apply[T : Reader](t: T): Either[String, TokenRequest] = for {
          obj <- ObjectReader(t)
          audience <- obj.read[String]("audience")
          expirationSeconds <- obj.readOpt[Int]("expirationSeconds")
      } yield TokenRequest (
          audience = audience,
          expirationSeconds = expirationSeconds
        )
    }
}

