package io.k8s.api.storage.v1beta1

import dev.hnaderi.k8s._
import dev.hnaderi.k8s.utils._

/** CSIStorageCapacity stores the result of one CSI GetCapacity call. For a given StorageClass, this describes the available capacity in a particular topology segment.  This can be used when considering where to instantiate new PersistentVolumes.

For example this can express things like: - StorageClass "standard" has "1234 GiB" available in "topology.kubernetes.io/zone=us-east1" - StorageClass "localssd" has "10 GiB" available in "kubernetes.io/hostname=knode-abc123"

The following three cases all imply that no capacity is available for a certain combination: - no object exists with suitable topology and storage class name - such an object exists, but the capacity is unset - such an object exists, but the capacity is zero

The producer of these objects can decide which approach is more suitable.

They are consumed by the kube-scheduler when a CSI driver opts into capacity-aware scheduling with CSIDriverSpec.StorageCapacity. The scheduler compares the MaximumVolumeSize against the requested size of pending volumes to filter out unsuitable nodes. If MaximumVolumeSize is unset, it falls back to a comparison against the less precise Capacity. If that is also unset, the scheduler assumes that capacity is insufficient and tries some other node. */
final case class CSIStorageCapacity(
  storageClassName : String,
  maximumVolumeSize : Option[io.k8s.apimachinery.pkg.api.resource.Quantity] = None,
  nodeTopology : Option[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector] = None,
  metadata : Option[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta] = None,
  capacity : Option[io.k8s.apimachinery.pkg.api.resource.Quantity] = None
) extends KObject {
  protected val _resourceKind = ResourceKind("storage.k8s.io", "CSIStorageCapacity", "v1beta1")


  /** Returns a new data with storageClassName set to new value */
  def withStorageClassName(value: String) : CSIStorageCapacity = copy(storageClassName = value)
  /** transforms storageClassName to result of function */
  def mapStorageClassName(f: String => String) : CSIStorageCapacity = copy(storageClassName = f(storageClassName))

  /** Returns a new data with maximumVolumeSize set to new value */
  def withMaximumVolumeSize(value: io.k8s.apimachinery.pkg.api.resource.Quantity) : CSIStorageCapacity = copy(maximumVolumeSize = Some(value))
  /** if maximumVolumeSize has a value, transforms to the result of function*/
  def mapMaximumVolumeSize(f: io.k8s.apimachinery.pkg.api.resource.Quantity => io.k8s.apimachinery.pkg.api.resource.Quantity) : CSIStorageCapacity = copy(maximumVolumeSize = maximumVolumeSize.map(f))

  /** Returns a new data with nodeTopology set to new value */
  def withNodeTopology(value: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : CSIStorageCapacity = copy(nodeTopology = Some(value))
  /** if nodeTopology has a value, transforms to the result of function*/
  def mapNodeTopology(f: io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector => io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector) : CSIStorageCapacity = copy(nodeTopology = nodeTopology.map(f))

  /** Returns a new data with metadata set to new value */
  def withMetadata(value: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : CSIStorageCapacity = copy(metadata = Some(value))
  /** if metadata has a value, transforms to the result of function*/
  def mapMetadata(f: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta => io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : CSIStorageCapacity = copy(metadata = metadata.map(f))

  /** Returns a new data with capacity set to new value */
  def withCapacity(value: io.k8s.apimachinery.pkg.api.resource.Quantity) : CSIStorageCapacity = copy(capacity = Some(value))
  /** if capacity has a value, transforms to the result of function*/
  def mapCapacity(f: io.k8s.apimachinery.pkg.api.resource.Quantity => io.k8s.apimachinery.pkg.api.resource.Quantity) : CSIStorageCapacity = copy(capacity = capacity.map(f))

  override def foldTo[T : Builder] : T = CSIStorageCapacity.encoder.apply(this)
}

object CSIStorageCapacity {

    implicit val encoder : Encoder[io.k8s.api.storage.v1beta1.CSIStorageCapacity] = new Encoder[io.k8s.api.storage.v1beta1.CSIStorageCapacity] {
        def apply[T : Builder](o: io.k8s.api.storage.v1beta1.CSIStorageCapacity) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("storageClassName", o.storageClassName)
            .write("maximumVolumeSize", o.maximumVolumeSize)
            .write("nodeTopology", o.nodeTopology)
            .write("metadata", o.metadata)
            .write("capacity", o.capacity)
            .write("kind", o.kind)
            .write("apiVersion", o.apiVersion)
            .build
        }
    }

    implicit val decoder: Decoder[CSIStorageCapacity] = new Decoder[CSIStorageCapacity] {
      def apply[T : Reader](t: T): Either[String, CSIStorageCapacity] = for {
          obj <- ObjectReader(t)
          storageClassName <- obj.read[String]("storageClassName")
          maximumVolumeSize <- obj.readOpt[io.k8s.apimachinery.pkg.api.resource.Quantity]("maximumVolumeSize")
          nodeTopology <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.LabelSelector]("nodeTopology")
          metadata <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta]("metadata")
          capacity <- obj.readOpt[io.k8s.apimachinery.pkg.api.resource.Quantity]("capacity")
      } yield CSIStorageCapacity (
          storageClassName = storageClassName,
          maximumVolumeSize = maximumVolumeSize,
          nodeTopology = nodeTopology,
          metadata = metadata,
          capacity = capacity
        )
    }
}

