/*
 *    Copyright [2020] Maximilian Hippler <hello@maximilian.dev>
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package dev.maximilian.util.database

import org.jetbrains.exposed.dao.id.EntityID
import org.jetbrains.exposed.sql.Database
import org.jetbrains.exposed.sql.SchemaUtils
import org.jetbrains.exposed.sql.deleteAll
import org.jetbrains.exposed.sql.deleteWhere
import org.jetbrains.exposed.sql.insert
import org.jetbrains.exposed.sql.select
import org.jetbrains.exposed.sql.selectAll
import org.jetbrains.exposed.sql.transactions.transaction
import org.jetbrains.exposed.sql.update

/**
 * A mutable map with a exposed database as underlying store.
 * All read/write operations are directly on the database and nothing is cached (except the iterator)
 */
class DatabaseProperties(private val db: Database, tableName: String = "properties") : MutableMap<String, String> {
    private val propertyTable = PropertyTable(tableName)

    init {
        transaction(db) {
            SchemaUtils.createMissingTablesAndColumns(propertyTable)
        }
    }

    override val size: Int = transaction(db) { propertyTable.slice(propertyTable.id).selectAll().count() }.toInt()

    override fun containsKey(key: String): Boolean = transaction(db) {
        propertyTable.select { propertyTable.id eq key }.count() != 0L
    }

    override fun containsValue(value: String): Boolean = transaction(db) {
        propertyTable.select { propertyTable.value eq value }.count() != 0L
    }

    override fun get(key: String): String? = transaction(db) {
        propertyTable.select { propertyTable.id eq key }.firstOrNull()?.get(propertyTable.value)
    }

    override fun isEmpty(): Boolean = size == 0

    override val entries: MutableSet<MutableMap.MutableEntry<String, String>> =
        DatabasePropertiesEntrySet(db, this, propertyTable)

    override val keys: MutableSet<String> = object : AbstractMutableSet<String>() {
        override val size: Int = this@DatabaseProperties.size
        override fun add(element: String): Boolean = throw UnsupportedOperationException()
        override fun iterator(): MutableIterator<String> = MutableDelegateIterator(entries.iterator()) { it.key }
    }

    override val values: MutableCollection<String> = object : AbstractMutableCollection<String>() {
        override val size: Int = this@DatabaseProperties.size
        override fun add(element: String): Boolean = throw UnsupportedOperationException()
        override fun iterator(): MutableIterator<String> = MutableDelegateIterator(entries.iterator()) { it.value }
    }

    override fun clear() {
        transaction(db) { propertyTable.deleteAll() }
    }

    override fun put(key: String, value: String): String? = transaction(db) {
        val old = get(key)

        if (old == null) {
            propertyTable.insert {
                it[propertyTable.id] = EntityID(key, propertyTable)
                it[propertyTable.value] = value
            }
        } else {
            propertyTable.update({ propertyTable.id eq key }) {
                it[propertyTable.id] = EntityID(key, propertyTable)
                it[propertyTable.value] = value
            }
        }

        old
    }

    override fun putAll(from: Map<out String, String>): Unit = transaction(db) {
        val databaseKeys = propertyTable.slice(propertyTable.id).selectAll().map { it[propertyTable.id].value }

        // Create all not existing
        from.minus(databaseKeys).forEach { entry ->
            propertyTable.insert {
                it[propertyTable.id] = EntityID(entry.key, propertyTable)
                it[value] = entry.value
            }
        }

        // Update all existing
        databaseKeys.minus(from.keys).associateWith { from[it] }.forEach { entry ->
            if (entry.value != null) {
                propertyTable.update({ propertyTable.id eq entry.key }) {
                    it[propertyTable.id] = EntityID(entry.key, propertyTable)
                    it[propertyTable.value] = entry.value!!
                }
            }
        }
    }

    override fun remove(key: String): String? =
        transaction(db) {
            get(key)?.also { propertyTable.deleteWhere { propertyTable.id eq key } }
        }

    private class MutableDelegateIterator<T, Y>(private val delegate: MutableIterator<Y>, private val next: (Y) -> T) :
        MutableIterator<T> {
        override fun hasNext(): Boolean = delegate.hasNext()
        override fun next(): T = next(delegate.next())
        override fun remove() = delegate.remove()
    }

    private class DatabasePropertiesEntrySet(
        private val db: Database,
        private val props: DatabaseProperties,
        private val propertyTable: PropertyTable
    ) :
        AbstractMutableSet<MutableMap.MutableEntry<String, String>>() {
        override val size: Int = props.size

        override fun add(element: MutableMap.MutableEntry<String, String>): Boolean {
            val sizeBefore = size
            props[element.key] = element.value
            return sizeBefore != size
        }

        override fun iterator(): MutableIterator<MutableMap.MutableEntry<String, String>> =
            object : MutableIterator<MutableMap.MutableEntry<String, String>> {
                private val delegate = transaction(db) {
                    propertyTable.selectAll().associate { it[propertyTable.id] to it[propertyTable.value] }
                }.toMutableMap().iterator()

                private var last: MutableMap.MutableEntry<String, String>? = null

                override fun hasNext(): Boolean = delegate.hasNext()

                override fun next(): MutableMap.MutableEntry<String, String> =
                    delegate.next().let { entry -> SpecialEntry(props, entry.key.value, entry.value) }.also {
                        last = it
                    }

                override fun remove() {
                    requireNotNull(last).also {
                        delegate.remove()
                        props.remove(it.key, it.value)
                        last = null
                    }
                }
            }
    }

    private class SpecialEntry(
        private val props: DatabaseProperties,
        override val key: String,
        override var value: String
    ) :
        MutableMap.MutableEntry<String, String> {
        override fun setValue(newValue: String): String = value.also {
            value = newValue
            props[key] = newValue
        }
    }
}
