package dev.speakeasyapi.sdk.utils;

import java.io.ByteArrayOutputStream;
import java.time.Instant;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;

import dev.speakeasyapi.sdk.SpeakeasyRequestResponseHandler;
import dev.speakeasyapi.sdk.client.SpeakeasyClient;

public class SpeakeasyInterceptor implements HandlerInterceptor {
    public static final String ControllerKey = "speakeasyMiddlewareController";

    private String serverUrl = "grpc.prod.speakeasyapi.dev:443";
    private boolean secureGrpc = true;
    private Logger logger = LoggerFactory.getLogger(SpeakeasyInterceptor.class);
    private SpeakeasyRequestResponseHandler requestResponseHandler;
    private Instant startTime;
    private String pathHint = "";

    public SpeakeasyInterceptor(String apiKey, String apiID, String versionID) {
        init(apiKey, apiID, versionID, this.logger);
    }

    public SpeakeasyInterceptor(String apiKey, String apiID, String versionID, Logger logger) {
        init(apiKey, apiID, versionID, logger);
    }

    public SpeakeasyInterceptor(SpeakeasyRequestResponseHandler requestResponseHandler) {
        this.requestResponseHandler = requestResponseHandler;
    }

    private void init(String apiKey, String apiID, String versionID, Logger logger) {
        this.logger = logger;

        String serverURL = System.getenv("SPEAKEASY_SERVER_URL");
        if (serverURL != null) {
            this.serverUrl = serverURL;
        }

        String serverSecure = System.getenv("SPEAKEASY_SERVER_SECURE");
        if (serverSecure == "false") {
            this.secureGrpc = false;
        }

        requestResponseHandler = new SpeakeasyRequestResponseHandler(
                new SpeakeasyClient(apiKey, apiID, versionID, this.serverUrl, this.secureGrpc), logger);
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        startTime = Instant.now();

        if (handler instanceof HandlerMethod) {
            pathHint = getPathHint((HandlerMethod) handler);
        }

        request.setAttribute(ControllerKey, new SpeakeasyMiddlewareController());

        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest req, HttpServletResponse res, Object handler, Exception ex) {
        SpeakeasyMiddlewareController controller = (SpeakeasyMiddlewareController) req.getAttribute(ControllerKey);

        if (StringUtils.hasText(controller.getPathHint())) {
            pathHint = controller.getPathHint();
        }

        new Thread(() -> {
            try {

                ByteArrayOutputStream requestOutputStream = null;
                ByteArrayOutputStream responseOutputStream = null;
                try {
                    responseOutputStream = ((SpeakeasyResponseWrapper) res).getSpeakeasyOutputStream();
                    requestOutputStream = (ByteArrayOutputStream) req
                            .getAttribute(SpeakeasyRequestWrapper.speakeasyOutputStringAttribute);
                } catch (Exception e) {
                    logger.error("speakeasy-sdk:", e);
                }

                requestResponseHandler.captureRequestResponse(req, requestOutputStream,
                        res, responseOutputStream, startTime, Instant.now(), pathHint,
                        controller.getCustomerID());
            } catch (Exception e) {
                logger.error("speakeasy-sdk:", e);
            }
        }).start();
    }

    private String getPathHint(HandlerMethod hm) {
        String controllerPath = "";

        if (hm.getBean().getClass().isAnnotationPresent(RequestMapping.class)) {
            RequestMapping ctrlMapping = hm.getBean().getClass().getAnnotation(RequestMapping.class);
            if (ctrlMapping != null && ctrlMapping.value() != null && ctrlMapping.value().length > 0) {
                controllerPath = ctrlMapping.value()[0];

                if (pathHint.endsWith("/")) {
                    controllerPath = pathHint.substring(0, pathHint.length() - 1);
                }

                if (!controllerPath.startsWith("/")) {
                    controllerPath = "/" + controllerPath;
                }
            }
        }

        String methodPath = "";

        if (hm.hasMethodAnnotation(RequestMapping.class)) {
            RequestMapping mapping = hm.getMethodAnnotation(RequestMapping.class);
            if (mapping != null && mapping.value() != null && mapping.value().length > 0) {
                methodPath = mapping.value()[0];
            }
        } else if (hm.hasMethodAnnotation(GetMapping.class)) {
            GetMapping mapping = hm.getMethodAnnotation(GetMapping.class);
            if (mapping != null && mapping.value() != null && mapping.value().length > 0) {
                methodPath = mapping.value()[0];
            }
        } else if (hm.hasMethodAnnotation(PostMapping.class)) {
            PostMapping mapping = hm.getMethodAnnotation(PostMapping.class);
            if (mapping != null && mapping.value() != null && mapping.value().length > 0) {
                methodPath = mapping.value()[0];
            }
        } else if (hm.hasMethodAnnotation(PutMapping.class)) {
            PutMapping mapping = hm.getMethodAnnotation(PutMapping.class);
            if (mapping != null && mapping.value() != null && mapping.value().length > 0) {
                methodPath = mapping.value()[0];
            }
        } else if (hm.hasMethodAnnotation(DeleteMapping.class)) {
            DeleteMapping mapping = hm.getMethodAnnotation(DeleteMapping.class);
            if (mapping != null && mapping.value() != null && mapping.value().length > 0) {
                methodPath = mapping.value()[0];
            }
        } else if (hm.hasMethodAnnotation(PatchMapping.class)) {
            PatchMapping mapping = hm.getMethodAnnotation(PatchMapping.class);
            if (mapping != null && mapping.value() != null && mapping.value().length > 0) {
                methodPath = mapping.value()[0];
            }
        }

        if (!methodPath.startsWith("/")) {
            methodPath = "/" + methodPath;
        }

        return controllerPath + methodPath;
    }
}
