// Copyright (c) 2016-2020 Association of Universities for Research in Astronomy, Inc. (AURA)
// For license information see LICENSE or https://opensource.org/licenses/BSD-3-Clause

package edu.gemini.grackle
package sql

import scala.annotation.tailrec

import cats.data.{NonEmptyList, State}
import cats.implicits._
import fs2.Stream
import io.circe.Json
import org.tpolecat.sourcepos.SourcePos

import Cursor.{Context, Env}
import Predicate._
import Query._
import QueryInterpreter.{mkErrorResult, mkOneError}
import circe.CirceMapping

/** An abstract mapping that is backed by a SQL database. */
trait SqlMapping[F[_]] extends CirceMapping[F] with SqlModule[F] { self =>

  override val validator: SqlMappingValidator =
    SqlMappingValidator(this)

  import FieldMappingType._
  import SqlQuery.{SqlJoin, SqlSelect, SqlUnion}
  import TableExpr.{DerivedTableRef, SubqueryRef, TableRef, WithRef}

  case class TableName(name: String)
  class TableDef(name: String) {
    implicit val tableName: TableName = TableName(name)
  }

  /**
   * Name of a SQL schema column and its associated codec, Scala type an defining
   * source position within an `SqlMapping`.
   *
   * `Column`s are considered equal if their table and column names are equal.
   *
   * Note that `ColumnRef` primarily play a role in mappings. During compilation
   * they will be used to construct `SqlColumns`.
   */
  case class ColumnRef(table: String, column: String, codec: Codec, scalaTypeName: String, pos: SourcePos) {
    override def equals(other: Any) =
      other match {
        case cr: ColumnRef => table == cr.table && column == cr.column
        case _ => false
      }

    override def hashCode(): Int =
      table.hashCode() + column.hashCode()
  }

  type Aliased[T] = State[AliasState, T]
  object Aliased {
    def pure[T](t: T): Aliased[T] = State.pure(t)
    def tableDef(table: TableExpr): Aliased[String] = State(_.tableDef(table))
    def tableRef(table: TableExpr): Aliased[String] = State(_.tableRef(table))
    def columnDef(column: SqlColumn): Aliased[(Option[String], String)] = State(_.columnDef(column))
    def columnRef(column: SqlColumn): Aliased[(Option[String], String)] = State(_.columnRef(column))
    def pushOwner(owner: ColumnOwner): Aliased[Unit] = State(_.pushOwner(owner))
    def popOwner: Aliased[ColumnOwner] = State(_.popOwner)
  }

  /**
   * State required to assign table and column aliases.
   *
   * Used when rendering an `SqlQuery` as a `Fragment`. Table aliases are assigned
   * as needed for recursive queries. Column aliases are assigned to disambiguate
   * collections of columns generated by subqueries and unions.
   */
  case class AliasState(
    next: Int,
    seenTables: Set[String],
    tableAliases: Map[(List[String], String), String],
    seenColumns: Set[String],
    columnAliases: Map[(List[String], String), String],
    ownerChain: List[ColumnOwner]
  ) {
    /** Update state to reflect a defining occurence of a table */
    def tableDef(table: TableExpr): (AliasState, String) =
      tableAliases.get((table.context.resultPath, table.name)) match {
        case Some(alias) => (this, alias)
        case None =>
          if (seenTables(table.name)) {
            val alias = s"${table.name}_alias_$next"
            val newState =
              copy(
                next = next+1,
                tableAliases = tableAliases + ((table.context.resultPath, table.name) -> alias)
              )
            (newState, alias)
          } else {
            val newState =
              copy(
                seenTables = seenTables + table.name,
                tableAliases = tableAliases + ((table.context.resultPath, table.name) -> table.name)
              )
            (newState, table.name)
          }
      }

    /** Yields the possibly aliased name of the supplied table */
    def tableRef(table: TableExpr): (AliasState, String) =
      tableAliases.get((table.context.resultPath, table.name)) match {
        case Some(alias) => (this, alias)
        case None => (this, table.name)
      }

    /** Update state to reflect a defining occurence of a column */
    def columnDef(column: SqlColumn): (AliasState, (Option[String], String)) = {
      val (newState0, table0) = column.namedOwner.map(named => tableDef(named).fmap(Option(_))).getOrElse((this, None))
      columnAliases.get((column.underlying.owner.context.resultPath, column.underlying.column)) match {
        case Some(name) => (newState0, (table0, name))
        case None =>
          val (next0, seenColumns0, column0) =
            if (newState0.seenColumns(column.column))
              (newState0.next+1, newState0.seenColumns, s"${column.column}_alias_${newState0.next}")
            else
              (newState0.next, newState0.seenColumns + column.column, column.column)

          val columnAliases0 =
             newState0.columnAliases + ((column.underlying.owner.context.resultPath, column.underlying.column) -> column0)

          val newState = newState0.copy(next = next0, seenColumns = seenColumns0, columnAliases = columnAliases0)

          (newState, (table0, column0))
      }
    }

    /** Yields the possibly aliased name of the supplied column */
    def columnRef(column: SqlColumn): (AliasState, (Option[String], String)) = {
      if (ownerChain.exists(_.directlyOwns(column))) {
        column.namedOwner.map(named => tableRef(named).
          fmap(Option(_))).getOrElse((this, None)).
          fmap(table0 => (table0, column.column))
      } else {
        val name = columnAliases.get((column.underlying.owner.context.resultPath, column.underlying.column)).getOrElse(column.column)
        column.namedOwner.map(named => tableRef(named).
          fmap(Option(_))).getOrElse((this, None)).
          fmap(table0 => (table0, name))
      }
    }

    /** Update state to reflect the current column owner while traversing
     *  the `SqlQuery` being rendered
     */
    def pushOwner(owner: ColumnOwner): (AliasState, Unit) = (copy(ownerChain = owner :: ownerChain), ())

    /** Update state to restore the current column owner while traversing
     *  the `SqlQuery` being rendered
     */
    def popOwner: (AliasState, ColumnOwner) = (copy(ownerChain = ownerChain.tail), ownerChain.head)
  }

  object AliasState {
    def empty: AliasState =
      AliasState(
        0,
        Set.empty[String],
        Map.empty[(List[String], String), String],
        Set.empty[String],
        Map.empty[(List[String], String), String],
        List.empty[ColumnOwner]
      )
  }

  /** Trait representing an owner of an `SqlColumn
   *
   *  ColumnOwners are tables, SQL queries and subqueries, common
   *  table expressions and the like. Most, but not all have a
   *  name (SqlSelect, SqlUnion and SqlJoin being unnamed
   *  examples)
   */
  sealed trait ColumnOwner extends Product with Serializable {
    def context: Context
    def owns(col: SqlColumn): Boolean
    def contains(other: ColumnOwner): Boolean
    def directlyOwns(col: SqlColumn): Boolean
    def findNamedOwner(col: SqlColumn): Option[TableExpr]

    /** The name, if any, of this `ColumnOwner` */
    def nameOption: Option[String] =
      this match {
        case named: TableExpr => Some(named.name)
        case _ => None
      }

    def isSameOwner(other: ColumnOwner): Boolean =
      (this, other) match {
        case (n1: TableExpr, n2: TableExpr) =>
          (n1.name == n2.name) && (context == other.context)
        case _ => false
      }

    def debugShow: String =
      (this match {
        case tr: TableExpr => tr.toDefFragment
        case sq: SqlQuery => sq.toFragment
        case sj: SqlJoin => sj.toFragment
      }).runA(AliasState.empty).value.toString
  }

  /** Trait representing an SQL column */
  trait SqlColumn {
    def owner: ColumnOwner
    def column: String
    def codec: Codec
    def scalaTypeName: String
    def pos: SourcePos

    /** The named owner of this column, if any */
    def namedOwner: Option[TableExpr] =
      owner.findNamedOwner(this)

    /** If this column is derived, the column it was derived from, itself otherwise */
    def underlying: SqlColumn = this

    /** Is this column a reference to a column of a table */
    def isRef: Boolean = false

    /** Yields a copy of this column with all occurences of `from` replaced by `to` */
    def subst(from: ColumnOwner, to: ColumnOwner): SqlColumn

    /** Yields a copy of this column in `other`
     *
     *  Only well defined if the move doesn't lose an owner name
     */
    def in(other: ColumnOwner): SqlColumn = {
      assert(other.nameOption.isDefined || owner.nameOption.isEmpty)
      subst(owner, other)
    }

    /** Derives a new column with a different owner with this column as underlying.
     *
     *  Used to represent columns on the outside of subqueries and common table
     *  expressions. Note that column aliases are tracked across derivation so
     *  that derived columns will continue to refer to the same underlying data
     *  irrespective of renaming.
     */
    def derive(other: ColumnOwner): SqlColumn =
      if(other == owner) this
      else SqlColumn.DerivedColumn(other, this)

    /** Equality on `SqlColumns`
     *
     *  Two `SqlColumns` are equal if their underlyings have the same name and owner.
     */
    override def equals(other: Any) =
      other match {
        case cr: SqlColumn =>
          val u0 = underlying
          val u1 = cr.underlying
          u0.column == u1.column && u0.owner.isSameOwner(u1.owner)
        case _ => false
      }

    override def hashCode(): Int = {
      val u = underlying
      u.owner.context.hashCode() + u.column.hashCode()
    }

    /** This column as a `Term` which can appear in a `Predicate` */
    def toTerm: Term[Option[Unit]] = SqlColumnTerm(this)

    /** Render a defining occurence of this `SqlColumn` */
    def toDefFragment(collated: Boolean): Aliased[Fragment]
    /** Render a reference to this `SqlColumn` */
    def toRefFragment(collated: Boolean): Aliased[Fragment]

    override def toString: String =
      owner match {
        case named: TableExpr => s"${named.name}.$column"
        case _ => column
      }
  }

  object SqlColumn {
    def mkDefFragment(prefix: Option[String], base: String, collated: Boolean, alias: String): Fragment = {
      val prefix0 = prefix.map(_+".").getOrElse("")
      val qualified = prefix0+base
      val collated0 =
        if (collated) s"""($qualified COLLATE "C")"""
        else qualified
      val aliased =
        if (base == alias) collated0
        else s"$collated0 AS $alias"
      Fragments.const(aliased)
    }

    def mkDefFragment(base: Fragment, collated: Boolean, alias: String): Fragment = {
      val collated0 =
        if (collated) Fragments.parentheses(base |+| Fragments.const(s""" COLLATE "C""""))
        else base
      collated0 |+| Fragments.const(s"AS $alias")
    }

    def mkRefFragment(prefix: Option[String], alias: String, collated: Boolean): Fragment = {
      val prefix0 = prefix.map(_+".").getOrElse("")
      val qualified = prefix0+alias
      val base = Fragments.const(qualified)
      if (collated) Fragments.parentheses(base |+| Fragments.const(s""" COLLATE "C""""))
      else base
    }

    /** Representation of a column of a table/view */
    case class TableColumn(owner: ColumnOwner, cr: ColumnRef) extends SqlColumn {
      def column: String = cr.column
      def codec: Codec = cr.codec
      def scalaTypeName: String = cr.scalaTypeName
      def pos: SourcePos = cr.pos

      def subst(from: ColumnOwner, to: ColumnOwner): SqlColumn =
        if(!owner.isSameOwner(from)) this
        else to match {
          case _: DerivedTableRef => derive(to)
          case _: SubqueryRef => derive(to)
          case _ => copy(owner = to)
        }

      override def isRef: Boolean = true

      def toDefFragment(collated: Boolean): Aliased[Fragment] =
        Aliased.columnDef(this).map {
          case (table0, column0) => mkDefFragment(table0, column, collated, column0)
        }

      def toRefFragment(collated: Boolean): Aliased[Fragment] =
        Aliased.columnRef(this).map {
          case (table0, column0) => mkRefFragment(table0, column0, collated)
        }
    }

    object TableColumn {
      def apply(context: Context, cr: ColumnRef): TableColumn =
        TableColumn(TableRef(context, cr.table), cr)
    }

    /** Representation of a synthetic null column
     *
     *  Primarily used to pad the disjuncts of an `SqlUnion`.
     */
    case class NullColumn(owner: ColumnOwner, col: SqlColumn) extends SqlColumn {
      def column: String = col.column
      def codec: Codec = col.codec
      def scalaTypeName: String = col.scalaTypeName
      def pos: SourcePos = col.pos

      override def underlying: SqlColumn = col.underlying

      def subst(from: ColumnOwner, to: ColumnOwner): SqlColumn =
        copy(owner = if(owner.isSameOwner(from)) to else owner, col = col.subst(from, to))

      def toDefFragment(collated: Boolean): Aliased[Fragment] =
        Aliased.columnDef(this).map {
          case (_, column0) =>
            val ascribed =
              Fragments.sqlTypeName(codec) match {
                case Some(name) => Fragments.const(s"(NULL :: $name)")
                case None => Fragments.const("NULL")
              }
            mkDefFragment(ascribed, collated, column0)
        }

      def toRefFragment(collated: Boolean): Aliased[Fragment] =
        Aliased.columnRef(this).map {
          case (table0, column0) => mkRefFragment(table0, column0, collated)
        }
    }

    /** Representation of a scalar subquery */
    case class SubqueryColumn(col: SqlColumn, subquery: SqlSelect) extends SqlColumn {
      def owner: ColumnOwner = col.owner
      def column: String = col.column
      def codec: Codec = col.codec
      def scalaTypeName: String = col.scalaTypeName
      def pos: SourcePos = col.pos

      def subst(from: ColumnOwner, to: ColumnOwner): SqlColumn = {
        val subquery0 =
          (from, to) match {
            case (tf: TableExpr, tt: TableExpr) => subquery.subst(tf, tt)
            case _ => subquery
          }
        copy(col = col.subst(from, to), subquery = subquery0)
      }

      def toDefFragment(collated: Boolean): Aliased[Fragment] =
        for {
          sub0  <- subquery.toFragment
          tc0   <- Aliased.columnDef(this)
        } yield mkDefFragment(Fragments.parentheses(sub0), collated, tc0._2)

      def toRefFragment(collated: Boolean): Aliased[Fragment] =
        Aliased.columnRef(this).map {
          case (table0, column0) => mkRefFragment(table0, column0, collated)
        }
    }

    /** Representation of COUNT aggregation */
    case class CountColumn(col: SqlColumn, cols: List[SqlColumn]) extends SqlColumn {
      def owner: ColumnOwner = col.owner
      def column: String = col.column
      def codec: Codec = col.codec
      def scalaTypeName: String = col.scalaTypeName
      def pos: SourcePos = col.pos

      def subst(from: ColumnOwner, to: ColumnOwner): SqlColumn =
        copy(col.subst(from, to), cols.map(_.subst(from, to)))

      def toDefFragment(collated: Boolean): Aliased[Fragment] =
        for {
          cols0 <- cols.traverse(_.toRefFragment(false))
          ct0   <- Aliased.columnDef(this)
        } yield {
          val count = Fragments.const("COUNT(DISTINCT(") |+| cols0.intercalate(Fragments.const(", ")) |+| Fragments.const(s"))")
          mkDefFragment(count, collated, ct0._2)
        }

      def toRefFragment(collated: Boolean): Aliased[Fragment] =
        Aliased.columnRef(this).map {
          case (table0, column0) => mkRefFragment(table0, column0, collated)
        }
    }

    /** Representation of a window aggregation */
    case class PartitionColumn(owner: ColumnOwner, column: String, partitionCols: List[SqlColumn], orders: List[OrderSelection[_]]) extends SqlColumn {
      def codec: Codec = intCodec
      def scalaTypeName: String = "Int"
      def pos: SourcePos = null

      def subst(from: ColumnOwner, to: ColumnOwner): SqlColumn =
        copy(owner = if(owner.isSameOwner(from)) to else owner, partitionCols = partitionCols.map(_.subst(from, to)))

      def partitionColsToFragment: Aliased[Fragment] =
        if (partitionCols.isEmpty) Aliased.pure(Fragments.empty)
        else
          partitionCols.traverse(_.toRefFragment(false)).map { fcols =>
            Fragments.const("PARTITION BY ") |+| fcols.intercalate(Fragments.const(", "))
          }

      def toDefFragment(collated: Boolean): Aliased[Fragment] =
        for {
          cols0   <- partitionColsToFragment
          tc0     <- Aliased.columnDef(this)
          orderBy <- SqlQuery.ordersToFragment(orders)
        } yield {
          //val base = Fragments.const("row_number() OVER ") |+| Fragments.parentheses(cols0 |+| orderBy)
          val base = Fragments.const("dense_rank() OVER ") |+| Fragments.parentheses(cols0 |+| orderBy)
          mkDefFragment(base, false, tc0._2)
        }

      def toRefFragment(collated: Boolean): Aliased[Fragment] =
        Aliased.columnRef(this).map {
          case (table0, column0) => mkRefFragment(table0, column0, collated)
        }
    }

    /** Representation of a column of an embedded subobject
     *
     *  Columns of embedded subobjects have a different context path from columns of
     *  their enclosing object, however they resolve to columns of the same `SqlSelect`.
     *  To satisfy the `SqlSelect` invariant that all its columns must share the same
     *  context path we have to wrap the embedded column so that its context path
     *  conforms.
     */
    case class EmbeddedColumn(owner: ColumnOwner, col: SqlColumn) extends SqlColumn {
      def column: String = col.column
      def codec: Codec = col.codec
      def scalaTypeName: String = col.scalaTypeName
      def pos: SourcePos = col.pos

      override def underlying: SqlColumn = col.underlying

      def subst(from: ColumnOwner, to: ColumnOwner): SqlColumn =
        copy(owner = if(owner.isSameOwner(from)) to else owner, col = col.subst(from, to))

      override def isRef: Boolean = col.isRef

      def toDefFragment(collated: Boolean): Aliased[Fragment] =
        Aliased.columnDef(this).map {
          case (table0, column0) => mkDefFragment(table0, column, collated, column0)
        }

      def toRefFragment(collated: Boolean): Aliased[Fragment] =
        Aliased.columnRef(this).map {
          case (table0, column0) => mkRefFragment(table0, column0, collated)
        }
    }

    /** Representation of a derived column
     *
     *  Used to represent columns on the outside of subqueries and common table
     *  expressions. Note that column aliases are tracked across derivation so
     *  that derived columns will continue to refer to the same underlying data
     *  irrespective of renaming.
     */
    case class DerivedColumn(owner: ColumnOwner, col: SqlColumn) extends SqlColumn {
      def column: String = col.column
      def codec: Codec = col.codec
      def scalaTypeName: String = col.scalaTypeName
      def pos: SourcePos = col.pos

      override def underlying: SqlColumn = col.underlying

      def subst(from: ColumnOwner, to: ColumnOwner): SqlColumn =
        copy(owner = if(owner.isSameOwner(from)) to else owner, col = col.subst(from, to))

      override def isRef: Boolean = col.isRef

      def toDefFragment(collated: Boolean): Aliased[Fragment] = {
        for {
          table0 <- namedOwner.map(named => Aliased.tableDef(named).map(Option(_))).getOrElse(Aliased.pure(None))
          tc     <- Aliased.columnDef(col)
        } yield mkDefFragment(table0, tc._2, collated, tc._2)
      }

      def toRefFragment(collated: Boolean): Aliased[Fragment] =
        Aliased.columnRef(this).map {
          case (table0, column0) => mkRefFragment(table0, column0, collated)
        }
    }
  }

  /** Wraps an `SqlColumn` as a `Term` which can appear in a `Predicate` */
  case class SqlColumnTerm(col: SqlColumn) extends Term[Option[Unit]] {
    def apply(c: Cursor): Result[Option[Unit]] = Result(Option(()))
    def children: List[Term[_]] = Nil
  }

  /** A pair of `ColumnRef`s, representing a SQL join. */
  case class Join(parent: ColumnRef, child: ColumnRef)

  case class SqlRoot(fieldName: String, orootTpe: Option[Type] = None, mutation: Mutation = Mutation.None)(
    implicit val pos: SourcePos
  ) extends RootMapping {

    /**
      * Operators which can be compiled to SQL are eliminated here, partly to avoid duplicating
      * work programmatically, but also because the operation isn't necessarily idempotent and
      * the result set doesn't necessarily contain the fields required for the filter predicates.
      */
    def stripCompiled(query: Query, context: Context): Query = {
      def loop(query: Query, context: Context): Query =
        query match {
          // Preserved non-Sql filters
          case FilterOrderByOffsetLimit(p@Some(pred), oss, off, lim, child) if !isSqlTerm(context, pred) =>
            FilterOrderByOffsetLimit(p, oss, off, lim, loop(child, context))

          case Filter(_, child) => loop(child, context)
          case Offset(_, child) => loop(child, context)
          case Limit(_, child) => loop(child, context)

          // Preserve OrderBy
          case o: OrderBy => o.copy(child = loop(o.child, context))

          case PossiblyRenamedSelect(s@Select(fieldName, _, _), resultName) =>
            val fieldContext = context.forField(fieldName, resultName).getOrElse(sys.error(s"No field '$fieldName' of type ${context.tpe}"))
            PossiblyRenamedSelect(s.copy(child = loop(s.child, fieldContext)), resultName)
          case Rename(_, Count(_, _)) => Empty
          case Count(countName, _) => Select(countName, Nil, Empty)

          case Group(queries) => Group(queries.map(q => loop(q, context)))
          case GroupList(queries) => GroupList(queries.map(q => loop(q, context)))
          case u: Unique => u.copy(child = loop(u.child, context.asType(context.tpe.list)))
          case e: Environment => e.copy(child = loop(e.child, context))
          case w: Wrap => w.copy(child = loop(w.child, context))
          case r: Rename => r.copy(child = loop(r.child, context))
          case u: UntypedNarrow => u.copy(child = loop(u.child, context))
          case n@Narrow(subtpe, _) => n.copy(child = loop(n.child, context.asType(subtpe)))
          case s: Skip => s.copy(child = loop(s.child, context))
          case other@(_: Component[_] | _: Defer | Empty | _: Introspect | _: Select | Skipped) => other
        }

      loop(query, context)
    }

    private def mkRootCursor(query: Query, context: Context, env: Env): F[Result[(Query, Cursor)]] = {
      (MappedQuery(query, context).map { mapped =>
        for {
          table <- mapped.fetch
          _     <- monitor.queryMapped(query, mapped.fragment, table.numRows, table.numCols)
        } yield {
          val stripped: Query = stripCompiled(query, context)
          val cursor: Cursor = SqlCursor(context.asType(context.tpe.list), table, mapped, None, env)
          Result((stripped, cursor))
        }
      }).getOrElse(mkErrorResult("Unable to map query").widen.pure[F])
    }

    def cursor(query: Query, env: Env, resultName: Option[String]): Stream[F,Result[(Query, Cursor)]] =
      Stream.eval {
        (for {
          rootTpe  <- orootTpe
          context  <- Context(rootTpe, fieldName, resultName)
        } yield {
          mkRootCursor(query, context, env)
        }).getOrElse(mkErrorResult(s"Type ${orootTpe.getOrElse("unspecified type")} has no field '$fieldName'").pure[F])
      }

    def withParent(tpe: Type): SqlRoot =
      copy(orootTpe = Some(tpe))

  }

  sealed trait SqlFieldMapping extends FieldMapping {
    final def withParent(tpe: Type): FieldMapping = this
  }

  case class SqlField(
    fieldName: String,
    columnRef: ColumnRef,
    key: Boolean = false,
    discriminator: Boolean = false,
    hidden: Boolean = false,
    associative: Boolean = false // a key which is also associative might occur multiple times in the table, ie. it is not a DB primary key
  )(implicit val pos: SourcePos) extends SqlFieldMapping

  case class SqlObject(fieldName: String, joins: List[Join])(
    implicit val pos: SourcePos
  ) extends SqlFieldMapping {
    final def hidden = false
  }
  object SqlObject {
    def apply(fieldName: String, joins: Join*): SqlObject = apply(fieldName, joins.toList)
  }

  case class SqlJson(fieldName: String, columnRef: ColumnRef)(
    implicit val pos: SourcePos
  ) extends SqlFieldMapping {
    def hidden: Boolean = false
  }

  /**
   * Common super type for mappings which have a programmatic discriminator, ie. interface and union mappings.
   */
  sealed trait SqlDiscriminatedType {
    def discriminator: SqlDiscriminator
  }

  /** Discriminator for the branches of an interface/union */
  trait SqlDiscriminator {
    /** yield a predicate suitable for filtering row corresponding to the supplied type */
    def narrowPredicate(tpe: Type): Option[Predicate]

    /** compute the concrete type of the value at the cursor */
    def discriminate(cursor: Cursor): Result[Type]
  }

  sealed trait SqlInterfaceMapping extends ObjectMapping with SqlDiscriminatedType

  object SqlInterfaceMapping {

    case class DefaultInterfaceMapping(tpe: Type, fieldMappings: List[FieldMapping], path: List[String], discriminator: SqlDiscriminator)(
      implicit val pos: SourcePos
    ) extends SqlInterfaceMapping

    def apply(
      tpe: Type,
      fieldMappings: List[FieldMapping],
      path: List[String] = Nil,
      discriminator: SqlDiscriminator
    )(
      implicit pos: SourcePos
    ): ObjectMapping =
      DefaultInterfaceMapping(tpe, fieldMappings.map(_.withParent(tpe)), path, discriminator)
  }

  sealed trait SqlUnionMapping extends ObjectMapping with SqlDiscriminatedType

  object SqlUnionMapping {

    case class DefaultUnionMapping(tpe: Type, fieldMappings: List[FieldMapping], path: List[String], discriminator: SqlDiscriminator)(
      implicit val pos: SourcePos
    ) extends SqlUnionMapping

    def apply(
      tpe: Type,
      fieldMappings: List[FieldMapping],
      path: List[String] = Nil,
      discriminator: SqlDiscriminator,
    )(
      implicit pos: SourcePos
    ): ObjectMapping =
      DefaultUnionMapping(tpe, fieldMappings.map(_.withParent(tpe)), path, discriminator)
  }

  /** Returns the discriminator columns for the context type */
  def discriminatorColumnsForType(context: Context): List[SqlColumn] =
    objectMapping(context).map(_.fieldMappings.collect {
      case cm: SqlField if cm.discriminator => SqlColumn.TableColumn(context, cm.columnRef)
    }).getOrElse(Nil)

  /** Returns the key columns for the context type */
  def keyColumnsForType(context: Context): List[SqlColumn] = {
    val cols =
      objectMapping(context).map { obj =>
        val objectKeys = obj.fieldMappings.collect {
          case cm: SqlField if cm.key => SqlColumn.TableColumn(context, cm.columnRef)
        }

        val interfaceKeys = context.tpe.underlyingObject match {
          case Some(ot: ObjectType) =>
            ot.interfaces.flatMap(nt => keyColumnsForType(context.asType(nt)))
          case _ => Nil
        }

        (objectKeys ++ interfaceKeys).distinct
      }.getOrElse(Nil)

    assert(cols.nonEmpty, s"No key columns for type ${context.tpe}")

    cols
  }

  /** Returns the columns for leaf field `fieldName` in `context` */
  def columnsForLeaf(context: Context, fieldName: String): List[SqlColumn] =
    fieldMapping(context, fieldName) match {
      case Some(SqlField(_, cr, _, _, _, _)) => List(SqlColumn.TableColumn(context, cr))
      case Some(SqlJson(_, cr)) => List(SqlColumn.TableColumn(context, cr))
      case Some(CursorFieldJson(_, _, _, required, _)) =>
        required.flatMap(r => columnsForLeaf(context, r))
      case Some(CursorField(_, _, _, required, _)) =>
        required.flatMap(r => columnsForLeaf(context, r))
      case None =>
        sys.error(s"No mapping for field '$fieldName' of type ${context.tpe}")
      case other =>
        sys.error(s"Non-leaf mapping for field '$fieldName' of type ${context.tpe}: $other")
    }

  /** Returns the aliased columns corresponding to `term` in `context` */
  def columnForSqlTerm[T](context: Context, term: Term[T]): Option[SqlColumn] =
    term match {
      case termPath: Path =>
        context.forPath(termPath.path.init).flatMap { parentContext =>
          columnForAtomicField(parentContext, termPath.path.last)
        }
      case SqlColumnTerm(col) => Some(col)
      case _ => None
    }

  /** Returns the aliased column corresponding to the atomic field `fieldName` in `context` */
  def columnForAtomicField(context: Context, fieldName: String): Option[SqlColumn] = {
    fieldMapping(context, fieldName) match {
      case Some(SqlField(_, cr, _, _, _, _)) => Some(SqlColumn.TableColumn(context, cr))
      case Some(SqlJson(_, cr)) => Some(SqlColumn.TableColumn(context, cr))
      case _ => None
    }
  }

  /** Returns the `Encoder` for the given type */
  def encoderForLeaf(tpe: Type): Option[io.circe.Encoder[Any]] =
    leafMapping[Any](tpe).map(_.encoder)

  /** Returns the `Encoder` for the given term in `context` */
  def encoderForTerm(context: Context, term: Term[_]): Option[Encoder] =
    term match {
      case pathTerm: Path =>
        for {
          cr <- columnForSqlTerm(context, pathTerm) // encoder is independent of query aliases
        } yield toEncoder(cr.codec)

      case SqlColumnTerm(col) => Some(toEncoder(col.codec))

      case (_: And)|(_: Or)|(_: Not)|(_: Eql[_])|(_: NEql[_])|(_: Lt[_])|(_: LtEql[_])|(_: Gt[_])|(_: GtEql[_])  => Some(booleanEncoder)
      case (_: AndB)|(_: OrB)|(_: XorB)|(_: NotB) => Some(intEncoder)
      case (_: ToUpperCase)|(_: ToLowerCase) => Some(stringEncoder)
      case _ => None
    }

  /** Returns the discriminator for the type at `context` */
  def discriminatorForType(context: Context): Option[SqlDiscriminatedType] =
    objectMapping(context) collect {
      //case d: SqlDiscriminatedType => d  // Fails in 2.13.6 due to https://github.com/scala/bug/issues/12398
      case i: SqlInterfaceMapping => i
      case u: SqlUnionMapping => u
    }

  /** Returns the table for the type at `context` */
  def parentTableForType(context: Context): Option[TableRef] =
    objectMapping(context).flatMap(_.fieldMappings.collectFirst { case SqlField(_, cr, _, _, _, _) => TableRef(context, cr.table) }.orElse {
      context.tpe.underlyingObject match {
        case Some(ot: ObjectType) =>
          ot.interfaces.collectFirstSome(nt => parentTableForType(context.asType(nt)))
        case _ => None
      }
    })

  /** Return an indicator of the kind of field mapping corresponding to `fieldName` in `context` */
  def fieldMappingType(context: Context, fieldName: String): Option[FieldMappingType] =
    fieldMapping(context, fieldName).flatMap {
      case CursorField(_, f, _, _, _) => Some(CursorFieldMapping(f))
      case CursorFieldJson(_, f, _, _, _) => Some(CursorFieldJsonMapping(f))
      case _: SqlJson => Some(JsonFieldMapping)
      case _: SqlField => Some(LeafFieldMapping)
      case _: SqlObject => Some(ObjectFieldMapping)
      case _ => None
    }

  /** Is `fieldName` in `context` Jsonb? */
  def isJsonb(context: Context, fieldName: String): Boolean =
    fieldMapping(context, fieldName) match {
      case Some(_: SqlJson) => true
      case Some(_: CursorFieldJson) => true
      case _ => false
    }

  /** Is `fieldName` in `context` computed? */
  def isComputedField(context: Context, fieldName: String): Boolean =
    fieldMapping(context, fieldName) match {
      case Some(_: CursorField[_]) => true
      case _ => false
    }

  /** Is `term` in `context`expressible in SQL? */
  def isSqlTerm(context: Context, term: Term[_]): Boolean =
    term.forall {
      case termPath: Path =>
        context.forPath(termPath.path.init).map { parentContext =>
          !isComputedField(parentContext, termPath.path.last)
        }.getOrElse(true)
      case True | False | _: Const[_] | _: And | _: Or | _: Not | _: Eql[_] | _: NEql[_] | _: Contains[_] | _: Lt[_] | _: LtEql[_] | _: Gt[_] |
            _: GtEql[_] | _: In[_] | _: AndB | _: OrB | _: XorB | _: NotB | _: Matches | _: StartsWith | _: IsNull[_] |
            _: ToUpperCase | _: ToLowerCase | _: Like | _: SqlColumnTerm => true
      case _ => false
    }

  /** Is the context type mapped to an associative table? */
  def isAssociative(context: Context): Boolean =
    objectMapping(context).map(_.fieldMappings.exists {
      case sf: SqlField => sf.associative
      case _ => false
    }).getOrElse(false)

  /** Does the type of `fieldName` in `context` represent a list of subobjects? */
  def nonLeafList(context: Context, fieldName: String): Boolean =
    context.tpe.underlyingField(fieldName).map { fieldTpe =>
      fieldTpe.nonNull.isList && (
        fieldMapping(context, fieldName).map {
          case SqlObject(_, joins) => joins.nonEmpty
          case _ => false
        }.getOrElse(false)
      )
    }.getOrElse(false)

  /** Does the supplied field correspond to a single, possibly structured, value? */
  def isSingular(context: Context, fieldName: String, query: Query): Boolean = {
    def loop(query: Query): Boolean =
      query match {
        case Limit(n, _) => n <= 1
        case _: Unique => true

        case Filter(_, child) => loop(child)
        case Offset(_, child) => loop(child)
        case OrderBy(_, child) => loop(child)
        case _ => false
      }

    !nonLeafList(context, fieldName) || loop(query)
  }

  /** Enumeration representing a kind of field mapping */
  sealed trait FieldMappingType
  object FieldMappingType {
    /** Field mapping is a subobject */
    case object ObjectFieldMapping extends FieldMappingType
    /** Field mapping is a leaf */
    case object LeafFieldMapping extends FieldMappingType
    /** Field mapping is a Json subobject */
    case object JsonFieldMapping extends FieldMappingType
    /** Field mapping is computed */
    case class CursorFieldMapping(f: Cursor => Result[Any]) extends FieldMappingType
    /** Field mapping is a computed Json value */
    case class CursorFieldJsonMapping(f: Cursor => Result[Json]) extends FieldMappingType
  }

  /** Representation of a table expression */
  sealed trait TableExpr extends ColumnOwner {
    /** The name of this `TableExpr` */
    def name: String

    /** Is the supplied column an immediate component of this `TableExpr`? */
    def directlyOwns(col: SqlColumn): Boolean = this == col.owner
    /** Find the innermost owner of the supplied column within this `TableExpr` */
    def findNamedOwner(col: SqlColumn): Option[TableExpr]

    /** Render a defining occurence of this `TableExpr` */
    def toDefFragment: Aliased[Fragment]
    /** Render a reference to this `TableExpr` */
    def toRefFragment: Aliased[Fragment]

    /** Is this `TableExpr` backed by an SQL union
     *
     *  This is used to determine whether or not non-nullable columns should be weakened
     *  to being nullable when fetched
     */
    def isUnion: Boolean

    /** Yields a copy of this `TableExpr` with all occurences of `from` replaced by `to` */
    def subst(from: TableExpr, to: TableExpr): TableExpr
  }

  object TableExpr {
    /** Table expression corresponding to a possibly aliased table */
    case class TableRef(context: Context, name: String) extends TableExpr {
      def owns(col: SqlColumn): Boolean = isSameOwner(col.owner)
      def contains(other: ColumnOwner): Boolean = isSameOwner(other)

      def findNamedOwner(col: SqlColumn): Option[TableExpr] =
        if (this == col.owner) Some(this) else None

      def isUnion: Boolean = false

      def subst(from: TableExpr, to: TableExpr): TableRef = this

      def toDefFragment: Aliased[Fragment] =
        for {
          alias <- Aliased.tableDef(this)
        } yield
          if (name == alias)
            Fragments.const(name)
          else
            Fragments.const(s"$name AS $alias")

      def toRefFragment: Aliased[Fragment] =
        Aliased.tableRef(this).map(Fragments.const)

      override def toString: String = name
    }

    /** Table expression corresponding to a subquery */
    case class SubqueryRef(context: Context, name: String, subquery: SqlQuery, lateral: Boolean) extends TableExpr {
      def owns(col: SqlColumn): Boolean = col.owner.isSameOwner(this) || subquery.owns(col)
      def contains(other: ColumnOwner): Boolean = isSameOwner(other) || subquery.contains(other)

      def findNamedOwner(col: SqlColumn): Option[TableExpr] =
        if (this == col.owner) Some(this) else subquery.findNamedOwner(col)

      def isUnion: Boolean = subquery.isUnion

      def subst(from: TableExpr, to: TableExpr): SubqueryRef =
        copy(subquery = subquery.subst(from, to))

      def toDefFragment: Aliased[Fragment] = {
        val lateral0 = if(lateral) Fragments.const("LATERAL ") else Fragments.empty
        for {
          alias <- Aliased.tableDef(this)
          sub <- subquery.toFragment
        } yield lateral0 |+| Fragments.parentheses(sub) |+| Fragments.const(s" AS $alias")
      }

      def toRefFragment: Aliased[Fragment] =
        Aliased.tableRef(this).map(Fragments.const)
    }

    /** Table expression corresponding to a common table expression */
    case class WithRef(context: Context, name: String, withQuery: SqlQuery) extends TableExpr {
      def owns(col: SqlColumn): Boolean = col.owner.isSameOwner(this) || withQuery.owns(col)
      def contains(other: ColumnOwner): Boolean = isSameOwner(other) || withQuery.contains(other)

      def findNamedOwner(col: SqlColumn): Option[TableExpr] =
        if (this == col.owner) Some(this) else withQuery.findNamedOwner(col)

      def isUnion: Boolean = withQuery.isUnion

      def subst(from: TableExpr, to: TableExpr): WithRef =
        copy(withQuery = withQuery.subst(from, to))

      def toDefFragment: Aliased[Fragment] =
        for {
          with0 <- withQuery.toFragment
        } yield Fragments.const(s" $name AS ") |+| Fragments.parentheses(with0)

      def toRefFragment: Aliased[Fragment] =
        Aliased.pure(Fragments.const(s"$name"))
    }

    /** Table expression derived from the given `TableExpr`.
     *
     *  Typically used where we need to refer to a table defined in a subquery or
     *  common table expression.
     */
    case class DerivedTableRef(context: Context, alias: Option[String], underlying: TableExpr, noalias: Boolean = false) extends TableExpr {
      assert(!underlying.isInstanceOf[WithRef] || noalias)

      def name = alias.getOrElse(underlying.name)

      def owns(col: SqlColumn): Boolean = col.owner.isSameOwner(this) || underlying.owns(col)
      def contains(other: ColumnOwner): Boolean = isSameOwner(other) || underlying.contains(other)

      def findNamedOwner(col: SqlColumn): Option[TableExpr] =
        if (this == col.owner) Some(this) else underlying.findNamedOwner(col)

      def isUnion: Boolean = underlying.isUnion

      def subst(from: TableExpr, to: TableExpr): DerivedTableRef =
        if(underlying == from) copy(underlying = to)
        else copy(underlying = underlying.subst(from, to))

      def toDefFragment: Aliased[Fragment] = {
        for {
          uname <- if (noalias) Aliased.pure(underlying.name) else Aliased.tableDef(underlying)
          name  <- Aliased.tableDef(this)
        } yield {
          if (name == uname)
            Fragments.const(s"$name")
          else
            Fragments.const(s"$uname AS $name")
        }
      }

      def toRefFragment: Aliased[Fragment] =
        Aliased.tableRef(this).map(Fragments.const)
    }
  }

  /** Representation of a SQL query in a context */
  sealed trait SqlQuery extends ColumnOwner {
    /** The context for this query */
    def context: Context

    /** This query in the given context */
    def withContext(context: Context, extraCols: List[SqlColumn], extraJoins: List[SqlJoin]): SqlQuery

    /** The columns of this query */
    def cols: List[SqlColumn]

    /** The codecs corresponding to the columns of this query */
    def codecs: List[(Boolean, Codec)]

    /** Yields a copy of this query with all occurences of `from` replaced by `to` */
    def subst(from: TableExpr, to: TableExpr): SqlQuery

    /** Nest this query as a subobject in the enclosing `parentContext` */
    def nest(
      parentContext: Context,
      extraCols: List[SqlColumn],
      oneToOne: Boolean,
      lateral: Boolean
    ): SqlQuery

    /** Add WHERE, ORDER BY, OFFSET and LIMIT to this query */
    def addFilterOrderByOffsetLimit(
      filter: Option[(Predicate, List[SqlJoin])],
      orderBy: Option[(List[OrderSelection[_]], List[SqlJoin])],
      offset: Option[Int],
      limit: Option[Int],
      predIsOneToOne: Boolean,
      parentConstraints: List[(SqlColumn, SqlColumn)]
    ): Option[SqlQuery]

    /** Yields an equivalent query encapsulating this query as a subquery */
    def toSubquery(name: String, lateral: Boolean): SqlSelect
    /** Yields an equivalent query encapsulating this query as a common table expression */
    def toWithQuery(name: String, refName: Option[String]): SqlSelect

    /** Yields a collection of `SqlSelects` which when combined as a union are equivalent to this query */
    def asSelects: List[SqlSelect] =
      this match {
        case ss: SqlSelect => ss :: Nil
        case su: SqlUnion => su.elems
      }

    /** Is this query an SQL Union */
    def isUnion: Boolean

    /** Does one row of this query correspond to exactly one complete GraphQL value */
    def oneToOne: Boolean

    /** Render this query as a `Fragment` */
    def toFragment: Aliased[Fragment]
  }

  object SqlQuery {
    /** Combine the given queries as a single SQL query */
    def combineAll(queries: List[SqlQuery]): Option[SqlQuery] = {
      if(queries.isEmpty) None
      else {
        val (selects, unions) =
          queries.partitionMap {
            case s: SqlSelect => Left(s)
            case u: SqlUnion => Right(u)
          }

        def combineSelects(sels: List[SqlSelect]): SqlSelect = {
          val fst = sels.head
          val withs = sels.flatMap(_.withs).distinct
          val cols = sels.flatMap(_.cols).distinct
          val joins = sels.flatMap(_.joins).distinct
          val wheres = sels.flatMap(_.wheres).distinct
          fst.copy(withs = withs, cols = cols, joins = joins, wheres = wheres)
        }

        val unionSelects = unions.flatMap(_.elems).distinct
        val allSelects = selects ++ unionSelects

        val ctx = allSelects.head.context

        assert(allSelects.forall(sel => sel.context == ctx && sel.limit.isEmpty && sel.offset.isEmpty && sel.orders.isEmpty && !sel.isDistinct))

        def combineCompatible(sels: List[SqlSelect]): List[SqlSelect] = {
          val (oneToOneSelects, multiRowSelects) = sels.partition(_.oneToOne)

          (multiRowSelects, oneToOneSelects) match {
            case (Nil, Nil) => Nil
            case (Nil, sel :: Nil) => sel :: Nil
            case (Nil, sels) => combineSelects(sels) :: Nil
            case (psels, Nil) => psels
            case (psels, sels) =>
              psels.map(psel => combineSelects(psel :: sels))
          }
        }

        val combinedSelects = selects.groupBy(sel => sel.table).values.flatMap(combineCompatible).toList

        (combinedSelects ++ unionSelects) match {
          case Nil => None
          case sel :: Nil => Some(sel)
          case sels => Some(SqlUnion(sels))
        }
      }
    }

    /** Compute the set of paths traversed by the given prediate */
    def wherePaths(pred: Predicate): List[List[String]] = {
      def loop(term: Term[_], acc: List[List[String]]): List[List[String]] = {
        term match {
          case Const(_)         => acc
          case pathTerm: Path   => pathTerm.path :: acc
          case And(x, y)        => loop(y, loop(x, acc))
          case Or(x, y)         => loop(y, loop(x, acc))
          case Not(x)           => loop(x, acc)
          case Eql(x, y)        => loop(y, loop(x, acc))
          case NEql(x, y)       => loop(y, loop(x, acc))
          case Contains(x, y)   => loop(y, loop(x, acc))
          case Lt(x, y)         => loop(y, loop(x, acc))
          case LtEql(x, y)      => loop(y, loop(x, acc))
          case Gt(x, y)         => loop(y, loop(x, acc))
          case GtEql(x, y)      => loop(y, loop(x, acc))
          case IsNull(x, _)     => loop(x, acc)
          case In(x, _)         => loop(x, acc)
          case AndB(x, y)       => loop(y, loop(x, acc))
          case OrB(x, y)        => loop(y, loop(x, acc))
          case XorB(x, y)       => loop(y, loop(x, acc))
          case NotB(x)          => loop(x, acc)
          case Matches(x, _)    => loop(x, acc)
          case StartsWith(x, _) => loop(x, acc)
          case ToUpperCase(x)   => loop(x, acc)
          case ToLowerCase(x)   => loop(x, acc)
          case Like(x, _, _)    => loop(x, acc)
          case _                => acc
        }
      }

      loop(pred, Nil)
    }

    /** Compute the set of columns referred to by the given prediate */
    def whereCols(f: Term[_] => SqlColumn, pred: Predicate): List[SqlColumn] = {
      def loop[T](term: T): List[SqlColumn] =
        term match {
          case _: Path          => f(term.asInstanceOf[Term[_]]) :: Nil
          case _: SqlColumnTerm => f(term.asInstanceOf[Term[_]]) :: Nil
          case Const(_)         => Nil
          case And(x, y)        => loop(x) ++ loop(y)
          case Or(x, y)         => loop(x) ++ loop(y)
          case Not(x)           => loop(x)
          case Eql(x, y)        => loop(x) ++ loop(y)
          case NEql(x, y)       => loop(x) ++ loop(y)
          case Contains(x, y)   => loop(x) ++ loop(y)
          case Lt(x, y)         => loop(x) ++ loop(y)
          case LtEql(x, y)      => loop(x) ++ loop(y)
          case Gt(x, y)         => loop(x) ++ loop(y)
          case GtEql(x, y)      => loop(x) ++ loop(y)
          case IsNull(x, _)     => loop(x)
          case In(x, _)         => loop(x)
          case AndB(x, y)       => loop(x) ++ loop(y)
          case OrB(x, y)        => loop(x) ++ loop(y)
          case XorB(x, y)       => loop(x) ++ loop(y)
          case NotB(x)          => loop(x)
          case Matches(x, _)    => loop(x)
          case StartsWith(x, _) => loop(x)
          case ToUpperCase(x)   => loop(x)
          case ToLowerCase(x)   => loop(x)
          case Like(x, _, _)    => loop(x)
          case _                => Nil
        }

      loop(pred)
    }

    /** Contextualise all terms in the given `Predicate` to the given context and owner */
    def contextualiseWhereTerms(context: Context, owner: ColumnOwner, pred: Predicate): Predicate = {
      def loop[T](term: T): T =
        (term match {
          case _: Path          => SqlColumnTerm(contextualiseTerm(context, owner, term.asInstanceOf[Term[_]]))
          case _: SqlColumnTerm => SqlColumnTerm(contextualiseTerm(context, owner, term.asInstanceOf[Term[_]]))
          case Const(_)         => term
          case And(x, y)        => And(loop(x), loop(y))
          case Or(x, y)         => Or(loop(x), loop(y))
          case Not(x)           => Not(loop(x))
          case e@Eql(x, y)      => e.subst(loop(x), loop(y))
          case n@NEql(x, y)     => n.subst(loop(x), loop(y))
          case c@Contains(x, y) => c.subst(loop(x), loop(y))
          case l@Lt(x, y)       => l.subst(loop(x), loop(y))
          case l@LtEql(x, y)    => l.subst(loop(x), loop(y))
          case g@Gt(x, y)       => g.subst(loop(x), loop(y))
          case g@GtEql(x, y)    => g.subst(loop(x), loop(y))
          case IsNull(x, y)     => IsNull(loop(x), y)
          case i@In(x, _)       => i.subst(loop(x))
          case AndB(x, y)       => AndB(loop(x), loop(y))
          case OrB(x, y)        => OrB(loop(x), loop(y))
          case XorB(x, y)       => XorB(loop(x), loop(y))
          case NotB(x)          => NotB(loop(x))
          case Matches(x, y)    => Matches(loop(x), y)
          case StartsWith(x, y) => StartsWith(loop(x), y)
          case ToUpperCase(x)   => ToUpperCase(loop(x))
          case ToLowerCase(x)   => ToLowerCase(loop(x))
          case Like(x, y, z)    => Like(loop(x), y, z)
          case _                => term
        }).asInstanceOf[T]

      loop(pred)
    }

    /** Yields a copy of the given `Predicate` with all occurences of `from` replaced by `to` */
    def substWhereTables(from: TableExpr, to: TableExpr, pred: Predicate): Predicate = {
      def loop[T](term: T): T =
        (term match {
          case SqlColumnTerm(col) => SqlColumnTerm(col.subst(from, to))
          case _: Path            => term
          case Const(_)           => term
          case And(x, y)          => And(loop(x), loop(y))
          case Or(x, y)           => Or(loop(x), loop(y))
          case Not(x)             => Not(loop(x))
          case e@Eql(x, y)        => e.subst(loop(x), loop(y))
          case n@NEql(x, y)       => n.subst(loop(x), loop(y))
          case c@Contains(x, y)   => c.subst(loop(x), loop(y))
          case l@Lt(x, y)         => l.subst(loop(x), loop(y))
          case l@LtEql(x, y)      => l.subst(loop(x), loop(y))
          case g@Gt(x, y)         => g.subst(loop(x), loop(y))
          case g@GtEql(x, y)      => g.subst(loop(x), loop(y))
          case IsNull(x, y)       => IsNull(loop(x), y)
          case i@In(x, _)         => i.subst(loop(x))
          case AndB(x, y)         => AndB(loop(x), loop(y))
          case OrB(x, y)          => OrB(loop(x), loop(y))
          case XorB(x, y)         => XorB(loop(x), loop(y))
          case NotB(x)            => NotB(loop(x))
          case Matches(x, y)      => Matches(loop(x), y)
          case StartsWith(x, y)   => StartsWith(loop(x), y)
          case ToUpperCase(x)     => ToUpperCase(loop(x))
          case ToLowerCase(x)     => ToLowerCase(loop(x))
          case Like(x, y, z)      => Like(loop(x), y, z)
          case _                  => term
        }).asInstanceOf[T]

      loop(pred)
    }

    /** Render the given `Predicates` as a where clause `Fragment` */
    def wheresToFragment(context: Context, wheres: List[Predicate]): Aliased[Fragment] = {
      def encoder1(enc: Option[Encoder], x: Term[_]): Encoder =
        enc.orElse(encoderForTerm(context, x)).getOrElse(sys.error(s"No encoder for term $x"))

      def encoder2(enc: Option[Encoder], x: Term[_], y: Term[_]): Encoder =
        enc.orElse(encoderForTerm(context, x)).orElse(encoderForTerm(context, y)).getOrElse(sys.error(s"No encoder for terms $x or $y"))

      def loop(term: Term[_], e: Encoder): Aliased[Fragment] = {

        def unaryOp(x: Term[_])(op: Fragment, enc: Option[Encoder]): Aliased[Fragment] = {
          val e = encoder1(enc, x)
          for {
            fx <- loop(x, e)
          } yield op |+| fx
        }

        def binaryOp(x: Term[_], y: Term[_])(op: Fragment, enc: Option[Encoder] = None): Aliased[Fragment] = {
          val e = encoder2(enc, x, y)
          for {
            fx <- loop(x, e)
            fy <- loop(y, e)
          } yield fx |+| op |+| fy
        }

        def binaryOp2(x: Term[_])(op: Fragment => Fragment, enc: Option[Encoder] = None): Aliased[Fragment] = {
          val e = encoder1(enc, x)
          for {
            fx <- loop(x, e)
          } yield op(fx)
        }

        term match {
          case Const(value) =>
            Aliased.pure(Fragments.bind(e, value))

          case SqlColumnTerm(col) =>
            col.toRefFragment(false)

          case pathTerm: Path =>
            sys.error(s"Unresolved term $pathTerm in WHERE clause")

          case True =>
            Aliased.pure(Fragments.const("true"))

          case False =>
            Aliased.pure(Fragments.const("false"))

          case And(x, y) =>
            binaryOp(x, y)(Fragments.const(" AND "), Some(booleanEncoder))

          case Or(x, y) =>
            binaryOp(x, y)(Fragments.const(" OR "), Some(booleanEncoder))

          case Not(x) =>
            unaryOp(x)(Fragments.const(" NOT "), Some(booleanEncoder))

          case Eql(x, y) =>
            binaryOp(x, y)(Fragments.const(" = "))

          case Contains(x, y) =>
            binaryOp(x, y)(Fragments.const(" = "))

          case NEql(x, y) =>
            binaryOp(x, y)(Fragments.const(" != "))

          case Lt(x, y) =>
            binaryOp(x, y)(Fragments.const(" < "))

          case LtEql(x, y) =>
            binaryOp(x, y)(Fragments.const(" <= "))

          case Gt(x, y) =>
            binaryOp(x, y)(Fragments.const(" > "))

          case GtEql(x, y) =>
            binaryOp(x, y)(Fragments.const(" >= "))

          case In(x, y) =>
            val e = encoder1(None, x)
            val ys = NonEmptyList.fromList(y).getOrElse(sys.error(s"At least one alternative required in for In"))
            binaryOp2(x)(fx => Fragments.in(fx, ys, e))

          case AndB(x, y) =>
            binaryOp(x, y)(Fragments.const(" & "), Some(intEncoder))

          case OrB(x, y) =>
            binaryOp(x, y)(Fragments.const(" | "), Some(intEncoder))

          case XorB(x, y) =>
            binaryOp(x, y)(Fragments.const(" # "), Some(intEncoder))

          case NotB(x) =>
            unaryOp(x)(Fragments.const(" NOT "), Some(intEncoder))

          case Matches(x, regex) =>
            binaryOp2(x)(
              fx =>
                Fragments.const("regexp_matches(") |+|
                fx |+| Fragments.const(s", ") |+| Fragments.bind(stringEncoder, regex.toString) |+|
                Fragments.const(s")"),
              Some(stringEncoder)
            )

          case StartsWith(x, prefix) =>
            binaryOp2(x)(
              fx =>
                fx |+| Fragments.const(s" LIKE ") |+| Fragments.bind(stringEncoder, prefix + "%"),
              Some(stringEncoder)
            )

          case ToUpperCase(x) =>
            binaryOp2(x)(
              fx =>
                Fragments.const("upper(") |+| fx |+| Fragments.const(s")"),
              Some(stringEncoder)
            )

          case ToLowerCase(x) =>
            binaryOp2(x)(
              fx =>
                Fragments.const("lower(") |+| fx |+| Fragments.const(s")"),
              Some(stringEncoder)
            )

          case IsNull(x, isNull) =>
            val sense = if (isNull) "" else "NOT"
            binaryOp2(x)(
              fx =>
                fx |+| Fragments.const(s" IS $sense NULL "),
            )

          case Like(x, pattern, caseInsensitive) =>
            val op = if(caseInsensitive) " ILIKE " else " LIKE "
            binaryOp2(x)(
              fx =>
                fx |+| Fragments.const(s" $op ") |+| Fragments.bind(stringEncoder, pattern),
              Some(stringEncoder)
            )

          case other => sys.error(s"Unexpected term $other")
        }
      }

      wheres.traverse(pred => loop(pred, booleanEncoder)).map(fwheres => Fragments.const(" ") |+| Fragments.whereAnd(fwheres: _*))
    }


    /** Contextualise all terms in the given `OrderSelection` to the given context and owner */
    def contextualiseOrderTerms[T](context: Context, owner: ColumnOwner, os: OrderSelection[T]): OrderSelection[T] =
      os.subst(SqlColumnTerm(contextualiseTerm(context, owner, os.term)).asInstanceOf[Term[T]])

    /** Yields a copy of the given `OrderSelection` with all occurences of `from` replaced by `to` */
    def substOrderTables[T](from: TableExpr, to: TableExpr, os: OrderSelection[T]): OrderSelection[T] =
      os.term match {
        case SqlColumnTerm(col) => os.subst(SqlColumnTerm(col.subst(from, to)))
        case _ => os
      }

    /** Render the given `OrderSelections` as a `Fragment` */
    def ordersToFragment(orders: List[OrderSelection[_]]): Aliased[Fragment] =
      if (orders.isEmpty) Aliased.pure(Fragments.empty)
      else
        orders.traverse {
          case OrderSelection(term, ascending, nullsLast) =>
            val col = term match {
              case SqlColumnTerm(col) => col
              case other => sys.error(s"Unresolved term $other in ORDER BY")
            }
            val dir = if(ascending) "" else " DESC"
            val nulls = s" NULLS ${if(nullsLast) "LAST" else "FIRST"}"
            for {
              fc <- col.toRefFragment(Fragments.needsCollation(col.codec))
            } yield {
              fc |+| Fragments.const(s"$dir$nulls")
            }
        }.map(forders => Fragments.const(" ORDER BY ") |+| forders.intercalate(Fragments.const(",")))

    /** Yield a copy of the given `Term` with all referenced `SqlColumns` relativised to the given
     *  context and owned by by the given owner */
    def contextualiseTerm(context: Context, owner: ColumnOwner, term: Term[_]): SqlColumn = {
      def subst(col: SqlColumn): SqlColumn =
        if(!owner.owns(col)) col
        else col.derive(owner)

      term match {
        case SqlColumnTerm(col) => subst(col)
        case pathTerm: Path =>
          columnForSqlTerm(context, pathTerm).map(subst).getOrElse(sys.error(s"No column for term $pathTerm"))

        case other =>
          sys.error(s"Expected contextualisable term but found $other")
      }
    }

    /** Representation of an SQL SELECT */
    case class SqlSelect(
      context:   Context,                  // the GraphQL context of the query
      withs:     List[WithRef],            // the common table expressions
      table:     TableExpr,                // the table/subquery
      cols:      List[SqlColumn],          // the requested columns
      joins:     List[SqlJoin],            // joins for predicates/subobjects
      wheres:    List[Predicate],
      orders:    List[OrderSelection[_]],
      offset:    Option[Int],
      limit:     Option[Int],
      distinct:  List[SqlColumn],          // columns this query is DISTINCT on
      oneToOne:  Boolean,                  // does one row represent exactly one complete GraphQL value
      predicate: Boolean                   // does this SqlSelect represent a predicate
    ) extends SqlQuery {
      assert(SqlJoin.checkOrdering(table, joins))
      assert(cols.forall(owns0))
      assert(cols.nonEmpty)
      assert(cols.size == cols.distinct.size)
      assert(joins.size == joins.distinct.size)
      assert(distinct.diff(cols).isEmpty)

      private def owns0(col: SqlColumn): Boolean =
        isSameOwner(col.owner) || table.owns(col) || withs.exists(_.owns(col)) || joins.exists(_.owns(col))

      /** This query in the given context */
      def withContext(context: Context, extraCols: List[SqlColumn], extraJoins: List[SqlJoin]): SqlSelect =
        copy(context = context, cols = (cols ++ extraCols).distinct, joins = extraJoins ++ joins)

      def isUnion: Boolean = table.isUnion

      def isDistinct: Boolean = distinct.nonEmpty

      override def isSameOwner(other: ColumnOwner): Boolean = other.isSameOwner(TableRef(context, table.name))

      def owns(col: SqlColumn): Boolean = cols.contains(col) || owns0(col)
      def contains(other: ColumnOwner): Boolean = isSameOwner(other) || table.contains(other) || joins.exists(_.contains(other)) || withs.exists(_.contains(other))

      def directlyOwns(col: SqlColumn): Boolean =
        (table match {
          case tr: TableRef => tr.directlyOwns(col)
          case _ => false
        }) || joins.exists(_.directlyOwns(col)) || withs.exists(_.directlyOwns(col))

      def findNamedOwner(col: SqlColumn): Option[TableExpr] =
        table.findNamedOwner(col).orElse(joins.collectFirstSome(_.findNamedOwner(col))).orElse(withs.collectFirstSome(_.findNamedOwner(col)))

      def codecs: List[(Boolean, Codec)] =
        if (isUnion)
          cols.map(col => (true, col.codec))
        else {
          def nullable(col: SqlColumn): Boolean =
            !col.owner.isSameOwner(table)

          cols.map(col => (nullable(col), col.codec))
        }

      /** The columns, if any, on which this select is ordered */
      val orderCols: List[SqlColumn] = orders.map(os => columnForSqlTerm(context, os.term).getOrElse(sys.error(s"No column for ${os.term}"))).distinct

      /** Does the given column need collation? */
      def needsCollation(col: SqlColumn): Boolean =
        Fragments.needsCollation(col.codec) && orderCols.contains(col)

      /** Yield a name for this select derived from any names associated with its
       *  from clauses or joins
       */
      def syntheticName(suffix: String): String = {
        val joinNames = joins.map(_.child.name)
        (table.name :: joinNames).mkString("_").take(50-suffix.length)+suffix
      }

      /** Yields a copy of this select with all occurences of `from` replaced by `to` */
      def subst(from: TableExpr, to: TableExpr): SqlSelect = {
        copy(
          withs = withs.map(_.subst(from, to)),
          table = if(table.isSameOwner(from)) to else table.subst(from, to),
          cols = cols.map(_.subst(from, to)),
          joins = joins.map(_.subst(from, to)),
          wheres = wheres.map(o => substWhereTables(from, to, o)),
          orders = orders.map(o => substOrderTables(from, to, o)),
          distinct = distinct.map(_.subst(from, to))
        )
      }

      /** Nest this query as a subobject in the enclosing `parentContext` */
      def nest(
        parentContext: Context,
        extraCols:     List[SqlColumn],
        oneToOne:      Boolean,
        lateral:       Boolean
      ): SqlSelect = {
        val parentTable: TableRef =
          parentTableForType(parentContext).
            getOrElse(sys.error(s"No parent table for type ${parentContext.tpe}"))

        val inner = !context.tpe.isNullable && !context.tpe.isList

        def mkSubquery(multiTable: Boolean, nested: SqlSelect, joinCol: SqlColumn, suffix: String): SqlSelect = {
          def isMergeable: Boolean =
            !multiTable && !nested.joins.exists(_.isPredicate) && nested.wheres.isEmpty && nested.orders.isEmpty && nested.offset.isEmpty && nested.limit.isEmpty && !nested.isDistinct

          if(isMergeable) nested
          else {
            val exposeCol = nested.table match {
              case _: TableRef => joinCol :: Nil
              case _ => Nil
            }
            val base0 = nested.copy(cols = (exposeCol ++ nested.cols).distinct).toSubquery(syntheticName(suffix), lateral)
            base0.copy(cols = nested.cols.map(_.derive(base0.table)))
          }
        }

        def mkJoins(joins0: List[Join], multiTable: Boolean): SqlSelect = {
          val base = mkSubquery(multiTable, this, SqlColumn.TableColumn(table, joins0.last.child), "_nested")

          val initialJoins =
            joins0.init.map { j =>
              val parentTable = TableRef(parentContext, j.parent.table)
              val parentCol = SqlColumn.TableColumn(parentTable, j.parent)
              val childTable = TableRef(parentContext, j.child.table)
              val childCol = SqlColumn.TableColumn(childTable, j.child)
              SqlJoin(
                parentTable,
                childTable,
                List((parentCol, childCol)),
                inner
              )
            }

          val finalJoins = {
            val lastJoin = joins0.last
            val parentTable = TableRef(parentContext, lastJoin.parent.table)
            val parentCol = SqlColumn.TableColumn(parentTable, lastJoin.parent)

            if(!isAssociative(context)) {
              val childCol = SqlColumn.TableColumn(base.table, lastJoin.child)
              val finalJoin =
                SqlJoin(
                  parentTable,
                  base.table,
                  List((parentCol, childCol)),
                  inner
                )
              finalJoin :: Nil
            } else {
              val assocTable = TableExpr.DerivedTableRef(context, Some(base.table.name+"_assoc"), base.table, true)
              val childCol = SqlColumn.TableColumn(assocTable, lastJoin.child)

              val assocJoin =
                SqlJoin(
                  parentTable,
                  assocTable,
                  List((parentCol, childCol)),
                  inner
                )

              val finalJoin =
                SqlJoin(
                  assocTable,
                  base.table,
                  keyColumnsForType(context).map { key => (key.in(assocTable), key) },
                  false
                )
              List(assocJoin, finalJoin)
            }
          }

          val allJoins = initialJoins ++ finalJoins ++ base.joins

          SqlSelect(
            context = parentContext,
            withs = base.withs,
            table = allJoins.head.parent,
            cols = base.cols,
            joins = allJoins,
            wheres = base.wheres,
            orders = Nil,
            offset = None,
            limit = None,
            distinct = Nil,
            oneToOne = oneToOne,
            predicate = false
          )
        }

        val fieldName = context.path.head
        val nested =
          fieldMapping(parentContext, fieldName) match {
            case Some(_: CursorFieldJson) | Some(SqlObject(_, Nil)) =>
              val embeddedCols = cols.map { col =>
                if(table.owns(col)) SqlColumn.EmbeddedColumn(parentTable, col)
                else col
              }
              val embeddedJoins = joins.map { join =>
                if(join.parent.isSameOwner(table)) {
                  val newOn = join.on match {
                    case (p, c) :: tl => (SqlColumn.EmbeddedColumn(parentTable, p), c) :: tl
                    case _ => join.on
                  }
                  join.copy(parent = parentTable, on = newOn)
                } else join
              }
              copy(
                context = parentContext,
                table = parentTable,
                cols = embeddedCols,
                joins = embeddedJoins
              )

            case Some(SqlObject(_, single@(_ :: Nil))) =>
              mkJoins(single, false)

            case Some(SqlObject(_, firstJoin :: tail)) =>
              val nested = mkJoins(tail, true)
              val base = mkSubquery(false, nested, SqlColumn.TableColumn(nested.table, firstJoin.child), "_multi")

              val initialJoin = {
                val parentCol = SqlColumn.TableColumn(parentTable, firstJoin.parent)
                val childCol = SqlColumn.TableColumn(base.table, firstJoin.child)
                  SqlJoin(
                    parentTable,
                    base.table,
                    List((parentCol, childCol)),
                    inner
                  )
              }

              SqlSelect(
                context = parentContext,
                withs = base.withs,
                table = parentTable,
                cols = base.cols,
                joins = initialJoin :: base.joins,
                wheres = base.wheres,
                orders = Nil,
                offset = None,
                limit = None,
                distinct = Nil,
                oneToOne = oneToOne,
                predicate = false
              )

            case _ => None
              sys.error(s"Non-subobject mapping for field '$fieldName' of type ${parentContext.tpe}")
          }

        assert(cols.lengthCompare(nested.cols) == 0)
        nested.copy(cols = (nested.cols ++ extraCols).distinct)
      }

      /** Add WHERE, ORDER BY and LIMIT to this query */
      def addFilterOrderByOffsetLimit(
        filter:  Option[(Predicate, List[SqlJoin])],
        orderBy: Option[(List[OrderSelection[_]], List[SqlJoin])],
        offset0: Option[Int],
        limit0: Option[Int],
        predIsOneToOne: Boolean,
        parentConstraints: List[(SqlColumn, SqlColumn)]
      ): Option[SqlSelect] = {
        assert(orders.isEmpty && offset.isEmpty && limit.isEmpty && !isDistinct)
        assert(filter.isDefined || orderBy.isDefined || offset0.isDefined || limit0.isDefined)
        assert(filter.map(f => isSqlTerm(context, f._1)).getOrElse(true))

        val keyCols = keyColumnsForType(context)
        val (pred, filterJoins) = filter.map { case (pred, joins) => (pred :: Nil, joins) }.getOrElse((Nil, Nil))
        val (oss, orderJoins) = orderBy.map { case (oss, joins) => (oss, joins) }.getOrElse((Nil, Nil))
        val orderCols0 = oss.map(os => columnForSqlTerm(context, os.term).getOrElse(sys.error(s"No column for term ${os.term}")))
        val orderCols = orderCols0.map(col => orderJoins.collectFirstSome(_.findNamedOwner(col)).map(owner => col.in(owner)).getOrElse(col.in(table)))

        val pred0 = parentConstraints.map {
          case (p, c) => Eql(p.toTerm, c.toTerm)
        } ++ pred

        val useWindow = parentConstraints.nonEmpty && (offset0.isDefined || limit0.isDefined)
        val useFlattenedWindow = true

        if (useWindow) {
          // Use window functions for offset and limit where we have a parent constraint ...

          def mkWindowPred(partitionTerm: Term[Int]): Predicate =
            (offset0, limit0) match {
              case (Some(off), Some(lim)) =>
                And(GtEql(partitionTerm, Const(off)), LtEql(partitionTerm, Const(off+lim)))
              case (None, Some(lim)) =>
                LtEql(partitionTerm, Const(lim))
              case (Some(off), None) =>
                GtEql(partitionTerm, Const(off))
              case (None, None) => True
            }

          val (_, partitionBy) = parentConstraints.head

          if(oneToOne && predIsOneToOne) {
            // Case 1) one row is one object in this context
            val nonNullKeys = keyCols.map(col => IsNull(col.toTerm, false))

            val pred1 = pred0.map(p => contextualiseWhereTerms(context, table, p))
            val oss0 = oss.map(os => contextualiseOrderTerms(context, table, os))
            val orders = oss0 ++ keyCols.diff(orderCols).map(col => OrderSelection(col.toTerm, true, true))

            // We could use row_number in this case
            val partitionCol = SqlColumn.PartitionColumn(table, "row_item", partitionBy :: Nil, orders)
            val exposeCol = parentConstraints.lastOption.map {
              case (_, col) => findNamedOwner(col).map(owner => col.derive(owner)).getOrElse(col.derive(table))
            }.toList

            val selWithRowItem =
              SqlSelect(
                context = context,
                withs = withs,
                table = table,
                cols = (partitionCol :: exposeCol ++ cols ++ orderCols).distinct,
                joins = (filterJoins ++ orderJoins ++ joins).distinct,
                wheres = (pred1 ++ nonNullKeys ++ wheres).distinct,
                orders = Nil,
                offset = None,
                limit = None,
                distinct = distinct,
                oneToOne = true,
                predicate = true
              )

            val numberedName = syntheticName("_numbered")
            val subWithRowItem = SubqueryRef(context, numberedName, selWithRowItem, true)

            val partitionTerm = partitionCol.derive(subWithRowItem).toTerm.asInstanceOf[Term[Int]]
            val windowPred = mkWindowPred(partitionTerm)

            val predQuery =
              SqlSelect(
                context = context,
                withs = Nil,
                table = subWithRowItem,
                cols = (cols ++ orderCols).distinct.map(_.derive(subWithRowItem)),
                joins = Nil,
                wheres = windowPred :: Nil,
                orders = Nil,
                offset = None,
                limit = None,
                distinct = Nil,
                oneToOne = true,
                predicate = true
              )

            Some(predQuery)
          } else if (useFlattenedWindow && (orderBy.isEmpty || orderCols.take(keyCols.size).diff(keyCols).isEmpty) && predIsOneToOne) {
            // Case 2a) No order, or key columns at the start of the order; simple predicate means we can elide a subquery
            val nonNullKeys = keyCols.map(col => IsNull(col.toTerm, false))

            val pred1 = pred0.map(p => contextualiseWhereTerms(context, table, p))
            val oss0 = oss.map(os => contextualiseOrderTerms(context, table, os))
            val orders = oss0 ++ keyCols.diff(orderCols).map(col => OrderSelection(col.toTerm, true, true))

            val partitionCol = SqlColumn.PartitionColumn(table, "row_item", partitionBy :: Nil, orders)
            val exposeCol = parentConstraints.lastOption.map {
              case (_, col) => findNamedOwner(col).map(owner => col.derive(owner)).getOrElse(col.derive(table))
            }.toList

            val selWithRowItem =
              SqlSelect(
                context = context,
                withs = withs,
                table = table,
                cols = (partitionCol :: exposeCol ++ cols ++ orderCols).distinct,
                joins = joins,
                wheres = (pred1 ++ nonNullKeys ++ wheres).distinct,
                orders = Nil,
                offset = None,
                limit = None,
                distinct = distinct,
                oneToOne = oneToOne,
                predicate = true
              )

            val numberedName = syntheticName("_numbered")
            val subWithRowItem = SubqueryRef(context, numberedName, selWithRowItem, true)

            val partitionTerm = partitionCol.derive(subWithRowItem).toTerm.asInstanceOf[Term[Int]]
            val windowPred = mkWindowPred(partitionTerm)

            val predQuery =
              SqlSelect(
                context = context,
                withs = Nil,
                table = subWithRowItem,
                cols = (cols ++ orderCols).distinct.map(_.derive(subWithRowItem)),
                joins = Nil,
                wheres = windowPred :: Nil,
                orders = Nil,
                offset = None,
                limit = None,
                distinct = Nil,
                oneToOne = oneToOne,
                predicate = true
              )

            Some(predQuery)
          } else if (orderBy.isEmpty || orderCols.take(keyCols.size).diff(keyCols).isEmpty) {
            // Case 2b) No order, or key columns at the start of the order
            val base0 = subqueryToWithQuery
            val baseRef = base0.table

            val predCols = keyCols.map(_.derive(baseRef))
            val nonNullKeys = predCols.map(col => IsNull(col.toTerm, false))

            val pred1 = pred0.map(p => contextualiseWhereTerms(context, baseRef, p))
            val oss0 = oss.map(os => contextualiseOrderTerms(context, baseRef, os))
            val orders = oss0 ++ keyCols.diff(orderCols).map(col => OrderSelection(col.derive(baseRef).toTerm, true, true))

            val partitionCol = SqlColumn.PartitionColumn(table, "row_item", partitionBy :: Nil, orders)
            val exposeCol = parentConstraints.lastOption.map {
              case (_, col) => baseRef.findNamedOwner(col).map(owner => col.derive(owner)).getOrElse(col.derive(baseRef))
            }.toList

            val selWithRowItem =
              SqlSelect(
                context = context,
                withs = Nil,
                table = baseRef,
                cols = partitionCol :: exposeCol ++ predCols,
                joins = (filterJoins ++ orderJoins).distinct,
                wheres = (pred1 ++ nonNullKeys ++ wheres).distinct,
                orders = Nil,
                offset = None,
                limit = None,
                distinct = Nil,
                oneToOne = true,
                predicate = true
              )

            val numberedName = syntheticName("_numbered")
            val subWithRowItem = SubqueryRef(context, numberedName, selWithRowItem, true)

            val partitionTerm = partitionCol.derive(subWithRowItem).toTerm.asInstanceOf[Term[Int]]
            val windowPred = mkWindowPred(partitionTerm)

            val numberedPredCols = keyCols.map(_.derive(subWithRowItem))

            val predQuery =
              SqlSelect(
                context = context,
                withs = Nil,
                table = subWithRowItem,
                cols = numberedPredCols,
                joins = Nil,
                wheres = windowPred :: Nil,
                orders = Nil,
                offset = None,
                limit = None,
                distinct = Nil,
                oneToOne = true,
                predicate = true
              )

            val predName = syntheticName("_pred")
            val predSub = SubqueryRef(context, predName, predQuery, parentConstraints.nonEmpty)

            val on = keyCols.map(key => (key.derive(baseRef), key.derive(predSub)))
            val predJoin = SqlJoin(baseRef, predSub, on, true)

            val joinCols = cols.filterNot(col => baseRef.owns(col))

            val base = base0.copy(
              table = baseRef,
              cols = (base0.cols ++ joinCols).distinct,
              joins = predJoin :: base0.joins,
              wheres = Nil
            )

            Some(base)
          } else if (useFlattenedWindow && predIsOneToOne) {
            // Case 3a) There is an order orthogonal to the key; simple predicate means we can elide a subquery
            val nonNullKeys = keyCols.map(col => IsNull(col.toTerm, false))

            val pred1 = pred0.map(p => contextualiseWhereTerms(context, table, p))
            val oss0 = oss.map(os => contextualiseOrderTerms(context, table, os))
            val orders = oss0 ++ keyCols.diff(orderCols).map(col => OrderSelection(col.toTerm, true, true))

            val distOrders =
              keyCols.map(col => OrderSelection(col.toTerm, true, true)) ++
              oss0.filterNot(os => keyCols.contains(columnForSqlTerm(context, os.term).getOrElse(sys.error(s"No column for term ${os.term}"))))

            val partitionCol = SqlColumn.PartitionColumn(table, "row_item", partitionBy :: Nil, orders)
            val distPartitionCol = SqlColumn.PartitionColumn(table, "row_item_dist", partitionBy :: keyCols, distOrders)
            val exposeCol = parentConstraints.lastOption.map {
              case (_, col) => findNamedOwner(col).map(owner => col.derive(owner)).getOrElse(col.derive(table))
            }.toList

            val selWithRowItem =
              SqlSelect(
                context = context,
                withs = withs,
                table = table,
                cols = (partitionCol :: distPartitionCol :: exposeCol ++ cols ++ orderCols).distinct,
                joins = joins,
                wheres = (pred1 ++ nonNullKeys ++ wheres).distinct,
                orders = Nil,
                offset = None,
                limit = None,
                distinct = distinct,
                oneToOne = oneToOne,
                predicate = true
              )

            val numberedName = syntheticName("_numbered")
            val subWithRowItem = SubqueryRef(context, numberedName, selWithRowItem, true)

            val partitionTerm = partitionCol.derive(subWithRowItem).toTerm.asInstanceOf[Term[Int]]
            val distPartitionTerm = distPartitionCol.derive(subWithRowItem).toTerm.asInstanceOf[Term[Int]]
            val windowPred = And(mkWindowPred(partitionTerm), LtEql(distPartitionTerm, Const(1)))

            val predQuery =
              SqlSelect(
                context = context,
                withs = Nil,
                table = subWithRowItem,
                cols = (cols ++ orderCols).distinct.map(_.derive(subWithRowItem)),
                joins = Nil,
                wheres = windowPred :: Nil,
                orders = Nil,
                offset = None,
                limit = None,
                distinct = Nil,
                oneToOne = oneToOne,
                predicate = true
              )

            Some(predQuery)
          } else {
            // Case 3b) There is an order orthogonal to the key
            val base0 = subqueryToWithQuery
            val baseRef = base0.table

            val predCols = keyCols.map(_.derive(baseRef))
            val nonNullKeys = predCols.map(col => IsNull(col.toTerm, false))

            val pred1 = pred0.map(p => contextualiseWhereTerms(context, baseRef, p))
            val oss0 = oss.map(os => contextualiseOrderTerms(context, baseRef, os))
            val orders = oss0 ++ keyCols.diff(orderCols).map(col => OrderSelection(col.derive(baseRef).toTerm, true, true))

            val distOrders =
              keyCols.map(col => OrderSelection(col.derive(baseRef).toTerm, true, true)) ++
              oss0.filterNot(os => keyCols.contains(columnForSqlTerm(context, os.term).getOrElse(sys.error(s"No column for term ${os.term}"))))

            val partitionCol = SqlColumn.PartitionColumn(table, "row_item", partitionBy :: Nil, orders)
            val distPartitionCol = SqlColumn.PartitionColumn(table, "row_item_dist", partitionBy :: predCols, distOrders)

            val exposeCol = parentConstraints.lastOption.map {
              case (_, col) => baseRef.findNamedOwner(col).map(owner => col.derive(owner)).getOrElse(col.derive(baseRef))
            }.toList

            val selWithRowItem =
              SqlSelect(
                context = context,
                withs = Nil,
                table = baseRef,
                cols = partitionCol :: distPartitionCol :: exposeCol ++ predCols,
                joins = (filterJoins ++ orderJoins).distinct,
                wheres = (pred1 ++ nonNullKeys ++ wheres).distinct,
                orders = Nil,
                offset = None,
                limit = None,
                distinct = Nil,
                oneToOne = true,
                predicate = true
              )

            val numberedName = syntheticName("_numbered")
            val subWithRowItem = SubqueryRef(context, numberedName, selWithRowItem, true)

            val partitionTerm = partitionCol.derive(subWithRowItem).toTerm.asInstanceOf[Term[Int]]
            val distPartitionTerm = distPartitionCol.derive(subWithRowItem).toTerm.asInstanceOf[Term[Int]]
            val windowPred = And(mkWindowPred(partitionTerm), LtEql(distPartitionTerm, Const(1)))

            val numberedPredCols = keyCols.map(_.derive(subWithRowItem))

            val predQuery =
              SqlSelect(
                context = context,
                withs = Nil,
                table = subWithRowItem,
                cols = numberedPredCols,
                joins = Nil,
                wheres = windowPred :: Nil,
                orders = Nil,
                offset = None,
                limit = None,
                distinct = Nil,
                oneToOne = true,
                predicate = true
              )

            val predName = syntheticName("_pred")
            val predSub = SubqueryRef(context, predName, predQuery, parentConstraints.nonEmpty)

            val on = keyCols.map(key => (key.derive(baseRef), key.derive(predSub)))
            val predJoin = SqlJoin(baseRef, predSub, on, true)

            val joinCols = cols.filterNot(col => baseRef.owns(col))

            val base = base0.copy(
              table = baseRef,
              cols = (base0.cols ++ joinCols).distinct,
              joins = predJoin :: base0.joins,
              wheres = Nil
            )

            Some(base)
          }
        } else {
          // No parent constraint so nothing to be gained from using window functions

          if((oneToOne && predIsOneToOne) || (offset0.isEmpty && limit0.isEmpty && filterJoins.isEmpty && orderJoins.isEmpty)) {
            // Case 1) one row is one object or query is simple enough to not require subqueries

            val (nonNullKeys, keyOrder) =
              offset0.orElse(limit0).map { _ =>
                val nonNullKeys0 = keyCols.map(col => IsNull(col.toTerm, false))
                val keyOrder0 = keyCols.diff(orderCols).map(col => OrderSelection(col.toTerm, true, true))
                (nonNullKeys0, keyOrder0)
              }.getOrElse((Nil, Nil))

            val pred1 = pred0.map(p => contextualiseWhereTerms(context, table, p))
            val oss0 = oss.map(os => contextualiseOrderTerms(context, table, os))
            val orders = oss0 ++ keyOrder

            val predQuery =
              SqlSelect(
                context = context,
                withs = withs,
                table = table,
                cols = cols,
                joins = (filterJoins ++ orderJoins ++ joins).distinct,
                wheres = (pred1 ++ nonNullKeys ++ wheres).distinct,
                orders = orders,
                offset = offset0,
                limit = limit0,
                distinct = distinct,
                oneToOne = true,
                predicate = true
              )

            Some(predQuery)
          } else if (orderBy.isEmpty || orderCols.take(keyCols.size).diff(keyCols).isEmpty) {
            // Case 2) No order, or key columns at the start of the order
            val base0 = subqueryToWithQuery
            val baseRef = base0.table

            val predCols = keyCols.map(_.derive(baseRef))
            val nonNullKeys = predCols.map(col => IsNull(col.toTerm, false))

            val pred1 = pred0.map(p => contextualiseWhereTerms(context, baseRef, p))
            val oss0 = oss.map(os => contextualiseOrderTerms(context, baseRef, os))
            val orders = oss0 ++ keyCols.diff(orderCols).map(col => OrderSelection(col.derive(baseRef).toTerm, true, true))

            val predQuery = SqlSelect(
              context = context,
              withs = Nil,
              table = baseRef,
              cols = predCols,
              joins = (filterJoins ++ orderJoins).distinct,
              wheres = (pred1 ++ nonNullKeys ++ wheres).distinct,
              orders = orders,
              offset = offset0,
              limit = limit0,
              distinct = predCols,
              oneToOne = true,
              predicate = true
            )

            val predName = syntheticName("_pred")
            val predSub = SubqueryRef(context, predName, predQuery, parentConstraints.nonEmpty)
            val on = keyCols.map(key => (key.derive(baseRef), key.derive(predSub)))
            val predJoin = SqlJoin(baseRef, predSub, on, true)

            val joinCols = cols.filterNot(col => table.owns(col))

            val base = base0.copy(
              table = baseRef,
              cols = (base0.cols ++ joinCols).distinct,
              joins = predJoin :: base0.joins,
              wheres = Nil
            )

            Some(base)
          } else {
            // Case 3) There is an order orthogonal to the key

            val base0 = subqueryToWithQuery
            val baseRef = base0.table

            val predCols = keyCols.map(_.derive(baseRef))
            val nonNullKeys = predCols.map(col => IsNull(col.toTerm, false))

            val pred1 = pred0.map(p => contextualiseWhereTerms(context, baseRef, p))

            val distOrders =
              keyCols.map(col => OrderSelection(col.derive(baseRef).toTerm, true, true))
              oss.filterNot(os => keyCols.contains(columnForSqlTerm(context, os.term).getOrElse(sys.error(s"No column for term ${os.term}"))))
            val distOrderCols = orderCols.diff(keyCols).map(_.derive(baseRef))

            val predQuery0 = SqlSelect(
              context = context,
              withs = Nil,
              table = baseRef,
              cols = predCols ++ distOrderCols,
              joins = (filterJoins ++ orderJoins).distinct,
              wheres = (pred1 ++ nonNullKeys ++ wheres).distinct,
              orders = distOrders,
              offset = None,
              limit = None,
              distinct = predCols,
              oneToOne = true,
              predicate = true
            )

            val predName0 = "dist"
            val predSub0 = SubqueryRef(context, predName0, predQuery0, parentConstraints.nonEmpty)

            val predCols0 = keyCols.map(_.derive(predSub0))
            val oss0 = oss.map(os => contextualiseOrderTerms(context, predSub0, os))
            val orders = oss0 ++ keyCols.diff(orderCols).map(col => OrderSelection(col.derive(predSub0).toTerm, true, true))

            val predQuery = SqlSelect(
              context = context,
              withs = Nil,
              table = predSub0,
              cols = predCols0,
              joins = Nil,
              wheres = Nil,
              orders = orders,
              offset = offset0,
              limit = limit0,
              distinct = Nil,
              oneToOne = true,
              predicate = true
            )

            val predName = syntheticName("_pred")
            val predSub = SubqueryRef(context, predName, predQuery, parentConstraints.nonEmpty)
            val on = keyCols.map(key => (key.derive(baseRef), key.derive(predSub)))
            val predJoin = SqlJoin(baseRef, predSub, on, true)

            val joinCols = cols.filterNot(col => table.owns(col))

            val base = base0.copy(
              table = baseRef,
              cols = (base0.cols ++ joinCols).distinct,
              joins = predJoin :: base0.joins,
              wheres = Nil
            )

            Some(base)
          }
        }
      }

      /** Yields an equivalent query encapsulating this query as a subquery */
      def toSubquery(name: String, lateral: Boolean): SqlSelect = {
        val ref = SubqueryRef(context, name, this, lateral)
        SqlSelect(context, Nil, ref, cols.map(_.derive(ref)), Nil, Nil, Nil, None, None, Nil, oneToOne, predicate)
      }

      /** Yields an equivalent query encapsulating this query as a common table expression */
      def toWithQuery(name: String, refName: Option[String]): SqlSelect = {
        val with0 = WithRef(context, name, this)
        val ref = TableExpr.DerivedTableRef(context, refName, with0, true)
        SqlSelect(context, with0 :: Nil, ref, cols.map(_.derive(ref)), Nil, Nil, Nil, None, None, Nil, oneToOne, predicate)
      }

      /** If the from clause of this query is a subquery, convert it to a
       *  common table expression
       */
      def subqueryToWithQuery: SqlSelect = {
        table match {
          case SubqueryRef(_, name, sq, _) =>
            val with0 = WithRef(context, name+"_base", sq)
            val ref = TableExpr.DerivedTableRef(context, Some(name), with0, true)
            copy(withs = with0 :: withs, table = ref)
          case _ =>
            this
        }
      }

      /** Render this `SqlSelect` as a `Fragment` */
      def toFragment: Aliased[Fragment] = {
        for {
          _       <- Aliased.pushOwner(this)
          withs0  <- if (withs.isEmpty) Aliased.pure(Fragments.empty)
                     else withs.traverse(_.toDefFragment).map(fwiths => Fragments.const("WITH ") |+| fwiths.intercalate(Fragments.const(",")))
          table0  <- table.toDefFragment
          cols0   <- cols.traverse(col => col.toDefFragment(needsCollation(col)))
          dcols   <- distinct.traverse(col => col.toRefFragment(needsCollation(col)))
          dist    =  if (dcols.isEmpty) Fragments.empty else Fragments.const("DISTINCT ON ") |+| Fragments.parentheses(dcols.intercalate(Fragments.const(", ")))
          joins0  <- joins.traverse(_.toFragment).map(_.combineAll)
          select  =  Fragments.const(s"SELECT ") |+| dist |+| cols0.intercalate(Fragments.const(", "))
          from    =  Fragments.const(" FROM ") |+| table0
          where   <- wheresToFragment(context, wheres)
          orderBy <- ordersToFragment(orders)
          off     =  offset.map(o => Fragments.const(s" OFFSET $o")).getOrElse(Fragments.empty)
          lim     =  limit.map(l => Fragments.const(s" LIMIT $l")).getOrElse(Fragments.empty)
          _       <- Aliased.popOwner
        } yield
          withs0 |+| select |+| from |+| joins0 |+| where |+| orderBy |+| off |+| lim
      }
    }

    object SqlSelect {
      def apply(
        context:   Context,                  // the GraphQL context of the query
        withs:     List[WithRef],            // the common table expressions
        table:     TableExpr,                // the table/subquery
        cols:      List[SqlColumn],          // the requested columns
        joins:     List[SqlJoin],            // joins for predicates/subobjects
        wheres:    List[Predicate],
        orders:    List[OrderSelection[_]],
        offset:    Option[Int],
        limit:     Option[Int],
        distinct:  List[SqlColumn],          // columns this query is DISTINCT on
        oneToOne:  Boolean,                  // does one row represent exactly one complete GraphQL value
        predicate: Boolean                   // does this SqlSelect represent a predicate
      ): SqlSelect =
        new SqlSelect(
          context,
          withs,
          table,
          cols.sortBy(col => (col.owner.context.resultPath, col.column)),
          joins,
          wheres,
          orders,
          offset,
          limit,
          distinct,
          oneToOne,
          predicate
        )
    }

    /** Representation of a UNION ALL of SQL SELECTs */
    case class SqlUnion(elems: List[SqlSelect]) extends SqlQuery {
      assert(elems.sizeCompare(2) >= 0)

      def isUnion: Boolean = true

      /** Does one row of this query correspond to exactly one complete GraphQL value */
      def oneToOne: Boolean = elems.forall(_.oneToOne)

      /** The context for this query */
      val context = elems.head.context
      assert(elems.tail.forall(_.context == context))

      /** This query in the given context */
      def withContext(context: Context, extraCols: List[SqlColumn], extraJoins: List[SqlJoin]): SqlUnion = {
        SqlUnion(elems = elems.map(_.withContext(context, extraCols, extraJoins)))
      }

      def owns(col: SqlColumn): Boolean = cols.exists(_ == col) || elems.exists(_.owns(col))
      def contains(other: ColumnOwner): Boolean = isSameOwner(other) || elems.exists(_.contains(other))
      def directlyOwns(col: SqlColumn): Boolean = owns(col)
      def findNamedOwner(col: SqlColumn): Option[TableExpr] = None

      override def isSameOwner(other: ColumnOwner): Boolean = other eq this

      /** The union of the columns of the underlying SELECTs in the order they will be
       *  yielded as the columns of this UNION
       */
      lazy val cols: List[SqlColumn] = elems.flatMap(_.cols).distinct

      def codecs: List[(Boolean, Codec)] =
        cols.map(col => (true, col.codec))

      def subst(from: TableExpr, to: TableExpr): SqlUnion =
        SqlUnion(elems.map(_.subst(from, to)))

      def toSubquery(name: String, lateral: Boolean): SqlSelect = {
        val sub = SubqueryRef(context, name, this, lateral)
        SqlSelect(context, Nil, sub, cols.map(_.derive(sub)), Nil, Nil, Nil, None, None, Nil, false, false)
      }

      def toWithQuery(name: String, refName: Option[String]): SqlSelect = {
        val with0 = WithRef(context, name, this)
        val ref = TableExpr.DerivedTableRef(context, refName, with0, true)
        SqlSelect(context, with0 :: Nil, ref, cols.map(_.derive(ref)), Nil, Nil, Nil, None, None, Nil, false, false)
      }

      /** Nest this query as a subobject in the enclosing `parentContext` */
      def nest(
        parentContext: Context,
        extraCols: List[SqlColumn],
        oneToOne: Boolean,
        lateral: Boolean
      ): SqlQuery = {
        val elems0 =
          elems.foldLeft(List.empty[SqlSelect]) { case (elems0, elem) =>
            val elem0 = elem.nest(parentContext, extraCols, oneToOne, lateral)
            elem0 :: elems0
          }

        SqlUnion(elems0)
      }

      /** Add WHERE, ORDER BY, OFFSET, and LIMIT to this query */
      def addFilterOrderByOffsetLimit(
        filter: Option[(Predicate, List[SqlJoin])],
        orderBy: Option[(List[OrderSelection[_]], List[SqlJoin])],
        offset: Option[Int],
        limit: Option[Int],
        predIsOneToOne: Boolean,
        parentConstraints: List[(SqlColumn, SqlColumn)]
      ): Option[SqlQuery] = {
        val withFilter =
          (filter, limit) match {
            case (None, None) => this
            // Push filters, offset and limit through into the branches of a union ...
            case _ =>
              val branchLimit = limit.map(_+offset.getOrElse(0))
              val branchOrderBy = limit.flatMap(_ => orderBy)
              val elems0 =
                elems.foldLeft(List.empty[SqlSelect]) { case (elems0, elem) =>
                  elem.addFilterOrderByOffsetLimit(filter, branchOrderBy, None, branchLimit, predIsOneToOne, parentConstraints) match {
                    case Some(elem0) => elem0 :: elems0
                    case None => elem :: elems0
                  }
                }
              SqlUnion(elems0)
          }

        (orderBy, offset, limit) match {
          case (None, None, None) => Some(withFilter)
          case _ =>
            val table: TableRef =
              parentTableForType(context).
                getOrElse(sys.error(s"No parent table for type ${context.tpe}"))
            val sel = withFilter.toSubquery(table.name, parentConstraints.nonEmpty)
            sel.addFilterOrderByOffsetLimit(None, orderBy, offset, limit, predIsOneToOne, parentConstraints)
         }
       }

      /** Render this `SqlUnion` as a `Fragment` */
      def toFragment: Aliased[Fragment] = {
        val alignedElems = {
          elems.map { elem =>
            val cols0 = cols.map { col =>
              elem.cols.find(_ == col).getOrElse(SqlColumn.NullColumn(elem, col))
            }
            elem.copy(cols = cols0)
          }
        }

        for {
          frags <- alignedElems.traverse(_.toFragment)
        } yield {
          frags.reduce((x, y) => Fragments.parentheses(x) |+| Fragments.const(" UNION ALL ") |+| Fragments.parentheses(y))
        }
      }
    }

    object SqlUnion {
      def apply(elems: List[SqlSelect]): SqlUnion =
        new SqlUnion(elems.distinct)
    }

    /** Representation of an SQL join */
    case class SqlJoin(
      parent:  TableExpr,                    // name of parent table
      child:   TableExpr,                    // child table/subquery
      on:      List[(SqlColumn, SqlColumn)], // join conditions
      inner:   Boolean
    ) extends ColumnOwner {
      assert(on.nonEmpty)
      assert(on.forall { case (p, c) => parent.owns(p) && child.owns(c) })

      def context = child.context
      def owns(col: SqlColumn): Boolean = child.owns(col)
      def contains(other: ColumnOwner): Boolean = isSameOwner(other) || child.contains(other)
      def directlyOwns(col: SqlColumn): Boolean =
        child match {
          case tr: TableRef => tr.directlyOwns(col)
          case dr: DerivedTableRef => dr.directlyOwns(col)
          case _ => false
        }

      def findNamedOwner(col: SqlColumn): Option[TableExpr] = child.findNamedOwner(col)

      override def isSameOwner(other: ColumnOwner): Boolean = other eq this

      /** Replace references to `from` with `to` */
      def subst(from: TableExpr, to: TableExpr): SqlJoin = {
        val newParent = if(parent.isSameOwner(from)) to else parent
        val newChild =
          if(!child.isSameOwner(from)) child
          else {
            (child, to) match {
              case (sr: SubqueryRef, to: TableRef) => sr.copy(context = to.context, name = to.name)
              case _ => to
            }
          }
        val newOn = on.map { case (p, c) => (p.subst(from, to), c.subst(from, to)) }
        copy(parent = newParent, child = newChild, on = newOn)
      }

      /** Return the columns of `table` referred to by the parent side of the conditions of this join */
      def colsOf(other: ColumnOwner): List[SqlColumn] =
        if (other.isSameOwner(parent)) on.map(_._1)
        else Nil

      /** Does this `SqlJoin` represent a predicate? */
      def isPredicate: Boolean =
        child match {
          case SubqueryRef(_, _, sq: SqlSelect, _) => sq.predicate
          case _ => false
        }

      /** Render this `SqlJoin` as a `Fragment` */
      def toFragment: Aliased[Fragment] = {
        val kind = if (inner) "INNER" else "LEFT"
        val join = s"$kind JOIN"

        val onFrag =
          on.traverse {
            case (p, c) =>
              for {
                fc <- c.toRefFragment(false)
                fp <- p.toRefFragment(false)
              } yield {
                fc |+| Fragments.const(" = ") |+| fp
              }
          }.map { fons => Fragments.const(s" ON ") |+| Fragments.and(fons: _*) }

        for {
          fchild <- child.toDefFragment
          fon    <- onFrag
        } yield Fragments.const(s" $join ") |+| fchild |+| fon
      }
    }

    object SqlJoin {
      /** Check that the given joins are correctly ordered relative to the given parent table */
      def checkOrdering(parent: TableExpr, joins: List[SqlJoin]): Boolean = {
        @tailrec
        def loop(joins: List[SqlJoin], seen: List[TableExpr]): Boolean = {
          joins match {
            case Nil => true
            case hd :: tl =>
              seen.exists(_.isSameOwner(hd.parent)) && loop(tl, hd.child :: seen)
          }
        }
        loop(joins, parent :: Nil)
      }
    }
  }

  /** Represents the mapping of a GraphQL query to an SQL query */
  final class MappedQuery(
    query: SqlQuery
  ) {
    /** Execute this query in `F` */
    def fetch: F[Table] = {
      for {
        rows <- self.fetch(fragment, query.codecs)
      } yield Table(query.cols, rows)
    }

    /** The query rendered as a `Fragment` with all table and column aliases applied */
    lazy val fragment: Fragment = query.toFragment.runA(AliasState.empty).value

    /** Return the value of the field `fieldName` in `context` from `table` */
    def selectAtomicField(context: Context, fieldName: String, table: Table): Result[Any] =
      columnForAtomicField(context, fieldName) match {
        case Some(col) =>
          table.select(col).filterNot(_ == FailedJoin).distinct match {
            case Nil => FailedJoin.rightIor
            case value :: Nil => value.rightIor
            case multi =>
              val obj = context.tpe.dealias
              if (obj.variantField(fieldName) || obj.field(fieldName).map(_.isNullable).getOrElse(true))
                // if the field is a non-schema attribute we won't be able to discover whether
                // or not it's nullable. Instead we assume that the presense of a None implies
                // nullability, hence stripping out Nones is justified.
                multi.filterNot(_ == None) match {
                  case Nil => None.rightIor
                  case value :: Nil => value.rightIor
                  case multi =>
                    mkErrorResult(s"Expected single value for field '$fieldName' of type $obj at ${context.path}, found $multi")
                }
              else
                mkErrorResult(s"Expected single value for field '$fieldName' of type $obj at ${context.path}, found $multi")
          }
        case None =>
          val obj = context.tpe.dealias
          mkErrorResult(s"Expected mapping for field '$fieldName' of type $obj")
      }

    /** Does `table` contain subobjects of the type of the `narrowedContext` type */
    def narrowsTo(narrowedContext: Context, table: Table): Boolean =
      keyColumnsForType(narrowedContext) match {
        case Nil => false
        case cols =>
          table.definesAll(cols)
      }

    /** Yield a `Table` containing only subojects of the `narrowedContext` type */
    def narrow(narrowedContext: Context, table: Table): Table = {
      keyColumnsForType(narrowedContext) match {
        case Nil => table
        case cols =>
          table.filterDefined(cols)
      }
    }

    /** Yield a list of `Tables` one for each of the subobjects of the context type
     *  contained in `table`.
     */
    def group(context: Context, table: Table): Iterator[Table] =
      table.group(keyColumnsForType(context))
  }

  object MappedQuery {
    /** Compile the given GraphQL query to SQL in the given `Context` */
    def apply(q: Query, context: Context): Option[MappedQuery] = {
      def loop(q: Query, context: Context, parentConstraints: List[(SqlColumn, SqlColumn)], exposeJoins: Boolean): Option[SqlQuery] = {
        lazy val parentTable: TableRef =
          parentTableForType(context).getOrElse(sys.error(s"No parent table for type ${context.tpe}"))

        def group(queries: List[Query]): Option[SqlQuery] = {
          val nodes =
            queries.foldLeft(List.empty[SqlQuery]) {
              case (nodes, q) =>
                loop(q, context, parentConstraints, exposeJoins) match {
                  case Some(n) =>
                    n :: nodes
                  case None =>
                    nodes
                }
            }

          SqlQuery.combineAll(nodes)
        }

        /* Compute the set of parent constraints to be inherited by the query for the value for `fieldName` */
        def parentConstraintsFromJoins(parentContext: Context, fieldName: String, resultName: String): List[(SqlColumn, SqlColumn)] = {
          fieldMapping(parentContext, fieldName) match {
            case Some(SqlObject(_, Nil)) => Nil

            case Some(SqlObject(_, join :: Nil)) =>
              val childContext =
                parentContext.forField(fieldName, resultName).
                  getOrElse(sys.error(s"No field '$fieldName' of type ${context.tpe}"))

              List((SqlColumn.TableColumn(parentContext, join.parent), SqlColumn.TableColumn(childContext, join.child)))

            case Some(SqlObject(_, joins)) =>
              val init =
                joins.init.map { join =>
                  val parentTable = TableRef(parentContext, join.parent.table)
                  val parentCol = SqlColumn.TableColumn(parentTable, join.parent)
                  val childTable = TableRef(parentContext, join.child.table)
                  val childCol = SqlColumn.TableColumn(childTable, join.child)
                  (parentCol, childCol)
                }

              val last = {
                val childContext =
                  parentContext.forField(fieldName, resultName).
                    getOrElse(sys.error(s"No field '$fieldName' of type ${context.tpe}"))

                val lastJoin = joins.last
                val parentTable = TableRef(parentContext, lastJoin.parent.table)
                val parentCol = SqlColumn.TableColumn(parentTable, lastJoin.parent)
                val childTable = TableRef(childContext, lastJoin.child.table)
                val childCol = SqlColumn.TableColumn(childTable, lastJoin.child)
                (parentCol, childCol)
              }

              init ++ (last :: Nil)

            case _ => Nil
          }
        }

        def parentConstraintsToSqlJoins(parentConstraints: List[(SqlColumn, SqlColumn)]): List[SqlJoin] =
          if (parentConstraints.sizeCompare(1) <= 0) Nil
          else {
            val (p, c) = parentConstraints.last
            (p.owner, c.owner) match {
              case (pt: TableExpr, _) =>
                // Intentionally reversed
                val fc = c.in(parentTable)
                List(SqlJoin(parentTable, pt, List((fc, p)), true))
              case _ => sys.error(s"Unnamed owner(s) for parent constraint ($p, $c)")
            }
          }

        q match {
          // Leaf or Json element: no subobjects
          case PossiblyRenamedSelect(Select(fieldName, _, child), _) if child == Empty || isJsonb(context, fieldName) =>
            val constraintCol = if(exposeJoins) parentConstraints.lastOption.map(_._2).toList else Nil
            val cols = columnsForLeaf(context, fieldName)
            val extraCols = keyColumnsForType(context) ++ constraintCol
            val extraJoins = parentConstraintsToSqlJoins(parentConstraints)
            Some(SqlSelect(context, Nil, parentTable, (cols ++ extraCols).distinct, extraJoins, Nil, Nil, None, None, Nil, true, false))

          // Non-leaf non-Json element: compile subobject queries
          case PossiblyRenamedSelect(Select(fieldName, _, child), resultName) =>
            val fieldContext = context.forField(fieldName, resultName).getOrElse(sys.error(s"No field '$fieldName' of type ${context.tpe}"))
            val constraintCol = if(exposeJoins) parentConstraints.lastOption.map(_._2).toList else Nil
            val extraCols = keyColumnsForType(context) ++ constraintCol
            val parentConstraints0 = parentConstraintsFromJoins(context, fieldName, resultName)
            val extraJoins = parentConstraintsToSqlJoins(parentConstraints)
            loop(child, fieldContext, parentConstraints0, false).map { sq =>
              val sq0 = sq.nest(context, extraCols, sq.oneToOne && isSingular(context, fieldName, child), parentConstraints0.nonEmpty)
              sq0.withContext(sq0.context, Nil, extraJoins)
            }

          case TypeCase(default, narrows) =>
            def isSimple(query: Query): Boolean = {
              def loop(query: Query): Boolean =
                query match {
                  case Empty => true
                  case PossiblyRenamedSelect(Select(_, _, Empty), _) => true
                  case Group(children) => children.forall(loop)
                  case _ => false
                }
              loop(query)
            }

            val subtpes = narrows.map(_.subtpe)
            val supertpe = context.tpe.underlying
            assert(supertpe.underlying.isInterface || supertpe.underlying.isUnion)
            subtpes.map(subtpe => assert(subtpe <:< supertpe))

            val discriminator = discriminatorForType(context)
            val narrowPredicates = subtpes.map { subtpe =>
              (subtpe, discriminator.flatMap(_.discriminator.narrowPredicate(subtpe)))
            }

            val exhaustive = schema.exhaustive(supertpe, subtpes)
            val exclusive = default == Empty
            val allSimple = narrows.forall(narrow => isSimple(narrow.child))

            val extraCols = (keyColumnsForType(context) ++ discriminatorColumnsForType(context)).distinct

            if (allSimple) {
              val dquery = loop(default, context, parentConstraints, exposeJoins).map(_.withContext(context, extraCols, Nil)).toList
              val nqueries =
                narrows.traverse { narrow =>
                  val subtpe0 = narrow.subtpe.withModifiersOf(context.tpe)
                  loop(narrow.child, context.asType(subtpe0), parentConstraints, exposeJoins).map(_.withContext(context, extraCols, Nil))
                }.getOrElse(Nil)
              SqlQuery.combineAll(dquery ++ nqueries)
            } else {
              val dquery =
                if(exhaustive) Nil
                else if (exclusive) {
                  val defaultPredicate =
                    And.combineAll(
                      narrowPredicates.collect {
                        case (_, Some(pred)) => Not(SqlQuery.contextualiseWhereTerms(context, parentTable, pred))
                      }
                    )

                  List(SqlSelect(context, Nil, parentTable, extraCols, Nil, defaultPredicate :: Nil, Nil, None, None, Nil, true, false))
                } else {
                  val defaultPredicate =
                    And.combineAll(
                      narrowPredicates.collect {
                        case (_, Some(pred)) => Not(pred)
                      }
                    )
                  loop(Filter(defaultPredicate, default), context, parentConstraints, exposeJoins).map(_.withContext(context, extraCols, Nil)).toList
                }

              val nqueries =
                narrows.traverse { narrow =>
                  val subtpe0 = narrow.subtpe.withModifiersOf(context.tpe)
                  val child = Group(List(default, narrow.child))
                  loop(child, context.asType(subtpe0), parentConstraints, exposeJoins).map(_.withContext(context, extraCols, Nil))
                }.getOrElse(Nil)

              val allSels = (dquery ++ nqueries).flatMap(_.asSelects).distinct

              allSels match {
                case Nil => None
                case sel :: Nil => Some(sel)
                case sels => Some(SqlUnion(sels))
              }
            }

          case n: Narrow =>
            sys.error(s"Narrow not matched by extractor: $n")

          case Group(queries) => group(queries)

          case GroupList(queries) => group(queries)

          case Wrap(_, child) =>
            loop(child, context, parentConstraints, exposeJoins)

          case Count(countName, child) =>
            def childContext(q: Query): Option[Context] =
              q match {
                case PossiblyRenamedSelect(Select(fieldName, _, _), resultName) =>
                  context.forField(fieldName, resultName)
                case FilterOrderByOffsetLimit(_, _, _, _, child) =>
                  childContext(child)
                case _ => None
              }

            val fieldContext = childContext(child).getOrElse(sys.error(s"No context for count of ${child}"))
            val countCol = columnForAtomicField(context, countName).getOrElse(sys.error(s"Count column $countName not defined"))

            loop(child, context, parentConstraints, exposeJoins).flatMap {
              case sq: SqlSelect =>
                sq.joins match {
                  case hd :: tl =>
                    val keyCols = keyColumnsForType(fieldContext)
                    val parentCols0 = hd.colsOf(parentTable)
                    val wheres = hd.on.map { case (p, c) => Eql(c.toTerm, p.toTerm) }
                    val ssq = sq.copy(table = hd.child, cols = SqlColumn.CountColumn(countCol.in(hd.child), keyCols.map(_.in(hd.child))) :: Nil, joins = tl, wheres = wheres)
                    val ssqCol = SqlColumn.SubqueryColumn(countCol, ssq)
                    Some(SqlSelect(context, Nil, parentTable, (ssqCol :: parentCols0).distinct, Nil, Nil, Nil, None, None, Nil, true, false))
                  case _ => None
                }
              case _ => None
            }

          case Rename(_, child) =>
            loop(child, context, parentConstraints, exposeJoins)

          case Unique(child) =>
            loop(child, context.asType(context.tpe.nonNull.list), parentConstraints, exposeJoins).map {
              case node => node.withContext(context, Nil, Nil)
            }

          case Filter(False, _) => None
          case Filter(True, child) => loop(child, context, parentConstraints, exposeJoins)

          case FilterOrderByOffsetLimit(pred, oss, offset, lim, child) =>
            val filterPaths = pred.map(SqlQuery.wherePaths).getOrElse(Nil).distinct
            val orderPaths = oss.map(_.map(_.term).collect { case path: Path => path.path }).getOrElse(Nil).distinct
            val filterOrderPaths = (filterPaths ++ orderPaths).distinct

            if (pred.map(p => !isSqlTerm(context, p)).getOrElse(false)) {
              // If the predicate must be evaluated programatically then there's
              // nothing we can do here, so just collect up all the columns/joins
              // needed for the filter/order and loop
              val expandedChildQuery = mergeQueries(child :: mkPathQuery(filterOrderPaths))
              loop(expandedChildQuery, context, parentConstraints, exposeJoins)
            } else {
              val filterQuery = mergeQueries(mkPathQuery(filterPaths))
              val orderQuery = mergeQueries(mkPathQuery(orderPaths))

              def extractJoins(sq: SqlQuery): List[SqlJoin] =
                sq match {
                  case sq: SqlSelect => sq.joins
                  case su: SqlUnion => su.elems.flatMap(_.joins)
                }

              val filter =
                for {
                  pred0 <- pred
                  sq    <- loop(filterQuery, context, parentConstraints, exposeJoins)
                } yield ((pred0, extractJoins(sq)), sq.oneToOne)

              val orderBy =
                for {
                  oss0 <- oss
                  sq   <- loop(orderQuery, context, parentConstraints, exposeJoins)
                } yield ((oss0, extractJoins(sq)), sq.oneToOne)

              val predIsOneToOne =
                filter.map(_._2).getOrElse(true) &&
                orderBy.map(_._2).getOrElse(true)

              // Ordering will be repeated programmatically, so include the columns/
              // joins for ordering in the child query
              val expandedChildQuery =
                if (orderPaths.isEmpty) child
                else mergeQueries(child :: mkPathQuery(orderPaths))

              for {
                expandedChild <- loop(expandedChildQuery, context, parentConstraints, true)
                res           <- expandedChild.addFilterOrderByOffsetLimit(filter.map(_._1), orderBy.map(_._1), offset, lim, predIsOneToOne, parentConstraints)
              } yield res
            }

          case fool@(_: Filter | _: OrderBy | _: Offset | _: Limit) =>
            sys.error(s"Filter/OrderBy/Offset/Limit not matched by extractor: $fool")

          case _: Introspect =>
            val extraCols = (keyColumnsForType(context) ++ discriminatorColumnsForType(context)).distinct
            Some(SqlSelect(context, Nil, parentTable, extraCols.distinct, Nil, Nil, Nil, None, None, Nil, true, false))

          case Environment(_, child) =>
            loop(child, context, parentConstraints, exposeJoins)

          case Empty | Skipped | Query.Component(_, _, _) | (_: Defer) | (_: UntypedNarrow) | (_: Skip) | (_: Select) =>
            None
        }
      }

      loop(q, context, Nil, false).map(new MappedQuery(_))
    }
  }

  /** Representation of an SQL query result */
  sealed trait Table {
    def numRows: Int
    def numCols: Int

    /** Yield the values of the given column */
    def select(col: SqlColumn): List[Any]
    /** A copy of this `Table` containing only the rows for which all the given columns are defined */
    def filterDefined(cols: List[SqlColumn]): Table
    /** True if all the given columns are defined, false otherwise */
    def definesAll(cols: List[SqlColumn]): Boolean
    /** Group this `Table` by the values of the given columns */
    def group(cols: List[SqlColumn]): Iterator[Table]

    def isEmpty: Boolean = false
  }

  object Table {
    def apply(cols: List[SqlColumn], rows: Vector[Array[Any]]): Table =
      apply(cols.zipWithIndex.toMap, rows)

    def apply(index: SqlColumn => Int, rows: Vector[Array[Any]]): Table = {
      if (rows.sizeCompare(1) == 0) OneRowTable(index, rows.head)
      else if (rows.isEmpty) EmptyTable
      else MultiRowTable(index, rows)
    }

    /** Specialized representation of an empty table */
    case object EmptyTable extends Table {
      def numRows: Int = 0
      def numCols: Int = 0

      def select(col: SqlColumn): List[Any] = Nil
      def filterDefined(cols: List[SqlColumn]): Table = this
      def definesAll(cols: List[SqlColumn]): Boolean = false
      def group(cols: List[SqlColumn]): Iterator[Table] = Iterator.empty[Table]

      override def isEmpty = true
    }

    /** Specialized representation of a table with exactly one row */
    case class OneRowTable(index: SqlColumn => Int, row: Array[Any]) extends Table {
      def numRows: Int = 1
      def numCols = row.size

      def select(col: SqlColumn): List[Any] = {
        val c = index(col)
        row(c) match {
          case FailedJoin => Nil
          case other => other :: Nil
        }
      }

      def filterDefined(cols: List[SqlColumn]): Table =
        if(definesAll(cols)) this else EmptyTable

      def definesAll(cols: List[SqlColumn]): Boolean = {
        val cs = cols.map(index)
        cs.forall(c => row(c) != FailedJoin)
      }

      def group(cols: List[SqlColumn]): Iterator[Table] = {
        cols match {
          case Nil => Iterator.single(this)
          case cols =>
            if (definesAll(cols)) Iterator.single(this)
            else Iterator.empty[Table]
        }
      }
    }

    case class MultiRowTable(index: SqlColumn => Int, rows: Vector[Array[Any]]) extends Table {
      def numRows = rows.size
      def numCols = rows.headOption.map(_.size).getOrElse(0)

      def select(col: SqlColumn): List[Any] = {
        val c = index(col)
        rows.iterator.map(r => r(c)).filterNot(_ == FailedJoin).distinct.toList
      }

      def filterDefined(cols: List[SqlColumn]): Table = {
        val cs = cols.map(index)
        Table(index, rows.filter(r => cs.forall(c => r(c) != FailedJoin)))
      }

      def definesAll(cols: List[SqlColumn]): Boolean = {
        val cs = cols.map(index)
        rows.exists(r => cs.forall(c => r(c) != FailedJoin))
      }

      def group(cols: List[SqlColumn]): Iterator[Table] = {
        cols match {
          case Nil => rows.iterator.map(r => OneRowTable(index, r))
          case cols =>
            val cs = cols.map(index)

            val discrim: Array[Any] => Any =
              cs match {
                case c :: Nil => row => row(c)
                case cs => row => cs.map(c => row(c))
              }


            val nonNull = rows.filter(r => cs.forall(c => r(c) != FailedJoin))
            val grouped = nonNull.groupBy(discrim)
            grouped.iterator.map { case (_, rows) => Table(index, rows) }
        }
      }
    }
  }

  /** Cursor positioned at a GraphQL result leaf */
  case class LeafCursor(context: Context, focus: Any, mapped: MappedQuery, parent: Option[Cursor], env: Env) extends Cursor {
    assert(focus != FailedJoin)

    def withEnv(env0: Env): Cursor = copy(env = env.add(env0))

    def mkChild(context: Context = context, focus: Any = focus): LeafCursor =
      LeafCursor(context, focus, mapped, Some(this), Env.empty)

    def isLeaf: Boolean = tpe.isLeaf

    def asLeaf: Result[Json] =
      encoderForLeaf(tpe).map(enc => enc(focus).rightIor).getOrElse(
        focus match {
          case s: String => Json.fromString(s).rightIor
          case i: Int => Json.fromInt(i).rightIor
          case l: Long => Json.fromLong(l).rightIor
          case f: Float => Json.fromFloat(f) match {
              case Some(j) => j.rightIor
              case None => mkErrorResult(s"Unrepresentable float %d")
            }
          case d: Double => Json.fromDouble(d) match {
              case Some(j) => j.rightIor
              case None => mkErrorResult(s"Unrepresentable double %d")
            }
          case d: BigDecimal => Json.fromBigDecimal(d).rightIor
          case b: Boolean => Json.fromBoolean(b).rightIor

          // This means we are looking at a column with no value because it's the result of a failed
          // outer join. This is an implementation error.
          case FailedJoin => sys.error("Unhandled failed join.")

          case other =>
            mkErrorResult(s"Not a leaf: $other")
        }
      )

    def isList: Boolean =
      tpe match {
        case ListType(_) => true
        case _ => false
      }

    def asList: Result[List[Cursor]] = (tpe, focus) match {
      case (ListType(tpe), it: List[_]) => it.map(f => mkChild(context.asType(tpe), focus = f)).rightIor
      case _ => mkErrorResult(s"Expected List type, found $tpe")
    }

    def isNullable: Boolean =
      tpe match {
        case NullableType(_) => true
        case _ => false
      }

    def asNullable: Result[Option[Cursor]] =
      (tpe, focus) match {
        case (NullableType(_), None) => None.rightIor
        case (NullableType(tpe), Some(v)) => Some(mkChild(context.asType(tpe), focus = v)).rightIor
        case _ => mkErrorResult(s"Not nullable at ${context.path}")
      }

    def narrowsTo(subtpe: TypeRef): Boolean = false
    def narrow(subtpe: TypeRef): Result[Cursor] =
      mkErrorResult(s"Cannot narrow $tpe to $subtpe")

    def hasField(fieldName: String): Boolean = false
    def field(fieldName: String, resultName: Option[String]): Result[Cursor] =
      mkErrorResult(s"Cannot select field '$fieldName' from leaf type $tpe")
  }

  /** Cursor positioned at a GraphQL result non-leaf */
  case class SqlCursor(context: Context, focus: Any, mapped: MappedQuery, parent: Option[Cursor], env: Env) extends Cursor {
    def withEnv(env0: Env): Cursor = copy(env = env.add(env0))

    def mkChild(context: Context = context, focus: Any = focus): SqlCursor =
      SqlCursor(context, focus, mapped, Some(this), Env.empty)

    def asTable: Result[Table] = focus match {
      case table: Table => table.rightIor
      case _ => mkErrorResult(s"Not a table")
    }

    def isLeaf: Boolean = false
    def asLeaf: Result[Json] =
      mkErrorResult(s"Not a leaf: $tpe")

    def isList: Boolean =
      tpe.isList

    def asList: Result[List[Cursor]] =
      tpe.item.map(_.dealias).map(itemTpe =>
        asTable.map { table =>
          val itemContext = context.asType(itemTpe)
          mapped.group(itemContext, table).map(table => mkChild(itemContext, focus = table)).to(List)
        }
      ).getOrElse(mkErrorResult(s"Not a list: $tpe"))

    def isNullable: Boolean =
      tpe.isNullable

    def asNullable: Result[Option[Cursor]] =
      (tpe, focus) match {
        case (NullableType(_), table: Table) if table.isEmpty => None.rightIor
        case (NullableType(tpe), _) => Some(mkChild(context.asType(tpe))).rightIor // non-nullable column as nullable schema type (ok)
        case _ => mkErrorResult(s"Not nullable at ${context.path}")
      }

    def narrowsTo(subtpe: TypeRef): Boolean = {
      val ctpe =
        discriminatorForType(context) match {
          case Some(disc) => disc.discriminator.discriminate(this).getOrElse(tpe)
          case _ => tpe
        }
      if (ctpe =:= tpe)
        asTable.map(table => mapped.narrowsTo(context.asType(subtpe), table)).right.getOrElse(false)
      else ctpe <:< subtpe
    }

    def narrow(subtpe: TypeRef): Result[Cursor] = {
      if (narrowsTo(subtpe)) {
        val narrowedContext = context.asType(subtpe)
        asTable.map { table =>
          mkChild(context = narrowedContext, focus = mapped.narrow(narrowedContext, table))
        }
      } else mkErrorResult(s"Cannot narrow $tpe to $subtpe")
    }

    def hasField(fieldName: String): Boolean = tpe.hasField(fieldName)

    def field(fieldName: String, resultName: Option[String]): Result[Cursor] = {
      tpe.underlyingObject.map { obj =>
        val fieldContext = context.forFieldOrAttribute(fieldName, resultName)
        val fieldTpe = fieldContext.tpe

        fieldMappingType(context, fieldName).toRightIor(mkOneError(s"No field mapping for field '$fieldName' of type $obj")).flatMap {
          case CursorFieldMapping(f) =>
            f(this).map(res => LeafCursor(fieldContext, res, mapped, Some(this), Env.empty))

          case CursorFieldJsonMapping(f) =>
            f(this).map(res => CirceCursor(fieldContext, focus = res, parent = Some(this), env = Env.empty))

          case JsonFieldMapping =>
            asTable.flatMap { table =>
              def mkCirceCursor(f: Json): Result[Cursor] =
                CirceCursor(fieldContext, focus = f, parent = Some(this), env = Env.empty).rightIor
              mapped.selectAtomicField(context, fieldName, table).flatMap(_ match {
                case Some(j: Json) if fieldTpe.isNullable => mkCirceCursor(j)
                case None => mkCirceCursor(Json.Null)
                case j: Json if !fieldTpe.isNullable => mkCirceCursor(j)
                case other =>
                  mkErrorResult(s"$fieldTpe: expected jsonb value found ${other.getClass}: $other")
              })
            }

          case LeafFieldMapping =>
            asTable.flatMap(table =>
              mapped.selectAtomicField(context, fieldName, table).map { leaf =>
                val leafFocus = leaf match {
                  case Some(f) if tpe.variantField(fieldName) && !fieldTpe.isNullable => f
                  case other => other
                }
                LeafCursor(fieldContext, leafFocus, mapped, Some(this), Env.empty)
              }
            )

          case ObjectFieldMapping =>
            asTable.map { table =>
              val focussed = mapped.narrow(fieldContext, table)
              mkChild(context = fieldContext, focus = focussed)
            }
        }
      }.getOrElse(mkErrorResult(s"Type $tpe has no field '$fieldName'"))
    }
  }
}
