package edu.uci.qa.browserdriver;

import java.net.MalformedURLException;
import java.util.HashMap;
import java.util.Map;

import org.openqa.selenium.Capabilities;
import org.openqa.selenium.WebDriver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import edu.uci.qa.browserdriver.utils.BrowserType;
import edu.uci.qa.browserdriver.drivers.*;

/**
 * The class abstracted by all BrowserDrivers.
 * Utilize this classes static methods in order to generically
 * create a new BrowserDriver.
 */
public abstract class BrowserDriver extends BrowserDriverBase implements WebDriver {

    private static final BrowserType defaultType = BrowserType.CHROME;

    private static Map<BrowserType, Class<? extends BrowserDriver>> driverIDToClassMap = new HashMap<BrowserType, Class<? extends BrowserDriver>>();
    private static Map<Class<? extends BrowserDriver>, BrowserType> driverClassToIDMap = new HashMap<Class<? extends BrowserDriver>, BrowserType>();

    private static void preInstantiation(BrowserType browser) {
        Logger logger = LoggerFactory.getLogger(BrowserDriver.class);
        logger.info(" Running BrowserDriver pre-instantiation...");
        /* Any code here will run before any browser is initialized. */
        
        // TODO: allow executable run or write exec copy in ~/.m2/repository/webdriver.
        System.setProperty("wdm.targetPath", "./webdrivers");
    }

    private void postInstantiation() {
        logger().info(" Running BrowserDriver post-instantiation...");
        if (this.getType() != BrowserType.HEADLESS)
            this.manage().window().maximize();
        
        /* Any code here will run after any browser is initialized */
        
        logger().info(" BrowserDriver successfully instantiated!");
    }
    
    
    /**
     * Creates a new BrowserDriver as the specified BrowseName Enviroment Variable
     * or the default BrowserType (Chrome).
     * 
     * @return new BrowserDriver.
     */
    public static BrowserDriver createBrowser() {
        BrowserType browserType = BrowserType.get(System.getenv("browsername"));
        if (browserType == BrowserType.NONE)
            browserType = defaultType;
        return createBrowser(browserType);
    }

    /**
     * Creates a new BrowserDriver as the specified BrowserType.
     * 
     * @param browser BrowserType that the returned BrowserDriver should be.
     * @return new BrowserDriver given the BrowserType.
     */
    public static BrowserDriver createBrowser(BrowserType browser) {
        BrowserDriver output = null;
        preInstantiation(browser);
        try {
            Class<? extends BrowserDriver> bClass = (Class<? extends BrowserDriver>) driverIDToClassMap.get(browser);
            output = (BrowserDriver) bClass.newInstance();
        } catch (IllegalAccessException | InstantiationException e) { 
            e.printStackTrace();
            return null;
        }
        output.logger().info(" Successfully instantiated " + output.getCapabilities().getBrowserName() + " browser!");
        output.postInstantiation();
        return output;
    }
    
    /**
     * Creates a new BrowserDriver via Remote connections (through a selenium grid/Sauce labs).
     *  
     * @param capabilities Selenium capabilities defining what RemoteBrowserDriver we want to run.
     * @return new RemoteBrowserDriver given the capabilities.
     * @throws MalformedURLException
     */
    public static BrowserDriver createRemoteBrowser(Capabilities capabilities) throws MalformedURLException {
        BrowserDriver output = null;
        preInstantiation(BrowserType.NONE);
        output = new SauceBrowserDriver(capabilities);
        output.postInstantiation();
        return output;
    }

    /**
     * Returns the BrowserType set by the BrowserDriver class instancing. 
     * @return The BrowserType of the browser.
     */
    public final BrowserType getType() {
        return driverClassToIDMap.get(getClass()) == null ? BrowserType.NONE : driverClassToIDMap.get(getClass());
    }

    private static void addToClassMapping(BrowserType type, Class<? extends BrowserDriver> driverClass) {
        if (driverIDToClassMap.containsKey(type))
            throw new IllegalArgumentException("Duplicated BrowserType: " + type);
        if (driverClassToIDMap.containsKey(driverClass))
            throw new IllegalArgumentException("Duplicate BrowserDriver class: " + driverClass);

        driverIDToClassMap.put(type, driverClass);
        driverClassToIDMap.put(driverClass, type);
    }

    static {
        addToClassMapping(BrowserType.CHROME, ChromeBrowserDriver.class);
        addToClassMapping(BrowserType.FIREFOX, FirefoxBrowserDriver.class);
        addToClassMapping(BrowserType.HEADLESS, HtmlBrowserDriver.class);
        addToClassMapping(BrowserType.PHANTOM, PhantomBrowserDriver.class);
        addToClassMapping(BrowserType.IE, InternetExplorerBrowserDriver.class);
        addToClassMapping(BrowserType.EDGE, EdgeBrowserDriver.class);
        addToClassMapping(BrowserType.OPERA, OperaBrowserDriver.class);
    }
}
