package edu.uci.qa.browserdriver;

import org.openqa.selenium.By;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.support.ui.ExpectedCondition;

import com.google.common.base.Function;

/**
 * Custom interface inherited by BrowserDriverBase in order to
 * easily document and manage all custom methods commonly shared by
 * all BrowserDrivers.
 */
public interface BrowserDriverInterface {
    
    /**
     * The default timeout time for the wait functions.
     * @return the default timeout in seconds.
     */
    long defaultTimeoutInSeconds();
    
    /**
     * Repeatedly applies this instance's input value to the given
     * function until one of the following occurs:
     * 
     * <ul>
     * <li>the function returns neither null nor false,</li>
     * <li>the function throws an un-ignored exception,</li>
     * <li>the timeout expires,</li>
     * <li>the current thread is interrupted</li>
     * </ul>
     * 
     * <p>
     * The wait timeout will be set to the {@link #defaultTimeoutInSeconds()}
     * 
     * @param <V> The function's expected return type.
     * @param condition the parameter to pass to the {@link ExpectedCondition}
     * @return The functions' value if the function returned something
     * different from null or false before the timeout expired.
     */
    <V> V waitUntil(Function<? super WebDriver, V> condition);    
    
    /** 
     * Repeatedly applies this instance's input value to the given
     * function until one of the following occurs:
     * 
     * <ul>
     * <li>the function returns neither null nor false,</li>
     * <li>the function throws an un-ignored exception,</li>
     * <li>the timeout expires,</li>
     * <li>the current thread is interrupted</li>
     * </ul>
     * 
     * @param <V> The function's expected return type.
     * @param condition the parameter to pass to the {@link ExpectedCondition}
     * @param timeout how long (in seconds) it waits for
     *        {@link ExpectedCondition} before expiring.
     * @return The functions' value if the function returned something
     * different from null or false before the timeout expired.
     */
    <V> V waitUntil(Function<? super WebDriver, V> condition, long timeout);
    
    /** 
     * Has the Browser sleep for the specified amount of time.
     * <p>
     * use {@link #waitUntil(ExpectedCondition condition) waitUntil} instead
     * since pausing threads is not recommended by Selenium.
     * @param timeout How long the browser waits in milliseconds.
     */
    void sleep(long timeout) throws InterruptedException;
    
    /** 
     * Executes JavaScript on the currently selected frame or window.
     * 
     * <p>
     * If the script has a return value (i.e. if the script contains a <code>return</code> statement),
     * then the following steps will be taken:
     *
     * <ul>
     * <li>For an HTML element, this method returns a WebElement</li>
     * <li>For a decimal, a Double is returned</li>
     * <li>For a non-decimal number, a Long is returned</li>
     * <li>For a boolean, a Boolean is returned</li>
     * <li>For all other cases, a String is returned.</li>
     * <li>For an array, return a List&lt;Object&gt; with each object following the rules above. We
     * support nested lists.</li>
     * <li>Unless the value is null or there is no return value, in which null is returned</li>
     * </ul>
     * 
     * <p>
     * Arguments must be a number, a boolean, a String, WebElement, or a List of any combination of
     * the above. An exception will be thrown if the arguments do not meet these criteria. 
     * 
     * @param script The JavaScript to execute
     * @param args The arguments of the script. May be empty
     * @return One of Boolean, Long, String, List or WebElement. Or null.
     */
    Object executeScript(String script, Object...args);

    /**
     * Waits for the clickability of the specified <code>locator</code> for
     * {@link #defaultTimeoutInSeconds()} at most, then attempts to click it.
     * 
     * @param locator The locating mechanism.
     */
    void click(By locator);
    
    /**
     * Waits for the clickability of the specified <code>locator</code> for
     * specified <code>timeout</code> at most in seconds, then attempts to click it.
     * 
     * @param locator The locating mechanism.
     * @param timeout how long (in seconds) it waits for locator visibility before expiring.
     */
    void click(By locator, long timeout);
    
    /**
     * Waits for the visibility of the specified <code>locator</code> for
     * {@link #defaultTimeoutInSeconds()} at most, then attempts to get the
     * visible (i.e. not hidden by CSS) innerText of this element,
     * including sub-elements, without any leading or trailing whitespace.
     * 
     * @param locator the locating mechanism.
     * @return The innerText of this element, if found, otherwise null.
     */
    String getText(By locator);
    
    /**
     * Waits for the visibility of the specified <code>locator</code> for
     * specified <code>timeout</code> at most in seconds, then attempts to get the
     * visible (i.e. not hidden by CSS) innerText of this element,
     * including sub-elements, without any leading or trailing whitespace.
     * 
     * @param locator the locating mechanism.
     * @param timeout how long (in seconds) it waits for locator visibility before expiring.
     * @return The innerText of this element, if found, otherwise null.
     */
    String getText(By locator, long timeout);
    
    /**
     * Use this method to simulate typing into an element, which may set its value.
     * Waits for the visibility of the specified <code>locator</code> for at most
     * {@link #defaultTimeoutInSeconds()}.
     * 
     * @param locator The locating mechanism.
     * @param keysToSend character sequence to send to the element
     */
    void sendKeys(By locator, String keysToSend);
    
    /**
     * Use this method to simulate typing into an element, which may set its value.
     * Waits for the visibility of the specified <code>locator</code> for at most
     * <code>timeout</code> in seconds.
     * 
     * @param locator The locating mechanism.
     * @param keysToSend character sequence to send to the element
     */
    void sendKeys(By locator, String keysToSend, long timeout);
    
    /**
     * Waits for the visibility of the specified <code>locator</code> for
     * {@link #defaultTimeoutInSeconds()} at most, then attempts to clear the
     * text entry value if this element is a text entry element. Has no effect
     * on other elements.
     * 
     * <p>
     * Text entry elements are INPUT and TEXTAREA elements.
     * Note that the events fired by this event may not be as you'd expect.
     * In particular, we don't fire any keyboard or mouse events.
     * If you want to ensure keyboard events are fired,
     * consider using something like {@link #sendKeys(By, String)} with the backspace key.
     * To ensure you get a change event, consider following with a call to
     * {@link #sendKeys(By, String)} with the tab key.
     * 
     * @param locator the locating mechanism.
     */
    void clear(By locator);
    
    /**
     * Waits for the visibility of the specified <code>locator</code> for at most
     * <code>timeout</code> in seconds at most, then attempts to clear the
     * text entry value if this element is a text entry element. Has no effect
     * on other elements.
     * 
     * <p>
     * Text entry elements are INPUT and TEXTAREA elements.
     * Note that the events fired by this event may not be as you'd expect.
     * In particular, we don't fire any keyboard or mouse events.
     * If you want to ensure keyboard events are fired,
     * consider using something like {@link #sendKeys(By, String)} with the backspace key.
     * To ensure you get a change event, consider following with a call to
     * {@link #sendKeys(By, String)} with the tab key.
     * 
     * @param locator the locating mechanism.
     */
    void clear(By locator, long timeout);
    
    
    /**
     * Gets the current browser's capabilities. 
     * @return The browser's capabilities.
     */
    Capabilities getCapabilities();
}
