package edu.uci.qa.browserdriver;

import org.openqa.selenium.By;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.JavascriptExecutor;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.openqa.selenium.support.ui.ExpectedConditions;
import org.openqa.selenium.support.ui.WebDriverWait;

import com.google.common.base.Function;

import edu.uci.qa.browserdriver.utils.WebDriverWrapper;

/**
 * Custom abstract class where all custom methods given by
 * BrowserDriverInterface are implemented
 */
public abstract class BrowserDriverBase extends WebDriverWrapper implements BrowserDriverInterface {
    public long defaultTimeoutInSeconds() {
        return 10l;
    }

    public <V> V waitUntil(Function<? super WebDriver, V> condition) {
        logger().info("Waiting for (at most) " + defaultTimeoutInSeconds() + " seconds.");
        return (V) (new WebDriverWait(webDriver(), defaultTimeoutInSeconds())).until(condition);
    }

    public <V> V waitUntil(Function<? super WebDriver, V> condition, long timeout) {
        logger().info("Waiting for (at most) " + timeout + " seconds.");
        return (V) (new WebDriverWait(webDriver(), timeout)).until(condition);
    }

    public Object executeScript(String script, Object... args) {
        logger().info("Executing JavaScript: " + script);
        return ((JavascriptExecutor) this.webDriver()).executeScript(script, args);
    }

    public Capabilities getCapabilities() {
        return ((RemoteWebDriver) this.webDriver()).getCapabilities();
    }
    
    public void click(By locator) {
        logger().info("Waiting for clickability of " + locator);
        waitUntil(ExpectedConditions.elementToBeClickable(locator)).click();
        logger().info(locator + " clicked!");
    }
    
    public void click(By locator, long timeout) {
        logger().info("Waiting for clickability of " + locator);
        waitUntil(ExpectedConditions.elementToBeClickable(locator), timeout).click();
        logger().info(locator + " clicked!");
    }
    
    public void clear(By locator) {
        logger().info("Waiting for visibility of " + locator);
        waitUntil(ExpectedConditions.visibilityOfElementLocated(locator)).clear();
        logger().info(locator + " cleared!");
    }
    
    public void clear(By locator, long timeout) {
        logger().info("Waiting for visibility of " + locator);
        waitUntil(ExpectedConditions.visibilityOfElementLocated(locator), timeout).clear();
        logger().info(locator + " cleared!");
    }
    
    public String getText(By locator) {
        logger().info("Waiting for visibility of " + locator);
        WebElement element = waitUntil(ExpectedConditions.visibilityOfElementLocated(locator));
        return element == null ? null : element.getText();
    }
    
    public String getText(By locator, long timeout) {
        logger().info("Waiting for visibility of " + locator);
        WebElement element = waitUntil(ExpectedConditions.visibilityOfElementLocated(locator), timeout);
        return element == null ? null : element.getText();
    }
    
    public void sendKeys(By locator, String keysToSend) {
        logger().info("Waiting for visibility of " + locator);
        waitUntil(ExpectedConditions.visibilityOfElementLocated(locator)).sendKeys(keysToSend);
        logger().info("Successfully sent the following character sequence: " + keysToSend);
    }
    
    public void sendKeys(By locator, String keysToSend, long timeout) {
        logger().info("Waiting for visibility of " + locator);
        waitUntil(ExpectedConditions.visibilityOfElementLocated(locator), timeout).sendKeys(keysToSend);
        logger().info("Successfully sent the following character sequence: " + keysToSend);
    }

    public void sleep(long millis) throws InterruptedException {
        logger().info("Sleeping for " + millis + " milliseconds.");
        Thread.sleep(millis);
    }
}
