package im.dart.boot.common.data;

import im.dart.boot.common.constant.Magic;
import im.dart.boot.common.utils.Checker;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.function.Consumer;

/**
 * @author Kevin.Xu
 */
@ApiModel("分页信息")
public class Page extends Base {
    /**
     * 排序
     */
    @ApiModel("排序")
    public static class Sort extends Base {
        /**
         * 顺序:true 倒序:false
         */
        @ApiModelProperty("顺序:true 倒序:false")
        private boolean asc;
        /**
         * 排序字段
         */
        @ApiModelProperty("排序字段")
        private String filed;

        private Sort(String filed, boolean asc) {
            this.asc = asc;
            this.filed = filed;
        }

        public boolean isAsc() {
            return asc;
        }

        public String getFiled() {
            return filed;
        }

        public static Sort of(String filed, boolean asc) {
            return new Sort(filed, asc);
        }
    }

    /**
     * 分页数据的父类
     */
    @ApiModel("分页数据的父类")
    public static class PageData extends Base {
        @ApiModelProperty("每页数据量")
        private Integer size = 10;
        @ApiModelProperty("当前页，从0开始")
        private Integer page = 0;
        @ApiModelProperty("排序")
        private LinkedHashSet<Sort> sorts;

        public Integer getSize() {
            if (Checker.isEmpty(size)) {
                return 10;
            }
            return size;
        }

        public void setSize(Integer size) {
            this.size = size;
        }

        public Integer getPage() {
            if (Checker.isEmpty(page)) {
                return 0;
            }
            return page;
        }

        public void setPage(Integer page) {
            this.page = page;
        }

        public LinkedHashSet<Sort> getSorts() {
            return sorts;
        }

        public void setSorts(LinkedHashSet<Sort> sorts) {
            this.sorts = sorts;
        }

        public void forEachSort(final Consumer<Sort> consumer) {
            if (Checker.isEmpty(sorts)) {
                return;
            }
            for (Sort sort : sorts) {
                if (Checker.isEmpty(sort.getFiled())) {
                    continue;
                }
                consumer.accept(sort);
            }
        }
    }

    /**
     * 分页数据的请求数据
     *
     * @param <T> TODO
     */
    @ApiModel("分页数据的请求数据")
    public static class Request<T extends Base> extends PageData {

        @ApiModelProperty("请求数据结构")
        private T param;
        @ApiModelProperty("开始时间")
        private Long start;
        @ApiModelProperty("结束时间")
        private Long end;

        public T getParam() {
            return param;
        }

        public void setParam(T param) {
            this.param = param;
        }

        public Long getStart() {
            return start;
        }

        public void setStart(Long start) {
            this.start = start;
        }

        public Long getEnd() {
            return end;
        }

        public void setEnd(Long end) {
            this.end = end;
        }
    }

    /**
     * 分页数据的结果数据
     *
     * @param <T> TODO
     */
    @ApiModel("分页数据的结果数据")
    public static class Result<T extends Base> extends PageData {

        @ApiModelProperty("总数据量")
        private Long total = Magic.L_ZERO;
        @ApiModelProperty("返回列表")
        private List<T> contents;

        public Result() {
        }

        public static <T extends Base> Result<T> of(List<T> contents, long total, Request<T> request) {
            return Result.of(contents, total, request.getPage(), request.getSize(), request.getSorts());
        }

        public static <T extends Base> Result<T> of(List<T> contents, long total, int page, int size, LinkedHashSet<Sort> sorts) {
            Result<T> result = new Result();
            result.setPage(page);
            result.setSize(size);
            result.setTotal(total);
            result.setSorts(sorts);
            result.setContents(contents);
            return result;
        }

        public Long getTotal() {
            return total;
        }

        public void setTotal(Long total) {
            this.total = total;
        }

        public List<T> getContents() {
            return contents;
        }

        public void setContents(List<T> contents) {
            this.contents = contents;
        }
    }
}
