package im.dart.boot.common.utils;

import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;

/**
 * 时间工具
 *
 * @author Kevin.Xu
 */
public class DateUtil {
    public static final long START_TIME_2022 = 1640966400000l;
    public static final String FMT_DEFAULT_STR = "yyyy-MM-dd HH:mm:ss";
    public static final DateTimeFormatter FMT_SIMPLE = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    public static final DateTimeFormatter FMT_DEFAULT = DateTimeFormatter.ofPattern(FMT_DEFAULT_STR);

    /**
     * 把毫秒数按指定格式格式化
     *
     * @param millisTime 毫秒数
     * @param fmt        格式
     * @return
     */
    public static String fmt(Long millisTime, String fmt) {
        if (millisTime == null || Checker.isEmpty(fmt)) {
            return null;
        }
        if (Checker.isEmpty(millisTime)) {
            return null;
        }
        long timeFmt = ((millisTime + "").length() == 10) ? millisTime * 1000 : millisTime;

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(fmt);
        Instant instant = Instant.ofEpochMilli(timeFmt);
        ZoneId zoneId = ZoneId.systemDefault();
        LocalDateTime time = LocalDateTime.ofInstant(instant, zoneId);
        return time.format(formatter);
    }

    /**
     * 按默认方式格式化毫秒数
     *
     * @param millisTime 毫秒数
     * @return yyyy-MM-dd HH:mm:ss
     */
    public static String fmt(Long millisTime) {
        return fmt(millisTime, FMT_DEFAULT_STR);
    }

    /**
     * 按指定格式获得当前时间
     *
     * @param fmt 格式
     * @return 格式化后的当前时间
     */
    public static String fmt(String fmt) {
        return LocalDateTime.now().format(DateTimeFormatter.ofPattern(fmt));
    }

    /**
     * @param millisTime 毫秒数
     * @return yyyyMMdd
     */
    public static Long fmtYYYYMMDD(Long millisTime) {
        String fmt = fmt(millisTime, "yyyyMMdd");
        if (fmt == null) {
            return null;
        }
        return Long.valueOf(fmt);
    }

    /**
     * 获得格式化后的当前时间
     *
     * @return 当前时间
     */
    public static String current() {
        return LocalDateTime.now().format(FMT_DEFAULT);
    }

    /**
     * @return yyyy-MM-dd
     */
    public static String simpleCurrent() {
        return LocalDateTime.now().format(FMT_SIMPLE);
    }

    /**
     * 获取当前时间
     *
     * @return 当前时间：毫秒
     */
    public static long currentTimeMillis() {
        return System.currentTimeMillis();
    }

    /**
     * 获取从2022-01-01 00:00:00开始的毫秒数
     *
     * @return 获取从2022-01-01 00:00:00开始的毫秒数
     */
    public static long start2022TimeMillis() {
        return System.currentTimeMillis() - START_TIME_2022;
    }

    /**
     * @return 当前时间：秒
     */
    public static long currentTimeSecond() {
        return (Long) (System.currentTimeMillis() / 1000);
    }

    /**
     * 今天的开始时间
     *
     * @return +8 的时间的毫秒数
     */
    public static long todayStart() {
        LocalDateTime time = LocalDateTime.of(LocalDate.now(), LocalTime.MIN);
        return time.toInstant(ZoneOffset.of("+8")).toEpochMilli();
    }

    /**
     * 本月的开始时间
     *
     * @return +8 的时间的毫秒数
     */
    public static long monthStart() {
        LocalDate date = LocalDate.now();
        LocalDateTime time = LocalDateTime.of(LocalDate.of(date.getYear(), date.getMonth(), 1), LocalTime.MIN);
        return time.toInstant(ZoneOffset.of("+8")).toEpochMilli();
    }

    /**
     * 今年开始时间
     *
     * @return +8 的时间的毫秒数
     */
    public static long yearStart() {
        LocalDateTime time = LocalDateTime.of(LocalDate.of(LocalDate.now().getYear(), 1, 1), LocalTime.MIN);
        return time.toInstant(ZoneOffset.of("+8")).toEpochMilli();
    }

    /**
     * 今天的结束时间
     *
     * @return +8 的时间的毫秒数
     */
    public static long todayEnd() {
        LocalDateTime time = LocalDateTime.of(LocalDate.now(), LocalTime.MAX);
        return time.toInstant(ZoneOffset.of("+8")).toEpochMilli();
    }

    /**
     * @return 当年（2022）
     */
    public static int currentYear() {
        return Integer.valueOf(fmt(System.currentTimeMillis(), "yyyy"));
    }

    /**
     * @return 当天 yyyyMMdd
     */
    public static int currentDay() {
        return Integer.valueOf(fmt(System.currentTimeMillis(), "yyyyMMdd"));
    }

    /**
     * 今天星期几
     *
     * @return
     */
    public static String todayWeek() {
        int week = Calendar.getInstance().get(Calendar.DAY_OF_WEEK);
        switch (week) {
            case 1:
                return "星期天";
            case 2:
                return "星期一";
            case 3:
                return "星期二";
            case 4:
                return "星期三";
            case 5:
                return "星期四";
            case 6:
                return "星期五";
            case 7:
                return "星期六";
            default:
                return "UNKNOW";
        }
    }

    public static String fmtRemainingNanoTime(long nano) {
        if (nano <= 0) {
            return "";
        }
        long nt = nano % 1000000;
        String mt = fmtTimeMillis(nano / 1000000);
        return (Checker.isEmpty(mt) ? "" : mt + " ") + nt + "(ns)";
    }

    public static String fmtTimeMillis(long timeMillis) {
        if (timeMillis <= 0) {
            return "";
        }

        long ms = timeMillis % 1000;
        String time = ms + "(ms)";

        long s = timeMillis / 1000;
        if (s <= 0) {
            return time;
        }

        long cs = s % 60;
        time = cs + "(s) " + time;

        long m = s / 60;
        if (m <= 0) {
            return time;
        }
        long cm = m % 60;
        time = cm + "(m) " + time;

        long h = m / 60;
        if (h <= 0) {
            return time;
        }
        return h + "(h) " + time;
    }

}