package im.dart.boot.common.utils;

import im.dart.boot.common.data.Base;
import im.dart.boot.common.data.Entry;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

/**
 * @author Kevin.Xu
 */
public class ReflectUtil {

    static Logger logger = Logger.getLogger(ReflectUtil.class.getName());

    private static final ConcurrentHashMap<Class, ArrayList<Field>> CLASS_FIELD_CACHE = new ConcurrentHashMap<>();

    static {
        CLASS_FIELD_CACHE.clear();
    }

    public static List<Field> fetchFields(Object obj) {
        if (Checker.isEmpty(obj)) {
            return null;
        }
        return ReflectUtil.fetchFields(obj.getClass());
    }

    public static List<Field> fetchFields(Class clazz) {
        if (clazz.equals(Object.class) || clazz.equals(Base.class)) {
            return new ArrayList<>();
        }

        ArrayList<Field> fields = CLASS_FIELD_CACHE.get(clazz);
        if (Checker.isNotEmpty(fields)) {
            return fields;
        }

        fields = new ArrayList<>();
        Field[] fieldList = clazz.getDeclaredFields();
        for (Field f : fieldList) {
            if (!Objects.equals("serialVersionUID", f.getName())) {
                fields.add(f);
            }
        }
        fields.addAll(fetchFields(clazz.getSuperclass()));
        CLASS_FIELD_CACHE.put(clazz, fields);
        return fields;
    }

    public static Field getFieldByName(Class clazz, String name) {
        if (Checker.isEmpty(name)) {
            return null;
        }
        String nl = name.toLowerCase();
        List<Field> fields = fetchFields(clazz);
        for (Field field : fields) {
            if (Objects.equals(nl, field.getName().toLowerCase())) {
                return field;
            }
        }
        return null;
    }

    public static boolean setFieldValue(Object obj, String field, Object value) {
        if (Checker.isEmpty(obj) || Checker.isEmpty(field)) {
            return false;
        }
        Field f = getFieldByName(obj.getClass(), field);
        if (Checker.isEmpty(f)) {
            return false;
        }
        return setFieldValue(obj, f, value);
    }

    public static boolean setFieldValue(Object obj, Field field, Object value) {
        if (Checker.isEmpty(obj) || Checker.isEmpty(field)) {
            return false;
        }

        Object newValue = Checker.isEmpty(value) ? null : value;
        return Runner.safeRun(() -> {
            field.setAccessible(true);
            field.set(obj, newValue);
            return true;
        });
    }

    public static List<Entry<String, Object>> fields(Object obj) {
        if (obj == null) {
            return new ArrayList<>();
        }
        List<Field> fields = fetchFields(obj.getClass());
        return fields.stream().map(tmp -> {
            return fetch(obj, tmp);
        }).filter(tmp -> {
            return Objects.nonNull(tmp);
        }).collect(Collectors.toList());
    }

    public static Map<String, Object> toMap(Object obj) {
        return toMap(obj, null);
    }

    public static Map<String, Object> toMap(Object obj, String... ignoreFields) {
        if (obj == null) {
            return new HashMap<>();
        }
        Set<String> ifs = new HashSet<>();
        if (Checker.isNotEmpty(ignoreFields)) {
            for (String iField : ignoreFields) {
                ifs.add(iField);
            }
        }
        List<Field> fields = fetchFields(obj.getClass());
        Map<String, Object> map = new HashMap<>();
        for (Field f : fields) {
            String name = f.getName();
            if (ifs.contains(name)) {
                continue;
            }

            Object v = fetchValue(obj, f);
            if (Checker.isNotEmpty(v)) {
                map.put(f.getName(), v);
            }
        }
        return map;
    }

    public static List<String> fetchNullField(Object obj) {
        if (obj == null) {
            return new ArrayList<>();
        }

        List<String> nullFields = new ArrayList<>();
        List<Field> fields = fetchFields(obj.getClass());
        for (Field field : fields) {
            Entry<String, Object> data = fetch(obj, field);
            if (Checker.isEmpty(data)) {
                nullFields.add(field.getName());
            }
        }
        return nullFields;
    }

    public static <T> T newObj(Class<T> clazz) {
        if (clazz == null) {
            return null;
        }

        return Runner.safeRun(() -> {
            Constructor<T> constructor = clazz.getDeclaredConstructor();
            return constructor.newInstance();
        });
    }

    public static Entry<String, Object> fetch(Object obj, Field field) {
        if (obj == null || field == null) {
            return null;
        }

        try {
            field.setAccessible(true);
            Object value = field.get(obj);
            if (!Checker.isEmptyString(value)) {
                return new Entry(field.getName(), value);
            }
        } catch (Exception e) {
            logger.log(Level.SEVERE, e.getMessage(), e);
        }
        return null;
    }

    public static Object fetchValue(Object obj, Field field) {
        if (obj == null || field == null) {
            return null;
        }

        try {
            field.setAccessible(true);
            return field.get(obj);
        } catch (Exception e) {
            logger.log(Level.SEVERE, e.getMessage(), e);
        }
        return null;
    }

    /**
     * 获取对象类的范型
     *
     * @param clazz 对象类 Service&lt;T&gt;
     * @param <T>   范型 T
     * @return 范型类型 Class_T
     */
    public static <T> Class<T> getGenericClass(Class clazz) {
        return getGenericClass(clazz, 0);
    }

    /**
     * 获取对象类的范型
     *
     * @param clazz 对象类 Service&lt;T&gt;
     * @param idx   需要获取的范型下标 0 开始
     * @param <T>   范型 T
     * @return 范型类型 Class_T
     */
    public static <T> Class<T> getGenericClass(Class clazz, int idx) {
        ParameterizedType type = (ParameterizedType) clazz.getGenericSuperclass();
        return (Class<T>) type.getActualTypeArguments()[idx];
    }

    public static Map<String, Object> analyzeStructure(Class clazz) {
        Map<String, Object> structure = new HashMap<>();
        List<Field> fields = ReflectUtil.fetchFields(clazz);
        for (Field field : fields) {
            Class clas = field.getType();
            if (Checker.isBaseType(clas)) {
                structure.put(field.getName(), clas.getSimpleName());
            } else {
                structure.put(field.getName(), analyzeStructure(clas));
            }
        }
        return structure;
    }
}
