package im.dart.boot.common.utils;

import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * 运行器
 *
 * @author Kevin.Xu
 */
public class Runner {

    static Logger logger = Logger.getLogger(Runner.class.getName());

    public static void sleep(long millis) {
        try {
            Thread.sleep(millis);
        } catch (Exception e) {
            logger.log(Level.SEVERE, e.getMessage(), e);
        }
    }

    public static void foreach(int count, ForWithException foreach) {
        for (int i = 0; i < count; i++) {
            try {
                foreach.apply(i);
            } catch (Exception e) {
                logger.log(Level.SEVERE, e.getMessage(), e);
            }
        }
    }

    public static void newThread(Runnable runnable) {
        new Thread(runnable).start();
    }

    public static void safeThread(ConsumerWithException consumer) {
        Thread safeThread = new Thread(() -> {
            try {
                consumer.apply();
            } catch (Exception e) {
                logger.log(Level.SEVERE, e.getMessage(), e);
            }
        });
        safeThread.setName("SafeThread");
        safeThread.setDaemon(true);
        safeThread.start();
    }

    public static void safeThreadWhile(ConsumerWithException consumer) {
        Thread safeThreadWhile = new Thread(() -> {
            while (true) {
                try {
                    consumer.apply();
                } catch (Exception e) {
                    logger.log(Level.SEVERE, e.getMessage(), e);
                }
            }
        });
        safeThreadWhile.setName("SafeThreadWhile");
        safeThreadWhile.setDaemon(true);
        safeThreadWhile.start();
    }


    public static <T> void safeRun(final Consumer<T> consumer, final T data) {
        try {
            consumer.accept(data);
        } catch (Exception e) {
            logger.log(Level.SEVERE, e.getMessage(), e);
        }
    }

    /**
     * 安全运行一个不需要返回值的函数
     *
     * @param consumer 函数
     */
    public static void safeRun(final ConsumerWithException consumer) {
        try {
            consumer.apply();
        } catch (Exception e) {
            logger.log(Level.SEVERE, e.getMessage(), e);
        }
    }

    /**
     * 安全运行一个带返回值的函数
     *
     * @param consumer 函数
     * @param <T>      返回值类型
     * @return 结果
     */
    public static <T> T safeRun(final FunctionWithException<T> consumer) {
        try {
            return consumer.apply();
        } catch (Exception e) {
            logger.log(Level.SEVERE, e.getMessage(), e);
        }
        return null;
    }

    /**
     * 安全运行一个需要返回值的函数,如果出错或者函数返回值为空,则返回默认值
     *
     * @param consumer     需要运行的函数
     * @param defaultValue 默认值
     * @param <T>          返回类型
     * @return 结果
     */
    public static <T> T safeRun(final FunctionWithException<T> consumer, final T defaultValue) {
        T value = null;
        try {
            value = consumer.apply();
        } catch (Exception e) {
            logger.log(Level.SEVERE, e.getMessage(), e);
        }
        return Checker.isEmpty(value) ? defaultValue : value;
    }

    public interface ConsumerWithException {
        /**
         * 使用
         *
         * @throws Exception
         */
        void apply() throws Exception;
    }

    public interface ForWithException {
        /**
         * 使用
         *
         * @throws Exception
         */
        void apply(int idx) throws Exception;
    }

    public interface FunctionWithException<T> {
        /**
         * 使用
         *
         * @return
         * @throws Exception
         */
        T apply() throws Exception;
    }

    public interface FunctionWithAssert {
        /**
         * 断言检查
         *
         * @return
         */
        boolean check();
    }
}
