package im.dart.boot.common.utils;

import im.dart.boot.common.constant.Charsets;

import java.io.CharArrayWriter;
import java.nio.charset.Charset;
import java.util.BitSet;
import java.util.Objects;

/**
 * <p>描述：URL编解码工具</p>
 *
 * <p>创建时间：2021-11-11 11:28</p>
 * <p>更新时间：暂无</p>
 *
 * @author Kevin.Xu
 * @version 1.0
 */
public class UrlCoder {

    static final BitSet DONT_NEED_ENCODING;
    static final int CASE_DIFF = ('a' - 'A');

    static {
        DONT_NEED_ENCODING = new BitSet(256);
        int i;
        for (i = 'a'; i <= 'z'; i++) {
            DONT_NEED_ENCODING.set(i);
        }
        for (i = 'A'; i <= 'Z'; i++) {
            DONT_NEED_ENCODING.set(i);
        }
        for (i = '0'; i <= '9'; i++) {
            DONT_NEED_ENCODING.set(i);
        }

        DONT_NEED_ENCODING.set(' ');
        DONT_NEED_ENCODING.set('-');
        DONT_NEED_ENCODING.set('_');
        DONT_NEED_ENCODING.set('.');
        DONT_NEED_ENCODING.set('*');
    }

    public static String encode(String source) {
        return encode(source, Charsets.UTF_8);
    }

    public static String encode(String source, Charset charset) {
        Objects.requireNonNull(source, "source");
        Objects.requireNonNull(charset, "Charset");

        boolean needToChange = false;
        StringBuilder out = new StringBuilder(source.length());
        CharArrayWriter charArrayWriter = new CharArrayWriter();

        for (int i = 0; i < source.length(); ) {
            int c = source.charAt(i);
            if (DONT_NEED_ENCODING.get(c)) {
//               原始代码
//                if (c == ' ') {
//                    c = '+';
//                    needToChange = true;
//                }
//                out.append((char) c);

                if (c == ' ') {
                    needToChange = true;
                    out.append("%20");
                } else {
                    out.append((char) c);
                }
                i++;
            } else {
                do {
                    charArrayWriter.write(c);
                    if (c >= 0xD800 && c <= 0xDBFF) {
                        if ((i + 1) < source.length()) {
                            int d = source.charAt(i + 1);
                            if (d >= 0xDC00 && d <= 0xDFFF) {
                                charArrayWriter.write(d);
                                i++;
                            }
                        }
                    }
                    i++;
                } while (i < source.length() && !DONT_NEED_ENCODING.get((c = source.charAt(i))));

                charArrayWriter.flush();
                String str = new String(charArrayWriter.toCharArray());
                byte[] ba = str.getBytes(charset);
                for (int j = 0; j < ba.length; j++) {
                    out.append('%');
                    char ch = Character.forDigit((ba[j] >> 4) & 0xF, 16);
                    if (Character.isLetter(ch)) {
                        ch -= CASE_DIFF;
                    }
                    out.append(ch);
                    ch = Character.forDigit(ba[j] & 0xF, 16);
                    if (Character.isLetter(ch)) {
                        ch -= CASE_DIFF;
                    }
                    out.append(ch);
                }
                charArrayWriter.reset();
                needToChange = true;
            }
        }
        return (needToChange ? out.toString() : source);
    }

    public static String decode(String source) {
        return decode(source, Charsets.UTF_8);
    }

    public static String decode(String source, Charset charset) {
        Objects.requireNonNull(source, "source");
        Objects.requireNonNull(charset, "Charset");


        boolean needToChange = false;
        int numChars = source.length();
        StringBuilder sb = new StringBuilder(numChars > 500 ? numChars / 2 : numChars);
        int i = 0;

        char c;
        byte[] bytes = null;
        while (i < numChars) {
            c = source.charAt(i);
            switch (c) {
                case '+':
                    sb.append(' ');
                    i++;
                    needToChange = true;
                    break;
                case '%':
                    if ((i + 2) < numChars) {     // 新逻辑
                        String tmp = source.substring(i + 1, i + 3);
                        if (tmp == "20") {
                            sb.append(" ");
                            i += 2;
                            break;
                        }
                    }

                    try {
                        if (bytes == null)
                            bytes = new byte[(numChars - i) / 3];
                        int pos = 0;

                        while (((i + 2) < numChars) && (c == '%')) {
                            int v = Integer.parseInt(source.substring(i + 1, i + 3), 16);
                            if (v < 0)
                                throw new IllegalArgumentException("URLDecoder: Illegal hex characters in escape (%) pattern - negative value");
                            bytes[pos++] = (byte) v;
                            i += 3;
                            if (i < numChars)
                                c = source.charAt(i);
                        }
                        if ((i < numChars) && (c == '%'))
                            throw new IllegalArgumentException("URLDecoder: Incomplete trailing escape (%) pattern");

                        sb.append(new String(bytes, 0, pos, charset));
                    } catch (NumberFormatException e) {
                        throw new IllegalArgumentException("URLDecoder: Illegal hex characters in escape (%) pattern - " + e.getMessage());
                    }
                    needToChange = true;
                    break;
                default:
                    sb.append(c);
                    i++;
                    break;
            }
        }
        return (needToChange ? sb.toString() : source);
    }
}
