package yamlesque

import scala.collection.mutable
import scala.language.implicitConversions

sealed trait Value extends geny.Writable {
  override def httpContentType = Some("application/yaml")

  /**
    * Returns the `String` value of this [[Value]], fails if it is not
    * a [[yamlesque.Str]]
    */
  def str = this match{
    case Str(value) => value
    case _ => throw Value.InvalidData(this, "Expected yamlesque.Str")
  }

  /**
    * Returns an Optional `String` value of this [[Value]] in case this [[Value]] is a 'String'.
    */
  def strOpt = this match{
    case Str(value) => Some(value)
    case _ => None
  }

  /**
    * Returns the key/value map of this [[Value]], fails if it is not
    * a [[yamlesque.Obj]]
    */
  def obj = this match{
    case Obj(value) => value
    case _ => throw Value.InvalidData(this, "Expected yamlesque.Obj")
  }
  /**
    * Returns an Optional key/value map of this [[Value]] in case this [[Value]] is a 'Obj'.
    */
  def objOpt = this match{
    case Obj(value) => Some(value)
    case _ => None
  }
  /**
    * Returns the elements of this [[Value]], fails if it is not
    * a [[yamlesque.Arr]]
    */
  def arr = this match{
    case Arr(value) => value
    case _ => throw Value.InvalidData(this, "Expected yamlesque.Arr")
  }
  /**
    * Returns The optional elements of this [[Value]] in case this [[Value]] is a 'Arr'.
    */
  def arrOpt = this match{
    case Arr(value) => Some(value)
    case _ => None
  }

  def transform[T](f: Visitor[T]): T = Value.transform(this, f)

  def writeBytesTo(out: java.io.OutputStream): Unit = {
    val renderer = new CompactPrinter(out)
    transform(renderer)
  }

  def render(codec: String = "utf-8"): String = {
    val stream = new java.io.ByteArrayOutputStream
    writeBytesTo(stream)
    new String(stream.toByteArray(), codec)
  }

}

object Value {

  def transform[T](value: Value, f: Visitor[T]): T = {
    val ctx = new Ctx {
      val pos = Position("", 0, 0)
      val line = ""
    }
    value match {
      case Null() => f.visitEmpty(ctx)
      case Str(s) => f.visitString(ctx, s)
      case Arr(items) =>
        val arrVisitor = f.visitArray(ctx)
        for ((item, idx) <- items.zipWithIndex) {
          arrVisitor.visitIndex(ctx, idx)
          val child = transform(item, arrVisitor.subVisitor())
          arrVisitor.visitValue(ctx, child)
        }
        arrVisitor.visitEnd()
      case Obj(items) =>
        val objVisitor = f.visitObject(ctx)
        for ((key, value) <- items) {
          objVisitor.visitKey(ctx, key)
          val child = transform(value, objVisitor.subVisitor())
          objVisitor.visitValue(ctx, child)
        }
        objVisitor.visitEnd()
    }
  }

  /**
    * Thrown when uPickle tries to convert a JSON blob into a given data
    * structure but fails because part the blob is invalid
    *
    * @param data The section of the JSON blob that uPickle tried to convert.
    *             This could be the entire blob, or it could be some subtree.
    * @param msg Human-readable text saying what went wrong
    */
  case class InvalidData(data: Value, msg: String)
    extends Exception(s"$msg (data: $data)")
}

case class Obj(values: mutable.LinkedHashMap[String, Value]) extends Value
object Obj{
  implicit def from(items: IterableOnce[(String, Value)]): Obj = {
    Obj(mutable.LinkedHashMap(items.iterator.toSeq:_*))
  }
  // Weird telescoped version of `apply(items: (String, Value)*)`, to avoid
  // type inference issues due to overloading the existing `apply` method
  // generated by the case class itself
  // https://github.com/lihaoyi/upickle/issues/230
  def apply[V](item: (String, V),
                        items: (String, Value)*)(implicit conv: V => Value): Obj = {
    val map = new mutable.LinkedHashMap[String, Value]()
    map.put(item._1, conv(item._2))
    for (i <- items) map.put(i._1, i._2)
    Obj(map)
  }

  def apply(): Obj = Obj(new mutable.LinkedHashMap[String, Value]())
}

case class Arr(values: mutable.ArrayBuffer[Value]) extends Value
object Arr{
  implicit def from[T](items: IterableOnce[T])(implicit conv: T => Value): Arr = {
    val buf = new mutable.ArrayBuffer[Value]()
    items.iterator.foreach{ item =>
      buf += (conv(item): Value)
    }
    Arr(buf)
  }

  def apply(items: Value*): Arr = {
    val buf = new mutable.ArrayBuffer[Value](items.length)
    items.foreach{ item =>
      buf += item
    }
    Arr(buf)
  }
}
case class Str(value: String) extends Value
case class Null() extends Value
