package io.datalbry.config.processor.kotlin

import com.google.devtools.ksp.processing.*
import com.google.devtools.ksp.symbol.KSAnnotated
import io.datalbry.config.api.ConfigSchema
import io.datalbry.config.processor.kotlin.extension.preconfiguredJackson
import io.datalbry.config.processor.kotlin.finder.PropertyDescriptionFinder
import io.datalbry.config.processor.kotlin.finder.PropertyTypeFinder
import javax.annotation.processing.SupportedSourceVersion
import javax.lang.model.SourceVersion

/**
 * [ConfigSchemaProcessor] is a composite annotation processor,
 * capable of deriving a [ConfigSchema] from [io.datalbry.config.api.annotation.ConfigType]
 * and [io.datalbry.config.api.annotation.ConfigSchema] annotated classes.
 *
 * The schema file will be written into the resources,
 * while the exact directory can be configured (defaults to [DEFAULT_SCHEMA_DIR]).
 * The filename can also be configured independently (defaults to [DEFAULT_SCHEMA_FILE]).
 *
 * @see io.datalbry.config.api.annotation.ConfigType for complex type definitions
 * @see io.datalbry.config.api.annotation.ConfigSchema for config property definitions
 *
 * @author timo gruen - 2021-04-07
 */
@SupportedSourceVersion(SourceVersion.RELEASE_8)
class ConfigSchemaProcessor(
    options: Map<String, String>,
    private val codeGenerator: CodeGenerator,
    private val logger: KSPLogger
): SymbolProcessor {

    private val jackson = preconfiguredJackson()
    private val schemaDir = options["directory"] ?: DEFAULT_SCHEMA_DIR
    private val schemaFile = options["filename"] ?: DEFAULT_SCHEMA_FILE

    override fun process(resolver: Resolver): List<KSAnnotated> {
        val propertyFinder = PropertyDescriptionFinder(logger, resolver)
        val typeFinder = PropertyTypeFinder(logger, resolver)

        val properties = propertyFinder.find()
        val types = typeFinder.find()

        val schema = ConfigSchema(properties, types)
        writeSchema(schema)
        return emptyList()
    }

    private fun writeSchema(schema: ConfigSchema) {
        if (schema.properties.isNotEmpty() || schema.types.isNotEmpty()) {
            val schemaFile = codeGenerator.createNewFile(
                Dependencies.ALL_FILES,
                schemaDir,
                schemaFile.split(".")[0],
                "json"
            )
            jackson.writeValue(schemaFile, schema)
        }
    }

    override fun finish() = Unit

    companion object {
        const val DEFAULT_SCHEMA_DIR = "META-INF/datalbry"
        const val DEFAULT_SCHEMA_FILE = "schema-config"
    }
}
