package io.datalbry.config.processor.kotlin.finder

import com.google.devtools.ksp.processing.KSPLogger
import com.google.devtools.ksp.processing.Resolver
import com.google.devtools.ksp.symbol.KSClassDeclaration
import com.google.devtools.ksp.symbol.KSPropertyDeclaration
import io.datalbry.config.api.PropertyDescription
import io.datalbry.config.api.annotation.ConfigSchema
import io.datalbry.config.processor.kotlin.mapper.KSAnnotationToPropertyDescriptorMapper
import io.datalbry.config.api.annotation.PropertyDescription as PropertyAnnotation

/**
 * [PropertyDescriptionFinder] is a [Finder] implementation to fetch [PropertyDescription]
 * from all [ConfigSchema] annotated symbols.
 *
 * @param logger to use for logging
 * @param resolver to resolve the symbols with
 *
 * @author timo gruen - 2021-04-07
 */
class PropertyDescriptionFinder(
    private val logger: KSPLogger,
    private val resolver: Resolver
): Finder<PropertyDescription> {

    private val mapper = KSAnnotationToPropertyDescriptorMapper()

    override fun find(): Set<PropertyDescription> {
        val annotatedClasses = resolver.getSymbolsWithAnnotation(ConfigSchema::class.java.canonicalName)
        return annotatedClasses
            .asSequence()
            .filterIsInstance<KSClassDeclaration>()
            .onEach { logger.logging("Start processing", it) }
            .map { it.getAllProperties().filter(this::isAnnotatedProperty) }
            .map { it.map(mapper::map) }
            .flatten()
            .flatten()
            .toSet()
    }

    private fun isAnnotatedProperty(p: KSPropertyDeclaration) = p.annotations.any {
        val annotationName = it.annotationType.resolve().declaration.qualifiedName?.asString()
        val propertyAnnotationName = PropertyAnnotation::class.java.canonicalName
        annotationName == propertyAnnotationName
    }

}
