package io.datalbry.config.processor.kotlin.finder

import com.google.devtools.ksp.processing.KSPLogger
import com.google.devtools.ksp.processing.Resolver
import com.google.devtools.ksp.symbol.KSAnnotation
import com.google.devtools.ksp.symbol.KSClassDeclaration
import com.google.devtools.ksp.symbol.KSPropertyDeclaration
import io.datalbry.config.api.PropertyType
import io.datalbry.config.api.annotation.ConfigType
import io.datalbry.config.processor.kotlin.extension.get
import io.datalbry.config.processor.kotlin.mapper.KSAnnotationToPropertyDescriptorMapper

/**
 * [PropertyTypeFinder] is a [Finder] implementation to fetch [PropertyType]
 * from all [ConfigType] annotated symbols.
 *
 * @param logger to use for logging
 * @param resolver to resolve the symbols with
 *
 * @author timo gruen - 2021-04-07
 */
class PropertyTypeFinder(
    private val logger: KSPLogger,
    private val resolver: Resolver
): Finder<PropertyType> {

    private val mapper = KSAnnotationToPropertyDescriptorMapper()

    override fun find(): Set<PropertyType> {
        val annotatedClasses = resolver.getSymbolsWithAnnotation(ConfigType::class.java.canonicalName)
        return annotatedClasses
            .asSequence()
            .filterIsInstance<KSClassDeclaration>()
            .map { it to it.getAllProperties().filter(this::isAnnotatedProperty) }
            .map { it.first to it.second.map(mapper::map) }
            .map { it.first to it.second.flatten() }
            .map { getKey(it.first) to it.second }
            .map { PropertyType(it.first, it.second.toSet()) }
            .toSet()
    }

    private fun getKey(clazz: KSClassDeclaration): String {
        val configTypeAnnotation = clazz.annotations.toList().getAnnotationByType<ConfigType>()
        val key = configTypeAnnotation.arguments["key"].value as String?
        return if (key.isNullOrBlank()) clazz.simpleName.getShortName() else key
    }

    private fun isAnnotatedProperty(p: KSPropertyDeclaration) = p.annotations.any {
        val annotationName = it.annotationType.resolve().declaration.qualifiedName?.asString()
        val propertyAnnotationName = io.datalbry.config.api.annotation.PropertyDescription::class.java.canonicalName
        annotationName == propertyAnnotationName
    }

    private inline fun <reified T> List<KSAnnotation>.getAnnotationByType() = this.first {
        val annotationName = it.annotationType.resolve().declaration.qualifiedName?.asString()
        val annotationNameToFind = T::class.java.canonicalName
        annotationName == annotationNameToFind
    }

}

