/*
 * Copyright (c) 2014-2022 Stream.io Inc. All rights reserved.
 *
 * Licensed under the Stream License;
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    https://github.com/GetStream/stream-android-push/blob/main/LICENSE
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.getstream.android.push.xiaomi

import com.squareup.moshi.JsonAdapter
import com.squareup.moshi.Moshi
import com.squareup.moshi.Types
import com.xiaomi.mipush.sdk.MiPushClient
import com.xiaomi.mipush.sdk.MiPushCommandMessage
import com.xiaomi.mipush.sdk.MiPushMessage
import io.getstream.android.push.PushDevice
import io.getstream.android.push.PushProvider
import io.getstream.android.push.delegate.PushDelegateProvider

/**
 * Helper class for delegating Xiaomi push messages to the Stream Chat SDK.
 */
public object XiaomiMessagingDelegate {

    internal var fallbackProviderName: String? = null

    private val mapAdapter: JsonAdapter<MutableMap<String, String>> by lazy {
        Moshi.Builder()
            .build()
            .adapter(Types.newParameterizedType(Map::class.java, String::class.java, String::class.java))
    }

    /**
     * Handles [miPushMessage] from Xiaomi.
     * If the [miPushMessage] wasn't sent from the Stream Server and doesn't contain the needed data,
     * return false to notify you that this remoteMessage needs to be handled by you.
     *
     * @param miPushMessage The message to be handled.
     * @return True if the [miPushMessage] was sent from the Stream Server and has been handled.
     */
    @JvmStatic
    public fun handleMiPushMessage(miPushMessage: MiPushMessage): Boolean {
        return PushDelegateProvider.delegates.any { it.handlePushMessage(miPushMessage.contentMap) }
    }

    /**
     * Register new Xiaomi Token.
     *
     * @param miPushCommandMessage provided by Xiaomi.
     * @param providerName Optional name for the provider name.
     */
    @JvmStatic
    public fun registerXiaomiToken(
        miPushCommandMessage: MiPushCommandMessage,
        providerName: String? = fallbackProviderName,
    ) {
        miPushCommandMessage
            .takeIf { it.command == MiPushClient.COMMAND_REGISTER }
            ?.commandArguments
            ?.get(0)
            ?.run {
                val pushDevice = PushDevice(
                    token = this,
                    pushProvider = PushProvider.XIAOMI,
                    providerName = providerName,
                )
                PushDelegateProvider.delegates.forEach { it.registerPushDevice(pushDevice) }
            }
    }

    /**
     * Transform [MiPushMessage.content] into a [Map].
     *
     * Return a [Map] obtained from the value of [MiPushMessage.content] or an empty map if content was empty.
     */
    private val MiPushMessage.contentMap: Map<String, String>
        get() = mapAdapter.fromJson(content) ?: emptyMap()
}
