/*
 * Copyright (c) 2014-2022 Stream.io Inc. All rights reserved.
 *
 * Licensed under the Stream License;
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    https://github.com/GetStream/stream-chat-android/blob/main/LICENSE
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.getstream.chat.android.offline.plugin.listener.internal

import io.getstream.chat.android.client.extensions.updateFailedMessage
import io.getstream.chat.android.client.extensions.updateMessageOnlineState
import io.getstream.chat.android.client.models.Message
import io.getstream.chat.android.client.plugin.listeners.EditMessageListener
import io.getstream.chat.android.client.setup.state.ClientState
import io.getstream.chat.android.client.utils.Result
import io.getstream.chat.android.client.utils.SyncStatus
import io.getstream.chat.android.offline.plugin.logic.internal.LogicRegistry

/**
 * Implementation of [EditMessageListener] that deals with state read and write.
 *
 * @param logic [LogicRegistry]
 * @param clientState [ClientState]
 */
internal class EditMessageListenerState(
    private val logic: LogicRegistry,
    private val clientState: ClientState,
) : EditMessageListener {

    /**
     * Method called when a message edit request happens. This method should be used to update messages locally.
     *
     * @param message [Message].
     */
    override suspend fun onMessageEditRequest(message: Message) {
        val isOnline = clientState.isNetworkAvailable
        val messagesToEdit = message.updateMessageOnlineState(isOnline)

        logic.channelFromMessage(messagesToEdit)?.stateLogic()?.upsertMessage(messagesToEdit)
        logic.threadFromMessage(messagesToEdit)?.stateLogic()?.upsertMessage(messagesToEdit)
    }

    /**
     * Method called when an edition in a message returns from the API. Updates the local messages accordingly.
     *
     * @param originalMessage [Message].
     * @param result the result of the API call.
     */
    override suspend fun onMessageEditResult(originalMessage: Message, result: Result<Message>) {
        val parsedMessage = if (result.isSuccess) {
            val message = result.data()
            message.copy(syncStatus = SyncStatus.COMPLETED)
        } else {
            originalMessage.updateFailedMessage(result.error())
        }

        logic.channelFromMessage(parsedMessage)?.stateLogic()?.upsertMessage(parsedMessage)
        logic.threadFromMessage(parsedMessage)?.stateLogic()?.upsertMessage(parsedMessage)
    }
}
