package io.getstream.chat.java.models;

import com.fasterxml.jackson.annotation.JsonProperty;
import io.getstream.chat.java.models.Command.CommandCreateRequestData.CommandCreateRequest;
import io.getstream.chat.java.models.Command.CommandUpdateRequestData.CommandUpdateRequest;
import io.getstream.chat.java.models.framework.StreamRequest;
import io.getstream.chat.java.models.framework.StreamResponse;
import io.getstream.chat.java.models.framework.StreamResponseObject;
import io.getstream.chat.java.services.CommandService;
import io.getstream.chat.java.services.framework.Client;
import java.util.Date;
import java.util.List;
import lombok.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import retrofit2.Call;

@Data
@NoArgsConstructor
public class Command {
  @NotNull
  @JsonProperty("created_at")
  private Date createdAt;

  @NotNull
  @JsonProperty("updated_at")
  private Date updatedAt;

  @NotNull
  @JsonProperty("name")
  private String name;

  @NotNull
  @JsonProperty("description")
  private String description;

  @Nullable
  @JsonProperty("args")
  private String args;

  @Nullable
  @JsonProperty("set")
  private String setValue;

  @Builder(
      builderClassName = "CommandCreateRequest",
      builderMethodName = "",
      buildMethodName = "internalBuild")
  public static class CommandCreateRequestData {
    @Nullable
    @JsonProperty("name")
    private String name;

    @Nullable
    @JsonProperty("description")
    private String description;

    @Nullable
    @JsonProperty("args")
    private String args;

    @Nullable
    @JsonProperty("set")
    private String setValue;

    public static class CommandCreateRequest extends StreamRequest<CommandCreateResponse> {
      @Override
      protected Call<CommandCreateResponse> generateCall(Client client) {
        return client.create(CommandService.class).create(this.internalBuild());
      }
    }
  }

  @RequiredArgsConstructor
  public static class CommandGetRequest extends StreamRequest<CommandGetResponse> {
    @NotNull private String name;

    @Override
    protected Call<CommandGetResponse> generateCall(Client client) {
      return client.create(CommandService.class).get(name);
    }
  }

  @Builder(
      builderClassName = "CommandUpdateRequest",
      builderMethodName = "",
      buildMethodName = "internalBuild")
  public static class CommandUpdateRequestData {
    @Nullable
    @JsonProperty("description")
    private String description;

    @Nullable
    @JsonProperty("args")
    private String args;

    @Nullable
    @JsonProperty("set")
    private String setValue;

    public static class CommandUpdateRequest extends StreamRequest<CommandUpdateResponse> {
      @NotNull private String name;

      private CommandUpdateRequest(@NotNull String name) {
        this.name = name;
      }

      @Override
      protected Call<CommandUpdateResponse> generateCall(Client client) {
        return client.create(CommandService.class).update(name, this.internalBuild());
      }
    }
  }

  @RequiredArgsConstructor
  public static class CommandDeleteRequest extends StreamRequest<CommandDeleteResponse> {
    @NotNull private String name;

    @Override
    protected Call<CommandDeleteResponse> generateCall(Client client) {
      return client.create(CommandService.class).delete(name);
    }
  }

  public static class CommandListRequest extends StreamRequest<CommandListResponse> {
    @Override
    protected Call<CommandListResponse> generateCall(Client client) {
      return client.create(CommandService.class).list();
    }
  }

  @Data
  @NoArgsConstructor
  @EqualsAndHashCode(callSuper = true)
  public static class CommandCreateResponse extends StreamResponseObject {
    @NotNull
    @JsonProperty("command")
    private Command command;
  }

  @Data
  @NoArgsConstructor
  @EqualsAndHashCode(callSuper = true)
  public static class CommandGetResponse extends Command implements StreamResponse {
    private RateLimit rateLimit;

    @NotNull
    @JsonProperty("duration")
    private String duration;
  }

  @Data
  @NoArgsConstructor
  @EqualsAndHashCode(callSuper = true)
  public static class CommandUpdateResponse extends StreamResponseObject {
    @NotNull
    @JsonProperty("command")
    private Command command;
  }

  @Data
  @NoArgsConstructor
  @EqualsAndHashCode(callSuper = true)
  public static class CommandDeleteResponse extends StreamResponseObject {
    @NotNull
    @JsonProperty("name")
    private String name;
  }

  @Data
  @NoArgsConstructor
  @EqualsAndHashCode(callSuper = true)
  public static class CommandListResponse extends StreamResponseObject {
    @NotNull
    @JsonProperty("commands")
    private List<Command> commands;
  }

  /**
   * Creates a create request
   *
   * @return the created request
   */
  @NotNull
  public static CommandCreateRequest create() {
    return new CommandCreateRequest();
  }

  /**
   * Creates a get request
   *
   * @param name the command name
   * @return the created request
   */
  @NotNull
  public static CommandGetRequest get(@NotNull String name) {
    return new CommandGetRequest(name);
  }

  /**
   * Creates an update request
   *
   * @param name the command name
   * @return the created request
   */
  @NotNull
  public static CommandUpdateRequest update(@NotNull String name) {
    return new CommandUpdateRequest(name);
  }

  /**
   * Creates a delete request
   *
   * @param name the command name
   * @return the created request
   */
  @NotNull
  public static CommandDeleteRequest delete(@NotNull String name) {
    return new CommandDeleteRequest(name);
  }

  /**
   * Creates a list request
   *
   * @return the created request
   */
  @NotNull
  public static CommandListRequest list() {
    return new CommandListRequest();
  }
}
