package arp.message.rocketmq;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.apache.rocketmq.acl.common.AclClientRPCHook;
import org.apache.rocketmq.acl.common.SessionCredentials;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyContext;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyStatus;
import org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.common.message.MessageExt;

import arp.process.publish.Message;
import arp.process.publish.ProcessMessageReceiver;

public class RocketmgMessageReceiver implements ProcessMessageReceiver {
	private DefaultMQPushConsumer consumer;
	private ConcurrentLinkedQueue<Message> messageQueue;
	private String topic = "arp_process_message";
	private RocketmqMessageDeserializationStrategy deserializationStrategy;

	public RocketmgMessageReceiver(String consumerGroup, String namesrvAddr)
			throws MQClientException {
		this(consumerGroup, namesrvAddr, new FSTDeserializationStrategy());
	}

	public RocketmgMessageReceiver(String consumerGroup, String namesrvAddr,
			RocketmqMessageDeserializationStrategy deserializationStrategy)
			throws MQClientException {
		this.deserializationStrategy = deserializationStrategy;
		messageQueue = new ConcurrentLinkedQueue<>();
		// 实例化消费者
		consumer = new DefaultMQPushConsumer(consumerGroup);
		// 设置NameServer的地址
		consumer.setNamesrvAddr(namesrvAddr);
		consumer.subscribe(topic, "*");
		// 注册回调实现类来处理从broker拉取回来的消息
		consumer.registerMessageListener(getMessageListener());
		// 启动消费者实例
		consumer.start();
	}

	private MessageListenerConcurrently getMessageListener() {
		return new MessageListenerConcurrently() {
			@Override
			public ConsumeConcurrentlyStatus consumeMessage(
					List<MessageExt> msgs, ConsumeConcurrentlyContext context) {
				if (msgs != null) {
					for (MessageExt msg : msgs) {
						try {
							messageQueue.offer(deserializationStrategy
									.deserialize(msg.getBody()));
						} catch (Exception e) {
							e.printStackTrace();
						}

					}
				}
				// 标记该消息已经被成功消费
				return ConsumeConcurrentlyStatus.CONSUME_SUCCESS;
			}
		};
	}

	public RocketmgMessageReceiver(String consumerGroup, String namespace,
			String namesrvAddr, String accessKey, String secretKey,
			String topic,
			RocketmqMessageDeserializationStrategy deserializationStrategy)
			throws MQClientException {
		this.deserializationStrategy = deserializationStrategy;
		this.topic = topic;
		messageQueue = new ConcurrentLinkedQueue<>();
		// 实例化消费者
		consumer = new DefaultMQPushConsumer(namespace, consumerGroup,
				new AclClientRPCHook(new SessionCredentials(accessKey,
						secretKey))); // ACL权限
		// 设置NameServer的地址
		consumer.setNamesrvAddr(namesrvAddr);
		consumer.subscribe(this.topic, "*");
		// 注册回调实现类来处理从broker拉取回来的消息
		consumer.registerMessageListener(getMessageListener());
		// 启动消费者实例
		consumer.start();
	}

	public RocketmgMessageReceiver(String consumerGroup, String namespace,
			String namesrvAddr, String accessKey, String secretKey, String topic)
			throws MQClientException {
		this(consumerGroup, namespace, namesrvAddr, accessKey, secretKey,
				topic, new FSTDeserializationStrategy());
	}

	@Override
	public List<Message> receive() throws Exception {
		List<Message> msgLst = new ArrayList<>();
		int size = messageQueue.size();
		for (int i = 0; i < size; i++) {
			msgLst.add(messageQueue.poll());
		}
		return msgLst;
	}
}
