package arp.message.kafka;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Properties;

import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.KafkaAdminClient;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.serialization.ByteArraySerializer;
import org.apache.kafka.common.serialization.StringSerializer;
import org.nustaq.serialization.FSTConfiguration;
import org.nustaq.serialization.FSTObjectOutput;

import arp.process.publish.Message;
import arp.process.publish.MessageSender;

public class KafkaMessageSender implements MessageSender {

	private KafkaProducer<String, byte[]> producer;

	private FSTConfiguration fstConf;

	private Properties props;

	public KafkaMessageSender(String servers) {
		fstConf = FSTConfiguration.createDefaultConfiguration();
		fstConf.setForceSerializable(true);

		props = new Properties();
		props.put(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG, servers);
		props.put(ProducerConfig.KEY_SERIALIZER_CLASS_CONFIG,
				StringSerializer.class);
		props.put(ProducerConfig.VALUE_SERIALIZER_CLASS_CONFIG,
				ByteArraySerializer.class);
		props.put(ProducerConfig.MAX_BLOCK_MS_CONFIG, 60000);
		producer = new KafkaProducer<>(props);
	}

	@Override
	public void send(Message msg) throws Exception {
		ByteArrayOutputStream bos = new ByteArrayOutputStream();
		FSTObjectOutput oos = fstConf.getObjectOutput(bos);
		oos.writeObject(msg);
		oos.flush();

		ProducerRecord<String, byte[]> record = new ProducerRecord<>(
				msg.getProcessDesc(), bos.toByteArray());
		producer.send(record).get();
	}

	@Override
	public void defineProcessesToSend(List<String> processesToSend) {
		AdminClient adminClient = KafkaAdminClient.create(props);

		// TODO 可配置
		Integer numPartitions = 1;
		Short replicationFactor = 1;

		List<NewTopic> topics = new ArrayList<>();
		for (String process : processesToSend) {
			topics.add(new NewTopic(process, Optional.of(numPartitions),
					Optional.of(replicationFactor)));
		}

		adminClient.createTopics(topics);
		adminClient.close();

	}
}
