package arp.message.kafka;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.KafkaAdminClient;
import org.apache.kafka.clients.admin.ListTopicsResult;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.KafkaFuture;
import org.apache.kafka.common.serialization.StringDeserializer;

import arp.process.publish.MonitorMessage;
import arp.process.publish.ProcessesMonitor;

import com.google.gson.Gson;

public class KafkaProcessesMonitor extends ProcessesMonitor {

	private Gson gson;

	private KafkaConsumer<String, String> consumer;

	private AdminClient adminClient;

	private boolean emptySubscription;

	private String monitorTopicPrefix;

	public KafkaProcessesMonitor(String servers, String consumerGroup,
			String monitorTopicPrefix) {
		gson = new Gson();
		this.monitorTopicPrefix = monitorTopicPrefix;
		Properties props = new Properties();
		props.put(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG, servers);
		props.put(ConsumerConfig.GROUP_ID_CONFIG, consumerGroup);
		props.put(ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG,
				StringDeserializer.class);
		props.put(ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG,
				StringDeserializer.class);
		consumer = new KafkaConsumer<>(props);

		Properties pro = new Properties();
		pro.put(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG, servers);
		adminClient = KafkaAdminClient.create(pro);
	}

	@Override
	protected void subscribeProcesses(List<String> processesToSubscribe) {
		if (processesToSubscribe != null && !processesToSubscribe.isEmpty()) {
			List<String> topics = new ArrayList<>();
			for (String process : processesToSubscribe) {
				topics.add(monitorTopicPrefix + process);
			}
			consumer.subscribe(topics);
			emptySubscription = false;
		} else {
			emptySubscription = true;
		}
	}

	@Override
	protected List<String> queryAllProcessesToSubscribe() {
		ListTopicsResult result = adminClient.listTopics();
		KafkaFuture<Set<String>> set = result.names();
		try {
			Set<String> topicsSet = set.get();
			List<String> rslt = new ArrayList<>();
			for (String topic : topicsSet) {
				rslt.add(topic);
			}
			return rslt;
		} catch (Exception e) {
			return null;
		}
	}

	@Override
	protected List<MonitorMessage> receive() throws Exception {
		List<MonitorMessage> messageList = new ArrayList<>();
		if (emptySubscription) {
			return messageList;
		}
		ConsumerRecords<String, String> records = consumer.poll(Duration
				.ofMillis(100));
		for (ConsumerRecord<String, String> record : records) {
			String msg = record.value();
			messageList.add(gson.fromJson(msg, MonitorMessage.class));
		}
		return messageList;
	}
}
