package arp.message.kafka;

import java.io.ByteArrayInputStream;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;

import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.KafkaAdminClient;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.apache.kafka.common.serialization.StringSerializer;
import org.nustaq.serialization.FSTConfiguration;
import org.nustaq.serialization.FSTObjectInput;

import arp.process.publish.Message;
import arp.process.publish.MonitorMessage;
import arp.process.publish.MonitorMessageConvertor;

import com.google.gson.Gson;

public class KafkaMonitorMessageConvertor extends MonitorMessageConvertor {

	FSTConfiguration fstConf;

	private String monitorTopicPrefix;
	private KafkaConsumer<String, byte[]> consumer;
	private KafkaProducer<String, String> producer;
	private Properties producerProps;
	private Gson gson;

	public KafkaMonitorMessageConvertor(String servers, String consumerGroup,
			String monitorTopicPrefix) {
		fstConf = FSTConfiguration.createDefaultConfiguration();
		fstConf.setForceSerializable(true);
		gson = new Gson();
		this.monitorTopicPrefix = monitorTopicPrefix;
		Properties consumerProps = new Properties();
		consumerProps.put(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG, servers);
		consumerProps.put(ConsumerConfig.GROUP_ID_CONFIG, consumerGroup);
		consumerProps.put(ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG,
				StringDeserializer.class);
		consumerProps.put(ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG,
				ByteArrayDeserializer.class);
		consumer = new KafkaConsumer<>(consumerProps);

		producerProps = new Properties();
		producerProps.put(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG, servers);
		producerProps.put(ProducerConfig.KEY_SERIALIZER_CLASS_CONFIG,
				StringSerializer.class);
		producerProps.put(ProducerConfig.VALUE_SERIALIZER_CLASS_CONFIG,
				StringSerializer.class);
		producerProps.put(ProducerConfig.MAX_BLOCK_MS_CONFIG, 60000);
		producer = new KafkaProducer<>(producerProps);
	}

	@Override
	protected void send(MonitorMessage monitorMessage) throws Exception {
		ProducerRecord<String, String> record = new ProducerRecord<>(
				monitorTopicPrefix + monitorMessage.getProcessDesc(),
				gson.toJson(monitorMessage));
		producer.send(record).get();
	}

	@Override
	protected MonitorMessage convertMessage(Message msg) {
		MonitorMessage monitorMessage = new MonitorMessage();
		monitorMessage.setProcessDesc(msg.getProcessDesc());
		monitorMessage.setProcessInputs(gson.toJson(msg.getProcessInput()));
		monitorMessage.setProcessOutput(gson.toJson(msg.getProcessOutput()));
		List<Map> processCreatedAggrs = new ArrayList<>();
		for (Object aggr : msg.getProcessCreatedAggrs()) {
			Map aggrDto = new HashMap();
			aggrDto.put("class", aggr.getClass().getName());
			aggrDto.put("aggr", aggr);
			processCreatedAggrs.add(aggrDto);
		}
		monitorMessage.setProcessCreatedAggrs(gson.toJson(processCreatedAggrs));
		List<Map> processDeletedAggrs = new ArrayList<>();
		for (Object aggr : msg.getProcessDeletedAggrs()) {
			Map aggrDto = new HashMap();
			aggrDto.put("class", aggr.getClass().getName());
			aggrDto.put("aggr", aggr);
			processDeletedAggrs.add(aggrDto);
		}
		monitorMessage.setProcessDeletedAggrs(gson.toJson(processDeletedAggrs));
		List<Map[]> processUpdatedAggrs = new ArrayList<>();
		for (Object[] aggrs : msg.getProcessUpdatedAggrs()) {
			Map[] aggrDtos = new Map[2];
			Map aggrDto0 = new HashMap();
			aggrDto0.put("class", aggrs[0].getClass().getName());
			aggrDto0.put("aggr", aggrs[0]);
			aggrDtos[0] = aggrDto0;
			Map aggrDto1 = new HashMap();
			aggrDto1.put("class", aggrs[1].getClass().getName());
			aggrDto1.put("aggr", aggrs[1]);
			aggrDtos[1] = aggrDto1;
			processUpdatedAggrs.add(aggrDtos);
		}
		monitorMessage.setProcessUpdatedAggrs(gson.toJson(processUpdatedAggrs));
		monitorMessage.setProcessFinishTime(msg.getProcessFinishTime());
		return monitorMessage;
	}

	@Override
	protected List<Message> receive() throws Exception {
		List<Message> messageList = new ArrayList<>();
		ConsumerRecords<String, byte[]> records = consumer.poll(Duration
				.ofMillis(100));
		for (ConsumerRecord<String, byte[]> record : records) {
			byte[] msg = record.value();
			FSTObjectInput ois = fstConf
					.getObjectInput(new ByteArrayInputStream(msg));
			messageList.add((Message) ois.readObject());
		}
		return messageList;
	}

	@Override
	protected void subscribeProcesses(List<String> processesToSubscribe) {
		if (processesToSubscribe != null) {
			consumer.subscribe(processesToSubscribe);
		}
	}

	@Override
	protected void defineProcessesToPublish(List<String> processesToPublish) {
		AdminClient adminClient = KafkaAdminClient.create(producerProps);

		// TODO 可配置
		Integer numPartitions = 1;
		Short replicationFactor = 1;

		List<NewTopic> topics = new ArrayList<>();
		for (String process : processesToPublish) {
			topics.add(new NewTopic(monitorTopicPrefix + process, Optional
					.of(numPartitions), Optional.of(replicationFactor)));
		}

		adminClient.createTopics(topics);
		adminClient.close();
	}

}
