package arp.message.kafka;

import java.io.ByteArrayInputStream;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.nustaq.serialization.FSTConfiguration;
import org.nustaq.serialization.FSTObjectInput;

import arp.process.publish.Message;
import arp.process.publish.ProcessListenerMessageReceiver;

public class KafkaMessageReceiver implements ProcessListenerMessageReceiver {
	private String name;
	private String servers;

	private FSTConfiguration fstConf;

	private KafkaConsumer[] consumers;

	public KafkaMessageReceiver(String servers, String name) {
		this.name = name;
		this.servers = servers;
		fstConf = FSTConfiguration.createDefaultConfiguration();
		fstConf.setForceSerializable(true);
	}

	@Override
	public List<Message> receive() throws Exception {
		List<Message> messageList = new ArrayList<>();
		for (KafkaConsumer<String, byte[]> consumer : consumers) {
			ConsumerRecords<String, byte[]> records = consumer.poll(Duration
					.ofMillis(100));
			for (ConsumerRecord<String, byte[]> record : records) {
				byte[] msg = record.value();
				FSTObjectInput ois = fstConf
						.getObjectInput(new ByteArrayInputStream(msg));
				messageList.add((Message) ois.readObject());
			}
		}
		return messageList;
	}

	@Override
	public void subscribeProcesses(List<String> processesToSubscribe) {
		if (processesToSubscribe != null) {
			consumers = new KafkaConsumer[processesToSubscribe.size()];
			for (int i = 0; i < processesToSubscribe.size(); i++) {
				String topic = processesToSubscribe.get(i);
				Properties props = new Properties();
				props.put(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG, servers);
				props.put(ConsumerConfig.GROUP_ID_CONFIG, topic + "FOR" + name);
				props.put(ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG,
						StringDeserializer.class);
				props.put(ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG,
						ByteArrayDeserializer.class);
				KafkaConsumer<String, byte[]> consumer = new KafkaConsumer<>(
						props);
				List<String> topics = new ArrayList<>();
				topics.add(topic);
				consumer.subscribe(topics);
				consumers[i] = consumer;
			}
		}
	}
}
