package arp.repository.springdatamongodb;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.LockSupport;

import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;

import arp.repository.PersistenceRepository;

import com.mongodb.client.result.UpdateResult;

public abstract class MappedMongodbRepository<E, DTO, I> extends
		PersistenceRepository<E, I> {

	protected MongoTemplate mongoTemplate;

	int lockRetryCount = 300;

	protected MappedMongodbRepository(MongoTemplate mongoTemplate) {
		if (mongoTemplate == null) {
			initAsMock();
		} else {
			this.mongoTemplate = mongoTemplate;
		}
	}

	protected abstract E toEntity(DTO dto);

	protected abstract Class<DTO> getDtoClass();

	protected abstract DTO toDTO(E entity);

	@Override
	protected E findByIdForUpdateImpl(I id) {
		boolean hasLock = acquireLock(id);
		if (!hasLock) {
			return null;
		}
		return findByIdImpl(id);
	}

	private void createLock(I id) {
		Class<DTO> dtoCls = getDtoClass();
		String collectionName = "arp_repo_state_" + dtoCls.getSimpleName();
		Map<String, Object> lock = new HashMap<>();
		lock.put("_id", id);
		lock.put("state", 1);
		try {
			mongoTemplate.insert(lock, collectionName);
		} catch (Exception e) {
		}
	}

	// 如果没有数据也返回，返回false，有数据的话那就争夺锁，抢到了返回true，抢不到抛出CanNotAcquireLockException
	private boolean acquireLock(I id) {
		Query query = new Query();
		query.addCriteria(Criteria.where("_id").is(id).and("state").is(0));
		Update update = new Update();
		update.set("state", 1);
		Class<DTO> dtoCls = getDtoClass();
		String collectionName = "arp_repo_state_" + dtoCls.getSimpleName();

		int counter = lockRetryCount;
		do {
			UpdateResult updateResult = mongoTemplate.updateFirst(query,
					update, collectionName);
			if (updateResult.getModifiedCount() > 0) {
				return true;
			}
			if (updateResult.getMatchedCount() == 0) {
				Query checkQuery = new Query();
				checkQuery.addCriteria(Criteria.where("_id").is(id));
				boolean exists = mongoTemplate.exists(checkQuery, dtoCls);
				if (!exists) {
					return false;
				} else {
					createLock(id);
				}
			}

			if (counter > 200) {
				--counter;
			} else if (counter > 100) {
				--counter;
				Thread.yield();
			} else if (counter > 0) {
				--counter;
				LockSupport.parkNanos(1L);
			} else {
				throw new CanNotAcquireLockException();
			}
		} while (true);
	}

	@Override
	protected E findByIdImpl(I id) {
		DTO dto = mongoTemplate.findById(id, getDtoClass());
		if (dto == null) {
			return null;
		}
		return toEntity(dto);
	}

	@Override
	protected E saveIfAbsentImpl(I id, E entity) {
		DTO dto = toDTO(entity);
		try {
			mongoTemplate.insert(dto);// 已有数据就会报错，利用这一点
			createLock(id);
			return null;
		} catch (Exception e) {
			while (!acquireLock(id)) {
			}
			return findByIdImpl(id);
		}
	}

	@Override
	protected void updateBatchImpl(Map<I, E> entitiesToUpdate) {
		entitiesToUpdate.forEach((id, ett) -> {
			updateImpl(id, ett);
		});
	}

	@Override
	protected void updateImpl(I id, E entity) {
		mongoTemplate.save(toDTO(entity));
	}

	private void unlock(I id) {
		Query query = new Query();
		query.addCriteria(Criteria.where("_id").is(id));
		Update update = new Update();
		update.set("state", 0);
		Class<DTO> dtoCls = getDtoClass();
		String collectionName = "arp_repo_state_" + dtoCls.getSimpleName();
		mongoTemplate.updateFirst(query, update, collectionName);
	}

	@Override
	protected void saveBatchImpl(Map<I, E> entities) {
		List<DTO> dtos = new ArrayList<>();
		entities.values().forEach((ett) -> {
			dtos.add(toDTO(ett));
		});
		mongoTemplate.insert(dtos, getDtoClass());
	}

	@Override
	protected void saveImpl(I id, E entity) {
		mongoTemplate.save(toDTO(entity));
	}

	@Override
	protected void removeBatchImpl(Set<I> ids) {
		ids.forEach((id) -> {
			removeImpl(id);
		});
	}

	@Override
	protected void removeImpl(I id) {
		DTO dto = mongoTemplate.findById(id, getDtoClass());
		if (dto != null) {
			mongoTemplate.remove(dto);
			removeLock(id);
		}
	}

	private void removeLock(I id) {
		Query query = new Query();
		query.addCriteria(Criteria.where("_id").is(id));
		Class<DTO> dtoCls = getDtoClass();
		String collectionName = "arp_repo_state_" + dtoCls.getSimpleName();
		mongoTemplate.remove(query, collectionName);
	}

	@Override
	protected void unlockBatchImpl(Set<I> ids) {
		ids.forEach((id) -> {
			unlockImpl(id);
		});
	}

	@Override
	protected void unlockImpl(I id) {
		unlock(id);
	}

	public I findMaxId() {
		if (mongoTemplate == null) {
			return null;
		}
		Query query = new Query();
		query.with(Sort.by(Sort.Direction.DESC, "_id"));
		query.limit(1);
		DTO maxDto = mongoTemplate.findOne(query, getDtoClass());
		if (maxDto == null) {
			return null;
		}
		return getId(toEntity(maxDto));
	}

	public List<E> findAllByField(String fieldName, Object fieldValue) {
		Query query = new Query();
		query.addCriteria(Criteria.where(fieldName).is(fieldValue));
		List<DTO> dtos = mongoTemplate.find(query, getDtoClass());
		List<E> etts = new ArrayList<>();
		for (DTO dto : dtos) {
			etts.add(toEntity(dto));
		}
		return etts;
	}
}
