package arp.repository.springdatamongodb;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.LockSupport;

import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;

import arp.repository.PersistenceRepository;

import com.mongodb.client.result.UpdateResult;

public abstract class MongodbRepository<E, I> extends
		PersistenceRepository<E, I> {

	protected MongoTemplate mongoTemplate;

	int lockRetryCount = 300;

	protected MongodbRepository(MongoTemplate mongoTemplate) {
		if (mongoTemplate == null) {
			initAsMock();
		} else {
			this.mongoTemplate = mongoTemplate;
		}
	}

	private void createLock(I id) {
		Class<E> ettCls = getEntityClass();
		String collectionName = "arp_repo_state_" + ettCls.getSimpleName();
		Map<String, Object> lock = new HashMap<>();
		lock.put("_id", id);
		lock.put("state", 1);
		try {
			mongoTemplate.insert(lock, collectionName);
		} catch (Exception e) {
		}
	}

	@Override
	protected E findByIdForUpdateImpl(I id) {
		boolean hasLock = acquireLock(id);
		if (!hasLock) {
			return null;
		}
		return findByIdImpl(id);
	}

	// 如果没有数据也返回，返回false，有数据的话那就争夺锁，抢到了返回true，抢不到抛出CanNotAcquireLockException
	private boolean acquireLock(I id) {
		Query query = new Query();
		query.addCriteria(Criteria.where("_id").is(id).and("state").is(0));
		Update update = new Update();
		update.set("state", 1);
		Class<E> ettCls = getEntityClass();
		String collectionName = "arp_repo_state_" + ettCls.getSimpleName();

		int counter = lockRetryCount;
		do {
			UpdateResult updateResult = mongoTemplate.updateFirst(query,
					update, collectionName);
			if (updateResult.getModifiedCount() > 0) {
				return true;
			}
			if (updateResult.getMatchedCount() == 0) {
				Query checkQuery = new Query();
				checkQuery.addCriteria(Criteria.where("_id").is(id));
				boolean exists = mongoTemplate.exists(checkQuery, ettCls);
				if (!exists) {
					return false;
				} else {
					createLock(id);
				}
			}

			if (counter > 200) {
				--counter;
			} else if (counter > 100) {
				--counter;
				Thread.yield();
			} else if (counter > 0) {
				--counter;
				LockSupport.parkNanos(1L);
			} else {
				throw new CanNotAcquireLockException();
			}
		} while (true);
	}

	protected abstract Class<E> getEntityClass();

	@Override
	protected E findByIdImpl(I id) {
		return mongoTemplate.findById(id, getEntityClass());
	}

	@Override
	protected E saveIfAbsentImpl(I id, E entity) {
		try {
			mongoTemplate.insert(entity);// 已有数据就会报错，利用这一点
			createLock(id);
			return null;
		} catch (Exception e) {
			while (!acquireLock(id)) {
			}
			return findByIdImpl(id);
		}
	}

	@Override
	protected void updateBatchImpl(Map<I, E> entitiesToUpdate) {
		entitiesToUpdate.forEach((id, ett) -> {
			updateImpl(id, ett);
		});
	}

	@Override
	protected void updateImpl(I id, E entity) {
		mongoTemplate.save(entity);
	}

	protected void unlock(I id) {
		Query query = new Query();
		query.addCriteria(Criteria.where("_id").is(id));
		Update update = new Update();
		update.set("state", 0);
		Class<E> ettCls = getEntityClass();
		String collectionName = "arp_repo_state_" + ettCls.getSimpleName();
		mongoTemplate.updateFirst(query, update, collectionName);
	}

	@Override
	protected void saveBatchImpl(Map<I, E> entities) {
		mongoTemplate.insert(entities.values(), getEntityClass());
	}

	@Override
	protected void saveImpl(I id, E entity) {
		mongoTemplate.save(entity);
	}

	@Override
	protected void removeBatchImpl(Set<I> ids) {
		ids.forEach((id) -> {
			removeImpl(id);
		});
	}

	@Override
	protected void removeImpl(I id) {
		E ett = findByIdImpl(id);
		if (ett != null) {
			mongoTemplate.remove(ett);
			removeLock(id);
		}
	}

	private void removeLock(I id) {
		Query query = new Query();
		query.addCriteria(Criteria.where("_id").is(id));
		Class<E> ettCls = getEntityClass();
		String collectionName = "arp_repo_state_" + ettCls.getSimpleName();
		mongoTemplate.remove(query, collectionName);
	}

	@Override
	protected void unlockBatchImpl(Set<I> ids) {
		ids.forEach((id) -> {
			unlockImpl(id);
		});
	}

	@Override
	protected void unlockImpl(I id) {
		unlock(id);
	}

	public I findMaxId() {
		if (mongoTemplate == null) {
			return null;
		}
		Query query = new Query();
		query.with(Sort.by(Sort.Direction.DESC, "_id"));
		query.limit(1);
		E maxEtt = mongoTemplate.findOne(query, getEntityClass());
		if (maxEtt == null) {
			return null;
		}
		return getId(maxEtt);
	}

	public List<E> findAllByField(String fieldName, Object fieldValue) {
		Query query = new Query();
		query.addCriteria(Criteria.where(fieldName).is(fieldValue));
		return mongoTemplate.find(query, getEntityClass());
	}

}
