package arp.repository.springdatamongodb;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.LockSupport;

import org.bson.Document;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;

import arp.enhance.ProcessInfo;
import arp.process.ProcessContext;
import arp.process.ThreadBoundProcessContextArray;
import arp.repository.PersistenceRepository;

public abstract class MongodbRepository<E, I> extends
		PersistenceRepository<E, I> {

	protected MongoTemplate mongoTemplate;

	int lockRetryCount = 300;

	protected MongodbRepository(MongoTemplate mongoTemplate) {
		if (mongoTemplate == null) {
			initAsMock();
		} else {
			this.mongoTemplate = mongoTemplate;
		}
	}

	private boolean createLock(I id) {
		Class<E> ettCls = getEntityClass();
		String collectionName = "arp_repo_state_" + ettCls.getSimpleName();
		Map<String, Object> lock = new HashMap<>();
		lock.put("_id", id);
		lock.put("state", 1);
		ProcessContext processContext = ThreadBoundProcessContextArray
				.getProcessContext();
		lock.put("piid", processContext.getProcessInfoId());
		try {
			mongoTemplate.insert(lock, collectionName);
			return true;
		} catch (Exception e) {
			return false;
		}
	}

	@Override
	protected E findByIdForUpdateImpl(I id) {
		boolean hasEntity = acquireLock(id);
		if (!hasEntity) {
			return null;
		}
		return findByIdImpl(id);
	}

	// 如果没有数据也返回，返回false，有数据的话那就争夺锁，抢到了返回true，抢不到抛出CanNotAcquireLockException
	private boolean acquireLock(I id) {
		ProcessContext processContext = ThreadBoundProcessContextArray
				.getProcessContext();
		Class<E> ettCls = getEntityClass();
		String collectionName = "arp_repo_state_" + ettCls.getSimpleName();
		Document cmd = new Document();
		cmd.put("findAndModify", collectionName);
		Document query = new Document();
		query.put("_id", id);
		query.put("state", 0);
		cmd.put("query", query);
		Document update = new Document();
		Document set = new Document();
		set.put("state", 1);
		set.put("piid", processContext.getProcessInfoId());
		update.put("$set", set);
		cmd.put("update", update);
		cmd.put("new", false);

		int counter = lockRetryCount;
		do {
			Document doc = mongoTemplate.executeCommand(cmd);
			Document lastErrorObject = doc.get("lastErrorObject",
					Document.class);
			boolean updatedExisting = lastErrorObject
					.getBoolean("updatedExisting");
			if (updatedExisting) {
				return true;
			}
			Query lockExistsCheckQuery = new Query();
			lockExistsCheckQuery.addCriteria(Criteria.where("_id").is(id));
			boolean lockExists = mongoTemplate.exists(lockExistsCheckQuery,
					collectionName);
			if (!lockExists) {
				Query entityExistsCheckQuery = new Query();
				entityExistsCheckQuery
						.addCriteria(Criteria.where("_id").is(id));
				boolean entityExists = mongoTemplate.exists(
						entityExistsCheckQuery, ettCls);
				if (!entityExists) {
					return false;
				} else {
					if (createLock(id)) {
						return true;
					} else {
						continue;
					}
				}
			}

			if (counter > 200) {
				--counter;
			} else if (counter > 100) {
				--counter;
				Thread.yield();
			} else if (counter > 0) {
				--counter;
				LockSupport.parkNanos(1L);
			} else {
				Document lockQueryCmd = new Document();
				lockQueryCmd.put("find", collectionName);
				Document filter = new Document();
				filter.put("_id", id);
				lockQueryCmd.put("filter", filter);
				Document lockQueryDoc = mongoTemplate
						.executeCommand(lockQueryCmd);
				Document cursor = lockQueryDoc.get("cursor", Document.class);
				List<Document> firstBatch = cursor.getList("firstBatch",
						Document.class);
				Document lockDoc = firstBatch.get(0);
				int piid = lockDoc.getInteger("piid");
				ProcessInfo processInfoGotLock = ProcessContext
						.getProcessInfo(piid);
				String processDesc;
				if (!processInfoGotLock.getProcessName().trim().isEmpty()) {
					processDesc = processInfoGotLock.getProcessName();
				} else {
					processDesc = processInfoGotLock.getClsName() + "."
							+ processInfoGotLock.getMthName();
				}
				throw new CanNotAcquireLockException(ettCls.getName(),
						processDesc);
			}
		} while (true);
	}

	protected abstract Class<E> getEntityClass();

	@Override
	protected E findByIdImpl(I id) {
		return mongoTemplate.findById(id, getEntityClass());
	}

	@Override
	protected E saveIfAbsentImpl(I id, E entity) {
		if (!createLock(id)) {
			acquireLock(id);
			return findByIdImpl(id);
		}
		mongoTemplate.insert(entity);
		return null;
	}

	@Override
	protected void updateBatchImpl(Map<I, E> entitiesToUpdate) {
		entitiesToUpdate.forEach((id, ett) -> {
			updateImpl(id, ett);
		});
	}

	@Override
	protected void updateImpl(I id, E entity) {
		mongoTemplate.save(entity);
	}

	protected void unlock(I id) {
		Query query = new Query();
		query.addCriteria(Criteria.where("_id").is(id));
		Update update = new Update();
		update.set("state", 0);
		Class<E> ettCls = getEntityClass();
		String collectionName = "arp_repo_state_" + ettCls.getSimpleName();
		mongoTemplate.updateFirst(query, update, collectionName);
	}

	@Override
	protected void saveBatchImpl(Map<I, E> entities) {
		mongoTemplate.insert(entities.values(), getEntityClass());
	}

	@Override
	protected void saveImpl(I id, E entity) {
		mongoTemplate.save(entity);
	}

	@Override
	protected void removeBatchImpl(Set<I> ids) {
		ids.forEach((id) -> {
			removeImpl(id);
		});
	}

	@Override
	protected void removeImpl(I id) {
		E ett = findByIdImpl(id);
		if (ett != null) {
			mongoTemplate.remove(ett);
			removeLock(id);
		}
	}

	private void removeLock(I id) {
		Query query = new Query();
		query.addCriteria(Criteria.where("_id").is(id));
		Class<E> ettCls = getEntityClass();
		String collectionName = "arp_repo_state_" + ettCls.getSimpleName();
		mongoTemplate.remove(query, collectionName);
	}

	@Override
	protected void unlockBatchImpl(Set<I> ids) {
		ids.forEach((id) -> {
			unlockImpl(id);
		});
	}

	@Override
	protected void unlockImpl(I id) {
		unlock(id);
	}

	public I findMaxId() {
		if (mongoTemplate == null) {
			return null;
		}
		Query query = new Query();
		query.with(Sort.by(Sort.Direction.DESC, "_id"));
		query.limit(1);
		E maxEtt = mongoTemplate.findOne(query, getEntityClass());
		if (maxEtt == null) {
			return null;
		}
		return getId(maxEtt);
	}

	public List<E> findAllByField(String fieldName, Object fieldValue) {
		Query query = new Query();
		query.addCriteria(Criteria.where(fieldName).is(fieldValue));
		return mongoTemplate.find(query, getEntityClass());
	}

	public List<E> findAll() {
		return mongoTemplate.findAll(getEntityClass());
	}

}
