package com.bots.service;

import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.stereotype.Component;

import arp.process.Process;
import arp.repository.CommonMemRepository;

import com.bots.aggr.TaskTemplate;
import com.bots.aggr.UseCase;
import com.bots.aggr.UserTask;

@Component
@Configuration
@EnableScheduling
public class TaskService {

	private ExecutorService executorService = Executors.newCachedThreadPool();

	private CommonMemRepository<UserTask> userTaskRepository;

	public TaskService() {
		userTaskRepository = new CommonMemRepository<>(UserTask.class);
	}

	@Process
	public void createTaskForUser(String taskTemplateType, long userId,
			Object... taskInputs) throws Exception {
		TaskTemplate taskTemplate = (TaskTemplate) Class.forName(
				taskTemplateType).newInstance();
		UserTask userTask = new UserTask(userId, taskTemplate.createTask(
				userId, taskInputs));
		userTaskRepository.save(userTask);
	}

	public void userStartTask(long userId) {
		executorService.submit(() -> {
			startTask(userId);
		});
	}

	@Process
	public void startTask(long userId) {
		UserTask userTask = userTaskRepository.findByIdForUpdate(userId);
		if (userTask == null) {
			return;
		}
		userTask.getTask().start();
	}

	@Process
	public UseCase queryUseCase(long userId, int caseIdx) {
		UserTask userTask = userTaskRepository.findById(userId);
		if (userTask == null) {
			return null;
		}
		return userTask.getTask().getCase(caseIdx);
	}

	public void clear() {
		userTaskRepository.deleteEntities(userTaskRepository.idSet());
	}

	public void userStartTaskWithStations(long userIdFrom, long usersCount,
			int stationsCount) {
		if (stationsCount > usersCount) {
			stationsCount = (int) usersCount;
		}
		Queue<Long>[] stationUserIds = new Queue[stationsCount];
		for (int i = 0; i < stationsCount; i++) {
			stationUserIds[i] = new LinkedList<Long>();
		}
		for (long i = 0; i < usersCount; i++) {
			int stationIdx = (int) (i % stationsCount);
			stationUserIds[stationIdx].add(userIdFrom + i);
		}
		for (Queue<Long> userIds : stationUserIds) {
			executorService.submit(() -> {
				for (Long userId : userIds) {
					startTask(userId);
					while (!checkTaskFinished(userId)) {
						try {
							Thread.sleep(100);
						} catch (Exception e) {
							e.printStackTrace();
						}
					}
				}
			});
		}
	}

	@Process
	public boolean checkTaskFinished(long userId) {
		UserTask userTask = userTaskRepository.findById(userId);
		if (userTask == null) {
			return true;
		}
		return userTask.getTask().isFinished();
	}
}
