package io.agora.avc.app.address

import android.content.res.Configuration
import android.os.Bundle
import android.text.Editable
import android.text.TextWatcher
import android.view.MotionEvent
import android.view.WindowManager
import android.view.inputmethod.EditorInfo
import androidx.appcompat.content.res.AppCompatResources
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.recyclerview.widget.DividerItemDecoration
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import io.agora.avc.R
import io.agora.avc.app.master.MainViewModel
import io.agora.avc.bo.LocalUser
import io.agora.avc.databinding.FragmentAddressBookSearchBinding
import io.agora.avc.extensions.getConferenceNicknameMaybeAssistant
import io.agora.avc.extensions.safeNavigate
import io.agora.avc.utils.KeyboardUtils
import io.agora.frame.base.NovelFragment

class AddressBookSearchFragment :
    NovelFragment<AddressBookViewModel, FragmentAddressBookSearchBinding>() {

    private var localUser: LocalUser? = null
    private var dept: GroupNode? = null
    private var operationType: Int = OPERATION_INVITE

    private val searchTextWatcher: TextWatcher = object : TextWatcher {
        override fun afterTextChanged(s: Editable?) {
            mViewModel?.searchContent(mBinding.etSearch.text.toString())
        }

        override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {
        }

        override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
        }
    }

    private val mainViewModel: MainViewModel by lazy {
        getViewModel(requireActivity().viewModelStore, MainViewModel::class.java)
    }

    private val addressBookListener: AddressBookListener = object : AddressBookListener {
        override fun onItemClicked(node: AddressBookNode, position: Int) {
            if (node is MemberNode && !node.isJoined) {
                mViewModel?.toggleMember(node, position)
                pageAdapter.notifyItemChanged(position)
                onBackPressed()
            }
        }

        override fun onToggleGroup(node: AddressBookNode, position: Int) {
            node.isChecked = !node.isChecked
        }

        override fun onRefreshClicked() {
            mViewModel?.searchContent(mBinding.etSearch.text.toString())
        }

        override fun onLongClicked(addressBookNode: AddressBookNode, position: Int) {
            //do nothing
        }

        override fun onCreateGroupClicked() {
            //do nothing
        }
    }

    private val pageAdapter by lazy {
        AddressBookAdapter(addressBookListener)
    }

    private var maxSizeLayoutManager: MaxSizeLayoutManager? = null

    private val selectedAdapter by lazy {
        UserSelectedAdapter().apply {
            setOnUserSelectedListener(object : UserSelectedAdapter.OnUserSelectedListener {
                override fun onItemClicked(node: MemberNode, position: Int) {
                    mViewModel?.uncheckMember(node)
                }
            })
        }
    }

    private val dividerItemDecoration by lazy {
        DividerItemDecoration(context, DividerItemDecoration.HORIZONTAL).also { decoration ->
            AppCompatResources.getDrawable(requireContext(), R.drawable.shape_user_selected_divider)
                ?.let {
                    decoration.setDrawable(it)
                }
        }
    }

    override fun getLayoutId(): Int {
        return R.layout.fragment_address_book_search
    }

    override fun allocObserver() {
        mViewModel?.deptLiveData?.observe(this) { dept ->
            this.dept = dept
            mBinding.actionBar.setTitle(dept.name)
        }
        mViewModel?.localUserLiveData?.observe(this) { user ->
            localUser = user
            mBinding.watermark.setMarkText(user.getConferenceNicknameMaybeAssistant())
        }
        mViewModel?.membersLiveData?.observe(this) { list ->
            pageAdapter.setNewData(list)
        }
        mViewModel?.selectedLiveData?.observe(this) { list ->
            selectedAdapter.setNewData(list)
            refreshSelectedNumber(list.size)
            invalidateCallable(list)
        }
    }

    private fun invalidateCallable(list: List<MemberNode>?) {
        val size = list?.size ?: 0
        adjustEditStyle(size)
    }

    private fun adjustEditStyle(size: Int) {
        mBinding.ivSearch.isVisible = size <= 0
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val operationType = arguments?.getInt(KEY_OPERATION_TYPE) ?: OPERATION_INVITE
        this.operationType = operationType
        mViewModel.setOperationType(operationType)
    }

    override fun initialize(savedInstanceState: Bundle?) {
        mBinding.rvUserList.adapter = pageAdapter
        mBinding.rvUserList.layoutManager = LinearLayoutManager(context)
        mBinding.rvUserList.setOnTouchListener { v, event ->
            if (event.action == MotionEvent.ACTION_DOWN && mBinding.etSearch.text.isNullOrEmpty()) {
                onBackPressed()
            }
            false
        }
        mBinding.rvUserList.addOnScrollListener(object : RecyclerView.OnScrollListener() {
            override fun onScrollStateChanged(recyclerView: RecyclerView, newState: Int) {
                if ((mainViewModel.softInputEvent.value ?: 0) > 0) {
                    hideKeyboard()
                }
            }
        })
        mBinding.rvSelected.adapter = selectedAdapter
        maxSizeLayoutManager = createLayoutManager()
        mBinding.rvSelected.layoutManager = maxSizeLayoutManager
        mBinding.rvSelected.addItemDecoration(dividerItemDecoration)
        dept?.name?.let {
            mBinding.actionBar.setTitle(it)
        }
        localUser?.let { user ->
            mBinding.watermark.setMarkText(user.getConferenceNicknameMaybeAssistant())
        }
        mBinding.etSearch.setOnEditorActionListener { v, actionId, event ->
            if (actionId == EditorInfo.IME_ACTION_SEARCH) {
                hideKeyboard()
                mViewModel?.searchContent(mBinding.etSearch.text.toString())
                true
            }
            false
        }
        mBinding.tvCount.setOnClickListener {
            safeNavigate(
                R.id.action_address_book_search_to_member_selected,
                bundleOf(KEY_OPERATION_TYPE to operationType)
            )
        }
        mBinding.etSearch.addTextChangedListener(searchTextWatcher)
    }

    private fun refreshSelectedNumber(size: Int) {
        if (size == 0) {
            mBinding.tvCount.isVisible = false
            return
        }
        mBinding.tvCount.isVisible = true
        if (size > MAX_VISIBLE_NUMBER) {
            mBinding.tvCount.text = OVER_VISIBLE_NUMBER_TEXT
        } else {
            mBinding.tvCount.text = "$size"
        }
    }

    override fun onStart() {
        super.onStart()
        activity?.window?.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_ALWAYS_VISIBLE or WindowManager.LayoutParams.SOFT_INPUT_ADJUST_RESIZE)
        KeyboardUtils.showSoftInput()
        mBinding.etSearch.requestFocus()
    }

    override fun onStop() {
        super.onStop()
        hideKeyboard()
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            maxSizeLayoutManager?.setMaxWidth(LANDSCAPE_MAX_WIDTH)
        } else {
            maxSizeLayoutManager?.setMaxWidth(PORTRAIT_MAX_WIDTH)
        }
    }

    private fun createLayoutManager(): MaxSizeLayoutManager {
        val maxWidth =
            if (resources.configuration.orientation == Configuration.ORIENTATION_PORTRAIT) {
                PORTRAIT_MAX_WIDTH
            } else {
                LANDSCAPE_MAX_WIDTH
            }
        return MaxSizeLayoutManager(context, RecyclerView.HORIZONTAL, false, maxWidth)
    }

    private fun hideKeyboard() {
        KeyboardUtils.hideSoftInput(mBinding.etSearch)
    }

    override fun onDestroyView() {
        mBinding.etSearch.setOnEditorActionListener(null)
        mBinding.etSearch.removeTextChangedListener(searchTextWatcher)
        super.onDestroyView()
    }

    override fun showLoading() {
        mBinding.progressBar.isVisible = true
    }

    override fun hideLoading() {
        mBinding.progressBar.isVisible = false
    }

    companion object {
        private const val TAG = "[UI][AddressBookSearch]"
        private const val LANDSCAPE_MAX_WIDTH = 540f
        private const val PORTRAIT_MAX_WIDTH = 260f
    }

}