package io.agora.avc.app.attendees

import android.content.res.Configuration
import android.graphics.Rect
import android.os.Bundle
import android.text.Editable
import android.text.TextWatcher
import android.view.Gravity
import android.view.View
import android.view.inputmethod.EditorInfo
import android.widget.FrameLayout
import android.widget.LinearLayout
import androidx.appcompat.app.AlertDialog
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.view.doOnPreDraw
import androidx.core.view.isVisible
import androidx.lifecycle.Observer
import androidx.recyclerview.widget.LinearLayoutManager
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import com.google.android.material.internal.CheckableImageButton
import com.google.android.material.tabs.TabLayout
import io.agora.avc.R
import io.agora.avc.app.meeting.MeetingFragment
import io.agora.avc.app.operation.OperationFragment
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.Room
import io.agora.avc.bo.RoomMode
import io.agora.avc.bo.UserSource
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.databinding.FragmentAttendeesBinding
import io.agora.avc.extensions.safeNavigate
import io.agora.avc.utils.*
import io.agora.avc.widget.divider.VerticalDivider
import io.agora.frame.base.BaseSheetDialogFragment
import io.agora.logger.Logger

class AttendeesFragment : BaseSheetDialogFragment<AttendeesViewModel, FragmentAttendeesBinding>() {

    private var roomMode: RoomMode? = null
    private var room: Room? = null
    private var localUser: LocalUser? = null
    private var confirmDialingDialog: AlertDialog? = null

    private val searchTextWatcher: TextWatcher = object : TextWatcher {
        override fun afterTextChanged(s: Editable?) {
            mViewModel?.searchAttendees(
                mBinding.tabLayout.selectedTabPosition,
                mBinding.etSearch.text.toString()
            )
        }

        override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {
        }

        override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
        }
    }

    private val dividerItemDecoration by lazy {
        VerticalDivider(
            context,
            R.drawable.shape_list_divider,
            ConvertUtils.dp2px(16f),
            ConvertUtils.dp2px(16f)
        )
    }

    private val pageAdapter by lazy {
        AttendeesAdapter()
    }

    private val bottomSheetCallback = object : BottomSheetBehavior.BottomSheetCallback() {
        override fun onStateChanged(bottomSheet: View, newState: Int) {
        }

        override fun onSlide(bottomSheet: View, slideOffset: Float) {
            activity?.let {
                if (KeyboardUtils.isSoftInputVisible(it)) {
                    hideKeyboard()
                }
            }
            resetInviteButtonRect(bottomSheet)
        }
    }

    override fun getLayoutId(): Int {
        return R.layout.fragment_attendees
    }

    override fun allocObserver() {
        mViewModel?.attendeesLiveData?.observe(this, Observer {
            pageAdapter.submitList(it)
        })

        mViewModel?.attendeesTypeLiveData?.observe(this, Observer { data ->
            setTabAll(data.allUser)
            setTabLogged(data.innerUser)
            setTabNonLogged(data.externalUser)
        })

        mViewModel?.roomModeChangedLiveData?.observe(this) {
            this.roomMode = it
            pageAdapter.setRoomModel(it)
            renderUI()
        }

        mViewModel?.roomChangedLiveData?.observe(this) {
            this.room = it
            setupInviteButtonStatus()
        }

        mViewModel?.localUserChangedLiveData?.observe(this) {
            this.localUser = it
            setupInviteButtonStatus()
        }
    }

    private fun renderUI() {
        mBinding.tabLayout.isVisible = this.roomMode == RoomMode.AGORA
    }

    override fun onStart() {
        super.onStart()
        mDialog.setCanceledOnTouchOutside(true)
        mBehavior.addBottomSheetCallback(bottomSheetCallback)
        onConfigurationChanged(resources.configuration)
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            switch2Landscape()
        } else {
            switch2Portrait()
        }
    }

    private fun switch2Portrait() {
        mDialog?.window?.let {
            BarUtils.setStatusBarVisibility(it, true)
        }
        val layoutParams = mBinding.container.layoutParams as FrameLayout.LayoutParams
        layoutParams.height = (ScreenUtils.getAppScreenHeight() * 0.9f).toInt()
        layoutParams.width = ScreenUtils.getScreenWidth()
        layoutParams.gravity = Gravity.LEFT
        mBinding.container.layoutParams = layoutParams
        mBehavior.isFitToContents = false
        mBehavior.skipCollapsed = false
        mBehavior.peekHeight = (ScreenUtils.getAppScreenHeight() * 0.6f).toInt()
        mBehavior.expandedOffset = (ScreenUtils.getScreenHeight() * 0.1f).toInt()
        mBehavior.halfExpandedRatio = 0.6f
        mBehavior.state = BottomSheetBehavior.STATE_HALF_EXPANDED
        mBinding.llFunction.doOnPreDraw {
            mBottomSheet?.let { bottomSheet ->
                resetInviteButtonRect(bottomSheet)
            }
        }
    }

    private fun switch2Landscape() {
        mDialog?.window?.let {
            BarUtils.setStatusBarVisibility(it, false)
        }
        val layoutParams = mBinding.container.layoutParams as FrameLayout.LayoutParams
        layoutParams.height = ScreenUtils.getAppScreenHeight() - ConvertUtils.dp2px(54f)
        layoutParams.width = ScreenUtils.getScreenWidth() / 2
        layoutParams.gravity = Gravity.RIGHT
        mBinding.container.layoutParams = layoutParams
        mBehavior.isFitToContents = true
        mBehavior.skipCollapsed = true
        mBehavior.state = BottomSheetBehavior.STATE_EXPANDED
        setInviteButtonBottomMargin(0)
    }

    override fun initialize(savedInstanceState: Bundle?) {
        adjustStartIconBounds()

        mBinding.rvUserList.adapter = pageAdapter
        mBinding.rvUserList.addItemDecoration(dividerItemDecoration)
        mBinding.rvUserList.layoutManager = LinearLayoutManager(context)

        mBinding.etSearch.addTextChangedListener(searchTextWatcher)

        pageAdapter.setIAttendeeClickListener(object : AttendeesAdapter.IAttendeeClickListener {
            override fun onAttendeeClicked(user: ARoomUser) {
                hideKeyboard()
                when {
                    !user.isAttendee -> {
                        if (user.isDialing) {
                            Logger.e(TAG, "can not open operation fragment,dialing the user")
                        } else {
                            showReCallConfirmDialog(user)
                        }
                    }
                    user.isAssistant && !user.online -> {//assistant leaving
                        ToastUtils.showShort(R.string.as_close_leave_info)
                        Logger.e(TAG, "can not open operation fragment,assistant is leaving")
                    }
                    mViewModel?.hasPermission(user) == true -> {
                        OperationFragment.navigateTo(
                            this@AttendeesFragment,
                            user,
                        )
                    }
                    else -> {
                        Logger.e(TAG, "can not open operation fragment,user has not permission")
                    }
                }
            }
        })

        mBinding.etSearch.setOnEditorActionListener { _, actionId, _ ->
            if (actionId == EditorInfo.IME_ACTION_SEARCH) {
                hideKeyboard()
            }
            true
        }

        mBinding.tilSearch.setEndIconOnClickListener {
            mBinding.etSearch.setText("")
        }

        mBinding.tabLayout.addOnTabSelectedListener(object : TabLayout.OnTabSelectedListener {
            override fun onTabSelected(tab: TabLayout.Tab?) {
                Logger.i(TAG, "selected tab:${tab?.position}")
                mViewModel?.searchAttendees(
                    tab?.position ?: 0,
                    mBinding.etSearch.text.toString()
                )
            }

            override fun onTabUnselected(tab: TabLayout.Tab?) {

            }

            override fun onTabReselected(tab: TabLayout.Tab?) {

            }
        })

        mBinding.btnInvite.setOnClickListener {
            safeNavigate(R.id.action_attendees_to_address_book_container)
        }

    }

    private fun adjustStartIconBounds() {
        val startIcon =
            mBinding.tilSearch.findViewById<CheckableImageButton>(R.id.text_input_start_icon)
        startIcon.minimumWidth = ConvertUtils.dp2px(28f)
        startIcon.minimumHeight = ConvertUtils.dp2px(28f)
        val layoutParams = startIcon.layoutParams as LinearLayout.LayoutParams
        layoutParams.rightMargin = 0
        layoutParams.marginStart = ConvertUtils.dp2px(16f)
    }

    private fun setTabAll(num: Int) {
        mBinding.tabLayout.getTabAt(0)?.text = "${getString(R.string.participants_all)}\n($num)"
    }

    private fun setTabLogged(num: Int) {
        mBinding.tabLayout.getTabAt(1)?.text = "${getString(R.string.participants_logged)}\n($num)"
    }

    private fun setTabNonLogged(num: Int) {
        mBinding.tabLayout.getTabAt(2)?.text =
            "${getString(R.string.participants_not_logged)}\n($num)"
    }

    private fun setupInviteButtonStatus() {
        mBinding.llFunction.isVisible =
            room?.isInternal() == true
                    && localUser?.isThirdPartyLoggedIn == true
                    && (localUser?.source == UserSource.AGORAN.value || localUser?.source == UserSource.EMAIL.value)
    }

    private fun resetInviteButtonRect(bottomSheet: View) {
        if (resources.configuration.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            return
        }
        val rect = Rect()
        bottomSheet.getGlobalVisibleRect(rect)
        Logger.d(TAG, "bottomSheet height:${bottomSheet.height},visible height:${rect.height()}")
        setInviteButtonBottomMargin(bottomSheet.height - rect.height())
    }

    private fun setInviteButtonBottomMargin(bottomMargin: Int) {
        val layoutParams = mBinding.llFunction.layoutParams
        if (layoutParams is ConstraintLayout.LayoutParams) {
            layoutParams.bottomMargin = bottomMargin
            mBinding.llFunction.layoutParams = layoutParams
        }
    }

    private fun showReCallConfirmDialog(user: ARoomUser) {
        if (confirmDialingDialog?.isShowing == true) {
            return
        }
        context?.let {
            confirmDialingDialog =
                MaterialAlertDialogBuilder(it, R.style.CustomMaterialAlertDialog)
                    .setTitle(
                        StringUtils.getString(
                            R.string.invite_second_title,
                            user.thirdPartyName
                        )
                    )
                    .setCancelable(false)
                    .setNegativeButton(R.string.cancel_button) { _, _ -> }
                    .setPositiveButton(R.string.terminate_button) { _, _ ->
                        mViewModel?.startCall(user)
                    }
                    .show()
        }
    }

    override fun onStop() {
        mBehavior.removeBottomSheetCallback(bottomSheetCallback)
        super.onStop()
    }

    override fun onDestroyView() {
        mBinding.etSearch.setOnEditorActionListener(null)
        mBinding.etSearch.removeTextChangedListener(searchTextWatcher)
        hideKeyboard()
        super.onDestroyView()
    }

    private fun hideKeyboard() {
        KeyboardUtils.hideSoftInput(mBinding.etSearch)
    }

    companion object {
        private const val TAG = "[UI][Attendees]"
    }

}
