package io.agora.avc.app.attendees

import android.app.Application
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.viewModelScope
import com.agora.valoran.Constants
import com.agora.valoran.bean.SimpleRoomUser
import io.agora.avc.R
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.PinyinInfo
import io.agora.avc.bo.Room
import io.agora.avc.bo.RoomMode
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.extensions.copy
import io.agora.avc.extensions.isMySelf
import io.agora.avc.manager.translation.TranslationManager
import io.agora.avc.repository.RoomRepository
import io.agora.avc.utils.RegexUtils
import io.agora.avc.utils.StringUtils
import io.agora.logger.Logger
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import javax.inject.Inject

class AttendeesViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val roomRepository: RoomRepository,
    private val translationManager: TranslationManager,
) : AppViewModel(application, appController) {
    val attendeesLiveData = MutableLiveData<List<ARoomUser>>()
    val attendeesTypeLiveData = MutableLiveData<AttendeesType>()
    val roomModeChangedLiveData = MutableLiveData<RoomMode?>()
    val roomChangedLiveData = MutableLiveData<Room?>()
    val localUserChangedLiveData = MutableLiveData<LocalUser?>()

    private var searchType: Int = 0

    private var searchKey: String = ""

    private val emptyUser by lazy {
        ARoomUser(streamId = -1)
    }

    override fun onResume() {
        super.onResume()
        queryLocalUser()
        queryRoom()
        queryRoomMode()
        queryAttendees()
    }

    fun startCall(user: ARoomUser) {
        Logger.i(TAG, "start call request")
        val source = arrayListOf<SimpleRoomUser>()
        source.add(SimpleRoomUser().apply {
            thirdPartyId = user.thirdPartyUid
            thirdPartyDepartment = user.thirdPartyDepartment
            thirdPartyName = user.thirdPartyName
            thirdPartyAlias = user.thirdPartyAlias
        })
        if (source.isNotEmpty()) {
            appController.inviteContacts(appController.getOperationSeqId(), source)
        }
    }

    fun hasPermission(operationUser: ARoomUser): Boolean {
        val hostUid = getRoom()?.hostUid
        val uid = getLocalUser()?.uid
        return operationUser.uid == uid || hostUid == null || hostUid.isEmpty() || hostUid == uid || operationUser.isAssistant && operationUser.online
    }

    private fun queryLocalUser() {
        localUserChangedLiveData.postValue(getLocalUser())
    }

    private fun queryRoom() {
        roomChangedLiveData.postValue(getRoom())
    }

    private fun queryRoomMode() {
        roomModeChangedLiveData.postValue(getRoomMode())
    }

    private fun queryAttendees() {
        viewModelScope.launch(Dispatchers.IO) {
            searchAttendees(searchType, searchKey)
        }
    }

    @Synchronized
    fun searchAttendees(type: Int, input: String) {
        if (searchType != type) {
            this.searchType = type
        }
        if (searchKey != input) {
            this.searchKey = input
        }
        val result = arrayListOf<ARoomUser>()
        var allUserCount: Int
        var internalUserCount = 0
        val users = roomRepository.getRoomUserList()
        allUserCount = users.size
        val enableAssistant = roomRepository.queryAssistState().let {
            it == Constants.AssistState.ENABLE_WITHOUT_ORIGIN_SOUND.value
                    || it == Constants.AssistState.ENABLE_WITH_ORIGIN_SOUND.value
        }
        users.forEach { user ->
            val standardUser = if (searchType == 0) {
                searchUser(input, user)
            } else if (searchType == 1 && user.isThirdPartyLoggedIn && user.isAttendee) {
                searchUser(input, user)
            } else if (searchType == 2 && !user.isThirdPartyLoggedIn && user.isAttendee) {
                searchUser(input, user)
            } else {
                null
            }

            standardUser?.showMedia =
                !(!user.isMySelf() && user.isAssistant && (!user.online || !enableAssistant))
            if (standardUser != null) {
                result.add(standardUser)
            }
            if (!user.isAttendee) {
                allUserCount--
            }
            if (user.isThirdPartyLoggedIn && user.isAttendee) {
                internalUserCount++
            }
        }
        //if the collection is empty and the search content is not empty, add an empty data
        if (result.isEmpty() && input.isNotEmpty()) {
            result.add(emptyUser)
        }
        attendeesLiveData.postValue(result)
        val externalUserCount = allUserCount.minus(internalUserCount)
        val attendeesType = AttendeesType(allUserCount, internalUserCount, externalUserCount)
        if (getRoomMode() == RoomMode.AGORA) {
            attendeesTypeLiveData.postValue(attendeesType)
        }
    }

    private fun searchUser(input: String, user: ARoomUser): ARoomUser? {
        if (input.trim().isEmpty()) {
            return if (user is LocalUser) {
                user.copy()
            } else {
                user.copy()
            }
        }
        return if (getRoomMode() == RoomMode.AGORA) {
            markAgoraUser(input, user)
        } else {
            markUserName(input, user)
        }
    }

    private fun markAgoraUser(input: String, user: ARoomUser): ARoomUser? {
        return if (user.isThirdPartyLoggedIn) {
            val tpnHighlight = matchUser(input, user.thirdPartyName ?: "")
            val copy = user.copy()
            if (tpnHighlight != null) {
                copy.thirdPartyNameStartPos = tpnHighlight[0]
                copy.thirdPartyNameEndPos = tpnHighlight[1]
            }
            val tpaHighlight = matchUser(input, user.thirdPartyAlias ?: "")
            if (tpaHighlight != null) {
                copy.thirdPartyAliasStartPos = tpaHighlight[0]
                copy.thirdPartyAliasEndPos = tpaHighlight[1]
            }
            var asHighlight: IntArray? = null
            if (user.isAssistant) {
                asHighlight = matchUser(input, StringUtils.getString(R.string.as_name))
            }
            if (tpnHighlight != null || tpaHighlight != null || asHighlight != null) {
                return copy
            }
            return null
        } else {
            markUserName(input, user)
        }
    }


    private fun markUserName(input: String, user: ARoomUser): ARoomUser? {
        val highlight = matchUser(input, user.name ?: "")
        val copy = user.copy()
        if (highlight != null) {
            copy.nameStartPos = highlight[0]
            copy.nameEndPos = highlight[1]
        }
        var asHighlight: IntArray? = null
        if (user.isAssistant) {
            asHighlight = matchUser(input, StringUtils.getString(R.string.as_name))
        }
        if (highlight != null || asHighlight != null) {
            return copy
        }
        return null
    }

    private fun matchUser(input: String, source: String): IntArray? {
        val searchInfo = translationManager.hanzi2Pinyin(input)
        val userInfo = translationManager.hanzi2Pinyin(source)
        val existChinese = translationManager.existChinese(input)
        var result = matchUserSource(searchInfo, userInfo)
        //If it is Chinese, use exact match
        if (existChinese || result != null) {
            return result
        }
        result = matchUserJP(searchInfo, userInfo)
        if (result != null) {
            return result
        }
        result = matchUserQP(searchInfo, userInfo)
        if (result != null) {
            return result
        }
        return null
    }

    private fun matchUserSource(searchInfo: PinyinInfo, userInfo: PinyinInfo): IntArray? {
        val startIndex = userInfo.source.indexOf(searchInfo.source)
        val endIndex: Int
        if (startIndex != -1) {
            endIndex = startIndex + searchInfo.source.length
            return intArrayOf(startIndex, endIndex)
        }
        return null
    }

    private fun matchUserQP(searchInfo: PinyinInfo, userInfo: PinyinInfo): IntArray? {
        if (searchInfo.quanPin.isEmpty()) {
            return null
        }
        val firstChat = searchInfo.quanPin[0]
        // If jp is not included, the first letter of the search content will not be used.
        if (!userInfo.jianPin.contains(firstChat)) {
            return null
        }
        // Mixed Chinese and English do not use QuanPin
        if (searchInfo.source.length > 1 && !RegexUtils.isEn(searchInfo.source)) {
            return null
        }
        val startIndex = userInfo.quanPin.indexOf(searchInfo.quanPin)
        val endIndex: Int
        if (startIndex != -1) {
            endIndex = startIndex + searchInfo.source.length
            var startPos = -1
            var endPos = -1
            var length = 0
            userInfo.number.forEachIndexed { index, num ->
                length += num
                if (startPos == -1 && startIndex < length) {
                    startPos = index
                }
                if (endPos == -1 && endIndex <= length) {
                    endPos = index + 1
                }
                if (startPos != -1 && endPos != -1) {
                    return intArrayOf(startPos, endPos)
                }
            }
        }
        return null
    }

    private fun matchUserJP(searchInfo: PinyinInfo, userInfo: PinyinInfo): IntArray? {
        if (!RegexUtils.isEn(searchInfo.source)) {
            return null
        }
        val startIndex = userInfo.jianPin.indexOf(searchInfo.jianPin)
        val endIndex: Int
        if (startIndex != -1) {
            endIndex = startIndex + searchInfo.source.length
            return intArrayOf(startIndex, endIndex)
        }
        return null
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.USER_INFO_CHANGED.ordinal,
            AppEvent.USER_LIST_CHANGED.ordinal,
            AppEvent.ROOM_ASSISTANT_CHANGED.ordinal,
            AppEvent.LOCAL_ASSISTANT_STATE_CHANGED.ordinal,
            AppEvent.USER_LIST_COUNT_CHANGED.ordinal -> {
                queryAttendees()
            }
        }
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.USER_LIST_CHANGED,
            AppEvent.USER_INFO_CHANGED,
            AppEvent.ROOM_ASSISTANT_CHANGED,
            AppEvent.LOCAL_ASSISTANT_STATE_CHANGED,
            AppEvent.USER_LIST_COUNT_CHANGED,
        )
    }

    private fun getRoomMode(): RoomMode {
        return if (getRoom()?.isInternal() == true) {
            RoomMode.AGORA
        } else {
            RoomMode.NORMAL
        }
    }

    companion object {
        private const val TAG = "[VM][Attendees]"
    }

}