package io.agora.avc.app.audio

import android.app.Application
import androidx.lifecycle.MutableLiveData
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.Room
import io.agora.avc.repository.RoomRepository
import io.agora.avc.utils.TimeUtils
import io.agora.frame.base.livedata.EventLiveData
import io.agora.frame.base.livedata.SingleLiveEvent
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import java.util.concurrent.TimeUnit
import javax.inject.Inject

class AudioViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val roomRepository: RoomRepository
) : AppViewModel(application, appController) {
    val meetingDurationEvent: SingleLiveEvent<String> = SingleLiveEvent()
    val localUserStatusChanged: MutableLiveData<LocalUser> = SingleLiveEvent()
    val bizConnectChangedEvent = EventLiveData<Boolean>()
    val roomInfoChangedEvent = EventLiveData<Room?>()

    private var countTimer: Disposable? = null

    override fun onCreate() {
        super.onCreate()
    }

    override fun onResume() {
        super.onResume()
        notifyBizConnectChanged()
        queryLocalUser()
        queryRoom()
    }

    private fun notifyBizConnectChanged() {
        bizConnectChangedEvent.postValue(appController.bizConnected())
    }

    private fun queryRoom() {
        roomInfoChangedEvent.postValue(roomRepository.getRoom())
    }


    fun startCountTime() {
        Logger.i(TAG, "start counting room elapse")
        Observable
            .interval(0, 1, TimeUnit.SECONDS)
            .subscribeOn(Schedulers.io())
            .subscribe(object : io.reactivex.Observer<Long> {
                override fun onComplete() {

                }

                override fun onSubscribe(d: Disposable) {
                    countTimer = d
                }

                override fun onNext(t: Long) {
                    val elapse = roomRepository.getRoom()?.elapse
                    val elapsedRealTime = TimeUtils.elapsedRealTime()
                    if (elapse != null && elapse > 0) {
                        val time = TimeUtils.getRoomTime(elapsedRealTime - elapse)
                        meetingDurationEvent.postValue(time)
                    }
                }

                override fun onError(e: Throwable) {
                }
            })
    }

    fun stopCountTime() {
        Logger.w(TAG, "stop counting room elapse")
        countTimer?.dispose()
    }

    fun quitRoom() {
        appController?.leaveRoom()
    }

    fun changeLocalAudioStatus(on: Boolean) {
        appController?.setLocalAudio(!on)
    }

    private fun queryLocalUser() {
        getLocalUser()?.let { user ->
            localUserStatusChanged.postValue(user)
        }
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.LOCAL_USER_CHANGED.ordinal -> {
                queryLocalUser()
            }
        }
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.LOCAL_USER_CHANGED,
            AppEvent.BIZ_STATUS_CHANGED,
        )
    }

    companion object {
        private const val TAG = "[VM][Audio]"
    }

}