package io.agora.avc.app.chat

import android.content.res.Configuration
import android.graphics.Rect
import android.os.Bundle
import android.view.Gravity
import android.view.View
import android.view.WindowManager
import android.widget.FrameLayout
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.lifecycle.Observer
import androidx.recyclerview.widget.LinearLayoutManager
import com.google.android.material.bottomsheet.BottomSheetBehavior
import io.agora.avc.R
import io.agora.avc.bo.Message
import io.agora.avc.databinding.AgoraFragmentChatBinding
import io.agora.avc.extensions.safeNavigate
import io.agora.avc.utils.BarUtils
import io.agora.avc.utils.ConvertUtils
import io.agora.avc.utils.ScreenUtils
import io.agora.frame.base.BaseSheetDialogFragment
import io.agora.logger.Logger

class ChatFragment : BaseSheetDialogFragment<ChatViewModel, AgoraFragmentChatBinding>() {

    private val adapter by lazy {
        ChatAdapter()
    }

    private val bottomSheetCallback = object : BottomSheetBehavior.BottomSheetCallback() {
        override fun onStateChanged(bottomSheet: View, newState: Int) {

        }

        override fun onSlide(bottomSheet: View, slideOffset: Float) {
            resetChatInputRect(bottomSheet)
        }
    }

    private fun resetChatInputRect(bottomSheet: View) {
        if (resources.configuration.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            return
        }
        val rect = Rect()
        bottomSheet.getGlobalVisibleRect(rect)
        Logger.d(TAG, "bottomSheet height:${bottomSheet.height},visible height:${rect.height()}")
        setChatInputBottomMargin(bottomSheet.height - rect.height())
    }

    private fun setChatInputBottomMargin(bottomMargin: Int) {
        val layoutParams = mBinding.llChatInput.layoutParams
        if (layoutParams is ConstraintLayout.LayoutParams) {
            layoutParams.bottomMargin = bottomMargin
            mBinding.llChatInput.layoutParams = layoutParams
        }
    }

    override fun getLayoutId(): Int {
        return R.layout.agora_fragment_chat
    }

    override fun allocObserver() {
        mViewModel?.messageListLiveData?.observe(this, Observer {
            adapter.setNewData(it)
            scrollToEnd(false)
        })

        mViewModel?.messageStatusChangedLiveData?.observe(this, Observer { data ->
            val index = data["index"] as? Int ?: return@Observer
            val message = data["message"] as? Message ?: return@Observer
            adapter.setData(index, message)
        })

        mViewModel?.chatMessageReceivedLiveData?.observe(this, Observer { data ->
            adapter.addData(data)
            scrollToEnd()
        })
    }

    override fun initialize(savedInstanceState: Bundle?) {
        mBinding.rvChatList.adapter = adapter
        mBinding.rvChatList.layoutManager = LinearLayoutManager(context)
        adapter.setItemClickListener(object : ChatAdapter.OnItemListener {
            override fun onClicked(id: Int) {
                mViewModel?.resendMessage(id)
            }
        })
        mBinding.tvInput.setTextIsSelectable(false)
        mBinding.tvInput.isLongClickable = false
        mBinding.tvInput.setOnClickListener {
            safeNavigate(R.id.chatInput)
        }
    }

    override fun onStart() {
        super.onStart()
        mDialog.window?.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_NOTHING)
        mDialog.setCanceledOnTouchOutside(true)
        mBehavior.addBottomSheetCallback(bottomSheetCallback)
        onConfigurationChanged(resources.configuration)
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            switch2Landscape()
        } else {
            switch2Portrait()
        }
    }

    private fun switch2Portrait() {
        mDialog?.window?.let {
            BarUtils.setStatusBarVisibility(it, true)
        }
        val layoutParams = mBinding.container.layoutParams as FrameLayout.LayoutParams
        layoutParams.height = (ScreenUtils.getAppScreenHeight() * 0.9f).toInt()
        layoutParams.width = ScreenUtils.getScreenWidth()
        layoutParams.gravity = Gravity.LEFT
        mBinding.container.layoutParams = layoutParams
        mBehavior.isFitToContents = false
        mBehavior.skipCollapsed = false
        mBehavior.peekHeight = (ScreenUtils.getAppScreenHeight() * 0.6f).toInt()
        mBehavior.expandedOffset = (ScreenUtils.getScreenHeight() * 0.1f).toInt()
        mBehavior.halfExpandedRatio = 0.6f
        mBehavior.state = BottomSheetBehavior.STATE_HALF_EXPANDED
        mBinding.llChatInput.post {
            if (mBinding == null) {
                return@post
            }
            mBottomSheet?.let { bottomSheet ->
                resetChatInputRect(bottomSheet)
            }
        }
    }

    private fun switch2Landscape() {
        mDialog?.window?.let {
            BarUtils.setStatusBarVisibility(it, false)
        }
        val layoutParams = mBinding.container.layoutParams as FrameLayout.LayoutParams
        layoutParams.height = ScreenUtils.getAppScreenHeight() - ConvertUtils.dp2px(54f)
        layoutParams.width = ScreenUtils.getScreenWidth() / 2
        layoutParams.gravity = Gravity.RIGHT
        mBinding.container.layoutParams = layoutParams
        mBehavior.isFitToContents = true
        mBehavior.skipCollapsed = true
        mBehavior.state = BottomSheetBehavior.STATE_EXPANDED
        setChatInputBottomMargin(0)
    }

    private fun scrollToEnd(animate: Boolean = true) {
        mBinding.rvChatList.post {
            if (mBinding == null) {
                return@post
            }
            (adapter.itemCount - 1).let {
                if (it > 0) {
                    if (animate) {
                        mBinding.rvChatList.smoothScrollToPosition(it)
                    } else {
                        mBinding.rvChatList.scrollToPosition(it)
                    }
                }
            }
        }
    }

    override fun onStop() {
        mViewModel?.remarkAllMessageRead()
        mBehavior.removeBottomSheetCallback(bottomSheetCallback)
        super.onStop()
    }

    companion object {
        private const val TAG = "[UI][Chat]"
    }

}
