package io.agora.avc.app.developer

import android.app.ActivityManager
import android.app.Application
import android.content.Context
import androidx.lifecycle.MutableLiveData
import io.agora.avc.MyApplication
import io.agora.avc.R
import io.agora.avc.auth.signin.SignInManager
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.DevConfig
import io.agora.avc.bo.UserInfo
import io.agora.avc.manager.splite.SPLiteProxy
import io.agora.avc.repository.AppConfigRepository
import io.agora.avc.repository.DeveloperRepository
import io.agora.avc.utils.AppUtils
import io.agora.avc.utils.GsonUtils
import io.agora.avc.utils.ToastUtils
import io.agora.logger.LogConverter
import io.agora.logger.Logger
import javax.inject.Inject

class DeveloperViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val appConfigRepository: AppConfigRepository,
    private val developerRepository: DeveloperRepository,
    private val signInManager: SignInManager,
    private val appEventBus: AppEventBus,
) : AppViewModel(application, appController) {

    val environmentChanged = MutableLiveData<Boolean>()
    val modelChangedEvent = MutableLiveData<DeveloperOptions>()
    val devConfigChangedEvent = MutableLiveData<DevConfig>()

    override fun onCreate() {
        super.onCreate()
    }

    override fun onResume() {
        super.onResume()
        environmentChanged.postValue(appConfigRepository.isTestServer())
        modelChangedEvent.postValue(developerRepository.getDeveloperOptions())
        devConfigChangedEvent.postValue(appConfigRepository.getDevConfig())
    }

    fun changeAppEnvironment() {
        Logger.i(TAG, "change app environment")
        appConfigRepository.saveTestServer(SPLiteProxy.isTestServer().not())
        appConfigRepository.clearToken()
        appConfigRepository.clearAppTips()
        killChildProcess()
        AppUtils.relaunchApp(true)
    }

    private fun killChildProcess() {
        Logger.i(TAG, "call kill child process")
        val systemService =
            getApplication<MyApplication>().getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        for (runningAppProcess in systemService.runningAppProcesses) {
            if (runningAppProcess.pid != android.os.Process.myPid()) {
                Logger.i(TAG, "kill process:${runningAppProcess.processName}")
                android.os.Process.killProcess(runningAppProcess.pid)
            }
        }
    }

    fun changeRtmConnected(disconnected: Boolean?) {
        Logger.i(TAG, "set rtm ${LogConverter.connected(disconnected?.not() ?: true)}")
        developerRepository.rtmEnable(disconnected?.not() ?: true)
        if (disconnected == true) {
            appController.connectRtmLink(false)
        } else {
            appController.connectRtmLink(true)
        }
    }

    fun changeUseMeta(useless: Boolean?) {
        Logger.i(TAG, "set rtc metadata ${LogConverter.enable(useless?.not() ?: true)}")
        developerRepository.metaDataEnable(useless?.not() ?: true)
        if (useless == true) {
            appController.useMetaData(false)
        } else {
            appController.useMetaData(true)
        }
    }

    fun changeUseDataChannel(useless: Boolean?) {
        Logger.i(TAG, "set rtc dataChannel ${LogConverter.enable(useless?.not() ?: true)}")
        developerRepository.dataChannelEnable(useless?.not() ?: true)
        if (useless == true) {
            appController.useDataChannel(false)
        } else {
            appController.useDataChannel(true)
        }
    }

    /**
     * do nothing,due to min version support secure
     */
    fun changeRoomSecure(secure: Boolean?) {
        Logger.i(TAG, "set room secure ${LogConverter.enable(secure ?: false)}")

    }

    fun testJavaCrash() {
        Logger.e(TAG, "test java crash!")
//        CrashReport.testJavaCrash()
    }

    fun changeLastMileTest(useless: Boolean?) {
        Logger.i(TAG, "set last mile test ${LogConverter.enable(useless?.not() ?: true)}")
        developerRepository.lastMileEnable(useless?.not() ?: true)
    }

    fun setVideoDataVisible(visible: Boolean?) {
        Logger.i(TAG, "set video stats ${LogConverter.visible(visible ?: false)}")
        developerRepository.showVideoData(visible ?: false)
        appController.showMediaStatics(visible ?: false)
        postOptions()
    }

    fun setVideoRateVisible(visible: Boolean?) {
        Logger.i(TAG, "set video rate ${LogConverter.visible(visible ?: false)}")
        developerRepository.videoScoreEnable(visible ?: false)
    }

    fun changeMediaBackground(it: Boolean?) {
        Logger.i(TAG, "change media background color")
        developerRepository.changeMainScreenVideoBG(it ?: false)
    }

    fun loginWeWork(login: Boolean?) {
        Logger.i(TAG, "Log in to WeWork automatically,login:$login")
        val userInfo = GsonUtils.fromJson(SPLiteProxy.getUserInfo(), UserInfo::class.java)
        if (userInfo == null) {
            ToastUtils.showShort(getApplication<MyApplication>().getString(R.string.media_token_error))
            return
        }
        developerRepository.loginWeWork(login ?: false)
        if (login == true) {
            signInManager.signInByTester()
        } else {
            signInManager.signOutByTester()
        }
    }

    fun setQualityReportVisible(visible: Boolean?) {
        Logger.i(TAG, "set quality report ${LogConverter.visible(visible ?: false)}")
        developerRepository.qualityReport(visible ?: false)
    }

    fun changeWifiAP(opened: Boolean?) {
        Logger.i(TAG, "set wifiAp opened:$opened")
        appConfigRepository.getDevConfig().apply {
            wifiAp = opened ?: false
            appConfigRepository.saveDevConfig(this)
        }
        appController.setWifiAp(opened ?: false)
    }

    fun postOptions() {
        appEventBus.notifyObservers(
            MessageEvent(
                AppEvent.DEVELOPER_OPTIONS_EVENT.ordinal,
                developerRepository.getDeveloperOptions()
            )
        )
    }

    fun openNetworkEvaluation(it: Boolean?) {
        Logger.i(TAG, "change network evaluation:$it")
        developerRepository.openNetworkEvaluation(it ?: false)
        appController.openNetworkEvaluation(it ?: false)
    }

    companion object {
        private const val TAG = "[VM][Developer]"
    }
}