package io.agora.avc.app.group

import android.os.Bundle
import androidx.appcompat.app.AlertDialog
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.navigation.fragment.findNavController
import androidx.recyclerview.widget.LinearLayoutManager
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.R
import io.agora.avc.app.address.*
import io.agora.avc.bo.LocalUser
import io.agora.avc.databinding.FragmentGroupTailoredBinding
import io.agora.avc.extensions.getConferenceNicknameMaybeAssistant
import io.agora.avc.extensions.safeNavigate
import io.agora.avc.utils.StringUtils
import io.agora.avc.utils.ToastUtils
import io.agora.avc.widget.OnActionTxtClickListener
import io.agora.frame.base.HolderFragment

const val KEY_GROUP_ID = "key_group_id"
const val KEY_GROUP_NAME = "key_group_name"

class GroupTailoredFragment :
    HolderFragment<GroupTailoredViewModel, FragmentGroupTailoredBinding>() {

    private var localUser: LocalUser? = null
    private var dept: GroupNode? = null
    private var callConfirmDialog: AlertDialog? = null
    private var groupId: String? = null
    private var contactGroup: ContactGroup? = null

    private val pageAdapter by lazy {
        GroupTailoredAdapter {
            this.groupId?.let {
                mViewModel.queryGroupDetails(it)
            }
        }
    }

    override fun getLayoutId(): Int {
        return R.layout.fragment_group_tailored
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        arguments?.getString(KEY_GROUP_ID)?.let {
            this.groupId = it
            mViewModel?.queryGroupDetails(it)
        }
    }

    override fun initialize(savedInstanceState: Bundle?) {
        mBinding.actionBar.setTitle(
            arguments?.getString(KEY_GROUP_NAME) ?: StringUtils.getString(R.string.group_default)
        )
        mBinding.rvUserList.adapter = pageAdapter
        mBinding.rvUserList.layoutManager = LinearLayoutManager(context)
        dept?.name?.let {
            mBinding.actionBar.setTitle(it)
        }
        localUser?.let { user ->
            mBinding.watermark.setMarkText(user.getConferenceNicknameMaybeAssistant())
        }
        mBinding.tvName.setOnClickListener {
            NameEditFragment.navigateTo(
                fragment = this,
                groupId = mViewModel.groupId,
                name = mBinding.tvName.text?.toString(),
            )
        }
        mBinding.tvAdd.setOnClickListener {
            if (!isGetDataError()) {
                mViewModel?.onAddClicked()
                safeNavigate(
                    R.id.action_group_tailored_to_address_book_container, bundleOf(
                        KEY_DEPT_ID to ROOT_DEPT_NO,
                        KEY_OPERATION_TYPE to OPERATION_ADD,
                        KEY_DATA_TYPE to DATA_ORG
                    )
                )
            } else {
                ToastUtils.showShort(R.string.invite_alert)
            }
        }
        mBinding.actionBar.setOnActionTxtClickListener(object : OnActionTxtClickListener {
            override fun onActionTxtClicked() {
                doOnActionTxtClicked()
            }
        })
    }

    override fun allocObserver() {
        mViewModel?.localUserLiveData?.observe(this) { user ->
            localUser = user
            mBinding.watermark.setMarkText(user.getConferenceNicknameMaybeAssistant())
        }
        mViewModel?.membersLiveData?.observe(this) { list ->
            pageAdapter.setNewData(list)
        }
        mViewModel?.groupNameLiveData?.observe(this) {
            mBinding.actionBar.setTitle(it)
            mBinding.tvName.text = it
        }
        mViewModel?.exitLiveData?.observe(this) {
            findNavController().navigateUp()
        }
        mViewModel?.contactGroupLiveData?.observe(this) {
            contactGroup = it
        }
    }

    private fun doOnActionTxtClicked() {
        mBinding.tvName.text?.toString().let {
            val groupName = if (StringUtils.isEmpty(mBinding.tvName.text?.toString())) {
                StringUtils.getString(R.string.group_default)
            } else {
                it!!
            }
            when {
                mViewModel.membersIsEmpty() -> {
                    ToastUtils.showShort(R.string.group_member_null)
                    return
                }
                !mViewModel.isChanged() -> {
                    onBackPressed()
                }
                else -> {
                    mViewModel.submit(groupName)
                }
            }
        }
    }

    override fun onBackPressed() {
        if (mViewModel.isChanged()) {
            MaterialAlertDialogBuilder(requireContext(), R.style.CustomMaterialAlertDialog)
                .setTitle(R.string.group_quit_ask_title)
                .setNegativeButton(R.string.group_quit_ask_button_no) { _, _ ->
                    super.onBackPressed()
                }
                .setPositiveButton(R.string.group_save_mobile) { _, _ ->
                    doOnActionTxtClicked()
                }
                .show()
        } else {
            super.onBackPressed()
        }
    }

    override fun showLoading() {
        mBinding.progressBar.isVisible = true
    }

    override fun hideLoading() {
        mBinding.progressBar.isVisible = false
    }

    private fun isGetDataError() = contactGroup == null && !StringUtils.isEmpty(groupId)

    override fun onDestroyView() {
        callConfirmDialog?.let {
            if (it.isShowing) {
                it.dismiss()
            }
        }
        super.onDestroyView()
    }

    companion object {
        private const val TAG = "[UI][GroupTailored]"
    }

}