package io.agora.avc.app.log

import android.os.Bundle
import androidx.lifecycle.Observer
import androidx.navigation.fragment.NavHostFragment
import androidx.recyclerview.widget.LinearLayoutManager
import io.agora.avc.R
import io.agora.avc.databinding.FragmentLogBinding
import io.agora.avc.widget.LoadingDialog
import io.agora.frame.base.NovelFragment
import io.agora.frame.base.livedata.StatusEvent
import io.agora.logger.Log
import io.agora.logger.LogToMemory
import io.agora.logger.Logger

class LogFragment : NovelFragment<LogViewModel, FragmentLogBinding>() {

    private var logFilter: Int = Log.LOG_LEVEL_WARNING or Log.LOG_LEVEL_ERROR

    private val adapter: LogAdapter by lazy { LogAdapter(requireContext()) }

    private val loadingView by lazy { LoadingDialog(requireContext()) }

    private var isTouched: Boolean = false

    private var logListener = object : LogToMemory.LogListener {
        override fun newLog(type: Int) {
            mViewModel?.queryLog(logFilter, false)
        }
    }

    override fun getLayoutId(): Int {
        return R.layout.fragment_log
    }

    override fun allocObserver() {
        mViewModel?.logLiveData?.observe(this, Observer {
            adapter.submitList(it)
            scrollToEnd()
        })

        mViewModel?.statusEvent?.observe(this, Observer {
            when (it) {
                StatusEvent.Status.LOADING -> {
                    loadingView.show()
                }
                StatusEvent.Status.SUCCESS -> {
                    loadingView.dismiss()
                }
                StatusEvent.Status.ERROR -> {
                    loadingView.dismiss()
                }
                else -> {
                }
            }
        })
    }

    override fun initialize(savedInstanceState: Bundle?) {
        mBinding.toolBar.setNavigationOnClickListener {
            NavHostFragment.findNavController(this).navigateUp()
        }

        mBinding.toolBar.inflateMenu(R.menu.developer_log_menu)

        mBinding.toolBar.menu.findItem(R.id.info)?.isChecked =
            logFilter and Log.LOG_LEVEL_INFO != 0
        mBinding.toolBar.menu.findItem(R.id.warning)?.isCheckable =
            logFilter and Log.LOG_LEVEL_WARNING != 0
        mBinding.toolBar.menu.findItem(R.id.error)?.isCheckable =
            logFilter and Log.LOG_LEVEL_ERROR != 0

        mBinding.toolBar.setOnMenuItemClickListener {
            it.isChecked = !it.isChecked

            when (it.itemId) {
                R.id.info -> {
                    logFilter =
                        logFilter and Log.LOG_LEVEL_INFO.inv() or if (it.isChecked) Log.LOG_LEVEL_INFO else 0
                }
                R.id.warning -> {
                    logFilter =
                        logFilter and Log.LOG_LEVEL_WARNING.inv() or if (it.isChecked) Log.LOG_LEVEL_WARNING else 0
                }
                R.id.error -> {
                    logFilter =
                        logFilter and Log.LOG_LEVEL_ERROR.inv() or if (it.isChecked) Log.LOG_LEVEL_ERROR else 0
                }
            }

            mViewModel?.queryLog(logFilter)

            true
        }

        mBinding.recyclerView.layoutManager = LinearLayoutManager(context)
        mBinding.recyclerView.adapter = adapter
        Logger.setLogListener(logListener)
        mBinding.recyclerView.setOnTouchListener { _, _ ->
            isTouched = true
            false
        }

        mViewModel?.queryLog(logFilter)
    }

    private fun scrollToEnd() {
        if (adapter.itemCount - 1 > 0 && !isTouched) {
            mBinding.recyclerView.postDelayed({
                mBinding.recyclerView.scrollToPosition(adapter.itemCount - 1)
            }, 200)
        }
    }

    override fun onDestroyView() {
        super.onDestroyView()
        Logger.setLogListener(null)
        if (loadingView.isShowing) {
            loadingView.dismiss()
        }
    }
}
