package io.agora.avc.app.master

import android.content.*
import android.content.res.Configuration
import android.os.Build
import android.os.Bundle
import android.view.KeyEvent
import androidx.appcompat.app.AlertDialog
import androidx.core.content.ContextCompat
import androidx.core.os.bundleOf
import androidx.databinding.ViewDataBinding
import androidx.fragment.app.Fragment
import androidx.lifecycle.Observer
import androidx.navigation.Navigation
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.R
import io.agora.avc.app.calling.IncomingFragment
import io.agora.avc.app.meeting.MeetingFragment
import io.agora.avc.app.web.KEY_URL
import io.agora.avc.app.web.getOAuthUrl
import io.agora.avc.biz.AVCTelephonyWatcher
import io.agora.avc.bo.*
import io.agora.avc.extensions.getConferenceNicknameMaybeAssistant
import io.agora.avc.manager.bugreport.DumpStatus
import io.agora.avc.manager.notice.NoticeEvent
import io.agora.avc.manager.notice.NoticeExecutor
import io.agora.avc.manager.notice.NotificationExecutor
import io.agora.avc.manager.rating.video.VideoScore
import io.agora.avc.manager.splite.SPLiteProxy
import io.agora.avc.net.constant.Constants
import io.agora.avc.receiver.MediaRequestReceiver
import io.agora.avc.receiver.ScreenStatusReceiver
import io.agora.avc.receiver.VolumeBroadcastReceiver
import io.agora.avc.service.DaemonService
import io.agora.avc.utils.*
import io.agora.avc.widget.*
import io.agora.frame.base.BaseActivity
import io.agora.logger.Logger
import kotlin.reflect.KClass


class AgoraActivity : BaseActivity<MainViewModel, ViewDataBinding>() {

    private val telephonyWatcher: AVCTelephonyWatcher by lazy {
        AVCTelephonyWatcher(context.applicationContext)
    }

    private val screenStatusMonitor: ScreenStatusReceiver by lazy {
        ScreenStatusReceiver()
    }

    private val volumeBroadcastReceiver by lazy {
        VolumeBroadcastReceiver().apply {
            volumeChangeListener = {
                mViewModel.systemVolumeChange(it)
                volumeBootHelper.onVolumeChange(mViewModel.isInRoom(), it)
            }
        }
    }

    private var videoRatingDialog: AlertDialog? = null

    private var appUpgradeDialog: AppUpgradeDialog? = null

    private var permissionSettingDialog: AlertDialog? = null

    private var qualityReportDialog: QualityReportDialog? = null

    private var shareLinkConfirmDialog: AlertDialog? = null

    private var wwAuthLoginDialog: AlertDialog? = null

    private var errorAlertDialog: AlertDialog? = null

    private var uploadingDialog: UploadingDialog? = null

    private val fragmentTracker: FragmentTracker by lazy {
        FragmentTracker()
    }

    lateinit var noticeExecutor: NoticeExecutor

    private var localUser: LocalUser? = null

    private var roomInfo: Room? = null

    private var needWakeBugReport: Boolean = false

    //It is forbidden to use lambda because the old Activity onDestroy triggers the timing.
    // After the new Activity, this will cause an Observer to be associated with a different lifecycle
    private val toastObserver: Observer<NoticeEvent?> = Observer<NoticeEvent?> {
        it?.let { _noticeEvent ->
            noticeExecutor.showNotice(_noticeEvent)
        }
    }

    private val volumeBootHelper by lazy {
        VolumeBootHelper(this).apply {
            onVolumeBoost = {
                mViewModel.adjustRemoteAudio(it)
            }
        }
    }

    private val privacyTermsDialog by lazy {
        PrivacyTermsDialog(context).apply {
            setPrivacyTermsDialogListener(object : PrivacyTermsDialog.OnPrivacyTermsDialogListener {
                override fun onConfirmClick() {
                    mViewModel?.agreePrivacy()
                }

                override fun onCancelClick() {
                    finish()
                }
            })
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        setRoomTheme(SPLiteProxy.getRoomMode(), false)
        super.onCreate(savedInstanceState)
        BarUtils.transparentStatusBar(this)
        noticeExecutor = NotificationExecutor(this)
        ToastUtils.getDefaultMaker().setNotUseSystemToast()
    }

    override fun getLayoutId(): Int {
        return R.layout.activity_agora
    }

    override fun allocObserver() {
        mViewModel?.appUpgradeEvent?.observe(this, Observer {
            showUpgrade(it)
        })

        mViewModel?.appNoticeEvent?.observe(this) { notice ->
            showNotice(notice)
        }

        mViewModel?.roomChangedConfirmLiveData?.observe(this, Observer {
            this.roomInfo = it
            if (it?.name != null) {
                val message = if (it.isInternal()) {
                    if (localUser?.isCloudRecording == true) {
                        ClipboardUtils.copyText(Constants.getAtlasRecordingUrl())
                        getString(R.string.cloud_recording_leave_join_agora, it.name)
                    } else {
                        getString(R.string.room_leave_join_agora, it.name)
                    }
                } else {
                    if (localUser?.isCloudRecording == true) {
                        ClipboardUtils.copyText(Constants.getAtlasRecordingUrl())
                        getString(R.string.cloud_recording_leave_join_public, it.name)
                    } else {
                        getString(R.string.room_leave_join_public, it.name)
                    }
                }

                MaterialAlertDialogBuilder(this, R.style.CustomMaterialAlertDialog)
                    .setTitle(R.string.notice_change_room_title)
                    .setMessage(message)
                    .setNegativeButton(R.string.notice_change_room_cancel) { _, _ ->
                        Logger.i(TAG, "My cancel switch room ${it.name}")
                    }
                    .setPositiveButton(R.string.notice_change_room_confirm) { _, _ ->
                        Logger.i(TAG, "My switch room to ${it.name}")
                        mViewModel?.changeRoom(it)
                    }
                    .create()
                    .show()
            }
        })

        mViewModel?.localUserChangedLiveData?.observe(this) {
            this.localUser = it
        }

        mViewModel?.exitRoomLiveData?.observe(this, Observer {
            volumeBootHelper.reset()
            exitRoom(it)
        })

        mViewModel?.loginRequestEvent?.observe(this, Observer {
            MaterialAlertDialogBuilder(this, R.style.CustomMaterialAlertDialog)
                .setTitle(R.string.notice_change_room_permission_title)
                .setMessage(getString(R.string.notice_change_room_permission_message))
                .setNegativeButton(R.string.notice_change_room_permission_cancel) { _, _ ->

                }
                .setPositiveButton(R.string.notice_change_room_permission_confirm) { _, _ ->
                    if (roomInfo != null) {
                        mViewModel?.forbidQualityReport()
                        mViewModel?.quitRoom()
                    }
                    showLoginDialog(COMPANY_AGORA)
                }
                .create()
                .show()
        })

        mViewModel?.videoRatingEvent?.observe(this, {
            showVideoRatingDialog(it)
        })

        mViewModel?.toastLiveData?.observe(this, toastObserver)

        mViewModel?.dialogLiveData?.observe(this) {
            it?.let { _noticeEvent ->
                _noticeEvent.code.stringsId?.let { _stringsId ->
                    showNotice(getString(_stringsId, _noticeEvent.obj))
                }
            }
        }

        mViewModel?.userVideoSelectEvent?.observe(this, Observer { users ->
            val names = arrayOfNulls<String>(users.size)
            users.forEachIndexed { index, user ->
                names[index] = "${user.getConferenceNicknameMaybeAssistant()}(${user.streamId})"
            }
            var index = 0
            MaterialAlertDialogBuilder(context, R.style.CustomMaterialAlertDialog)
                .setTitle(R.string.video_rating_choose_title)
                .setPositiveButton(R.string.confirm) { _, _ ->
                    mViewModel?.startVideoRating(users[index].streamId)
                }
                .setNegativeButton(R.string.cancel, null)
                .setSingleChoiceItems(names, index) { _, which ->
                    index = which
                }
                .show()
        })

        mViewModel?.permissionsDeniedLiveData?.observe(this, Observer {
            permissionSettingDialog =
                MaterialAlertDialogBuilder(context, R.style.CustomMaterialAlertDialog)
                    .setTitle(R.string.permission_request_title)
                    .setMessage(getString(R.string.permission_request_content, it))
                    .setPositiveButton(R.string.setting) { _, _ ->
                        PermissionUtils.launchAppDetailsSettings()
                    }
                    .setNegativeButton(R.string.cancel, null)
                    .show()
        })

        mViewModel?.collectReportEvent?.observe(this, Observer { quality ->
            quality?.let {
                qualityReportDialog = QualityReportDialog(context)
                qualityReportDialog?.setOnSubmitClickListener(object :
                    QualityReportDialog.OnSubmitClickListener {
                    override fun onSubmitClick(conversationQuality: ConversationQuality) {
                        mViewModel?.reportConversationQuality(conversationQuality)
                    }
                })
                qualityReportDialog?.setOnDismissListener(DialogInterface.OnDismissListener {
                    mViewModel?.notifyQualityDialogVisibleChanged(false)
                })
                qualityReportDialog?.setOnShowListener(DialogInterface.OnShowListener {
                    mViewModel?.notifyQualityDialogVisibleChanged(true)
                })
                qualityReportDialog?.setConversationQuality(it)
                qualityReportDialog?.show()
            }
        })

        mViewModel?.shareLinkConfirmLiveData?.observe(this, Observer {
            showShareLinkConfirmDialog()
        })

        mViewModel?.audioDumpStatusChangedEvent?.observe(this, Observer {
            if (it == DumpStatus.FINISHED) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N && isInPictureInPictureMode) {
                    Logger.i(TAG, "currently in pip mode, delayed wake up bugreport")
                    needWakeBugReport = true
                    return@Observer
                }
                wakeBugReport()
            }
        })

        mViewModel?.reportResultEvent?.observe(this, {
            showUploadResult(it)
        })

        mViewModel?.roomInfoChangedLiveData?.observe(this) {
            this.roomInfo = it
        }

        mViewModel?.weworkLoginEvent?.observe(this) { succeed ->
            if (succeed) {
                exitRoom(true)
            }
        }

        mViewModel?.joinRoomConfirmEvent?.observe(this, { room ->
            showJoinRoomConfirm(room)
        })

        mViewModel?.audioRatingEvent?.observe(this) {
            getNavController().navigate(
                R.id.action_global_audio_rating,
                bundleOf("data" to GsonUtils.toJson(it))
            )
        }

        mViewModel?.audioRatingUploadStatusChangedEvent?.observe(this) {
            showAudioScoringUploadResult(it)
        }

        mViewModel?.installApkEvent?.observe(this) { path ->
            mViewModel?.installApp(path)
        }

        mViewModel?.privacyTermsLiveData?.observe(this, Observer {
            privacyTermsDialog.show(it)
        })

        mViewModel?.incomingCallEvent?.observe(this) { data ->
            val currentFragmentId = getCurrentFragmentId()
            if (data == null) {
                if (currentFragmentId == R.id.incoming) {
                    getNavController().popBackStack()
                }
                return@observe
            }
            if (currentFragmentId != R.id.incoming) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N && isInPictureInPictureMode) {
                    exitPictureInPictureMode()
                }
                getNavController().navigate(R.id.action_global_incoming, bundleOf().apply {
                    putString(IncomingFragment.KEY_DATA, GsonUtils.toJson(data))
                })
            }
        }

        mViewModel?.apkDownloadProgress?.observe(this) { process ->
            appUpgradeDialog?.setProgress(process)
        }
    }

    private fun wakeBugReport() {
        if (getCurrentFragmentId() != R.id.bugReport) {
            getNavController().navigate(R.id.action_global_bug_report)
        }
    }

    override fun initialize(savedInstanceState: Bundle?) {
        supportFragmentManager.registerFragmentLifecycleCallbacks(fragmentTracker, true)

        KeyboardUtils.registerSoftInputChangedListener(window) { height ->
            mViewModel?.softInputChanged(height)
        }

        startTelephonyWatch()

        startScreenStatusMonitor()

        registerReceiver(
            volumeBroadcastReceiver,
            IntentFilter(VolumeBroadcastReceiver.VOLUME_CHANGED_ACTION)
        )

        mViewModel?.initialize()

        mViewModel?.analysisScheme(intent)
    }

    override fun onNewIntent(intent: Intent?) {
        super.onNewIntent(intent)
        handleIntentByAction(intent)
        mViewModel?.analysisScheme(intent)
    }

    fun exitPictureInPictureMode() {
        val intent = Intent(this, this.javaClass)
        intent.flags = Intent.FLAG_ACTIVITY_REORDER_TO_FRONT
        startActivity(intent)
        onPictureInPictureModeChanged(false)
    }

    private fun handleIntentByAction(intent: Intent?) {
        when (intent?.action) {
            MediaRequestReceiver.MEETING_EVENT_ACTION_VIDEO -> {
                popBackToMeetingIfNeed()
                mViewModel?.agreeOpenLocalVideoByRequest()
            }
        }
    }

    private fun showLoginDialog(type: Int) {
        wwAuthLoginDialog = WeWorkLoginDialog(this, type = type)
            .setWWAppInstalled(mViewModel?.isWWAppInstalled() ?: false)
            .setOnMenuClickListener(object : WeWorkLoginDialog.OnMenuClickListener {
                override fun loginWithPassword(type: Int) {
                    wwAuthLoginDialog?.dismiss()
                    getNavController().navigate(R.id.action_global_we_work_selection)
                }

                override fun loginWithQr(type: Int) {
                    wwAuthLoginDialog?.dismiss()
                    getNavController().navigate(
                        R.id.action_global_wework_web,
                        bundleOf(TYPE_COMPANY to type)
                    )
                }

                override fun loginWithEmail() {
                    wwAuthLoginDialog?.dismiss()
                    getNavController().navigate(
                        R.id.action_global_web,
                        bundleOf(KEY_URL to getOAuthUrl())
                    )
                }

                override fun cancel() {
                    wwAuthLoginDialog?.dismiss()
                }
            }).show()
    }

    private fun startScreenStatusMonitor() {
        screenStatusMonitor.setScreenStatusChangedListener(object :
            ScreenStatusReceiver.OnScreenStatusChangedListener {
            override fun onScreenStatusChanged(on: Boolean) {
                mViewModel?.onScreenStatusChanged(on)
            }
        })
        val intentFilter = IntentFilter()
        intentFilter.addAction(Intent.ACTION_SCREEN_ON)
        intentFilter.addAction(Intent.ACTION_SCREEN_OFF)
        registerReceiver(screenStatusMonitor, intentFilter)
    }

    private fun startTelephonyWatch() {
        telephonyWatcher.startWatch()
    }

    private fun showUpgrade(it: AppVersion?) {
        appUpgradeDialog = AppUpgradeDialog(context, it).setUpgradeCallback(object :
            AppUpgradeDialog.UpgradeCallback {
            override fun onUpgradeDismiss(notice: String) {
                showNotice(notice)
            }

            override fun onUpgradeClick(appVersion: AppVersion?) {
                mViewModel?.upgradeApp(appVersion)
            }
        })
        appUpgradeDialog?.showDialog()
    }

    private fun showNotice(notice: String) {
        AppNoticeDialog(context, notice).show()
    }

    private fun exitRoom(exit: Boolean) {
        if (exit) {
            getNavController().popBackStack(R.id.enter, false)
        }
    }

    private fun popBackToMeetingIfNeed() {
        if (getCurrentFragmentId() == R.id.audio) {
            getNavController().popBackStack(R.id.meeting, false)
        }
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        val orientation = newConfig.orientation
        if (orientation == Configuration.ORIENTATION_LANDSCAPE) {
            BarUtils.setStatusBarVisibility(this, false)
        } else {
            BarUtils.setStatusBarVisibility(this, true)
        }
    }

    fun setRoomTheme(roomMode: Int, postEvent: Boolean = true) {
        Logger.i(TAG, "change room mode:$roomMode")
        val modeTheme = if (roomMode == RoomMode.AGORA.value) {
            R.style.MyTheme_Private
        } else {
            R.style.MyTheme_DayNight
        }
        setTheme(modeTheme)
        if (postEvent) {
            mViewModel?.syncRoomModeEvent?.postValue(roomMode)
        }
    }

    override fun onDestroy() {
        supportFragmentManager.unregisterFragmentLifecycleCallbacks(fragmentTracker)

        KeyboardUtils.unregisterSoftInputChangedListener(window)

        stopTelephonyWatch()

        stopScreenStatusMonitor()

        releaseDialog()

        volumeBootHelper.destroy()

        unregisterReceiver(volumeBroadcastReceiver)

        super.onDestroy()
    }

    private fun releaseDialog() {
        videoRatingDialog?.let {
            if (it.isShowing) {
                it.dismiss()
            }
        }
        permissionSettingDialog?.let {
            if (it.isShowing) {
                it.dismiss()
            }
        }
        wwAuthLoginDialog?.let {
            if (it.isShowing) {
                it.dismiss()
            }
        }
        errorAlertDialog?.let {
            if (it.isShowing) {
                it.dismiss()
            }
        }
        uploadingDialog?.let {
            if (it.isShowing()) {
                it.dismiss()
            }
        }
    }

    private fun stopScreenStatusMonitor() {
        unregisterReceiver(screenStatusMonitor)
    }

    private fun stopTelephonyWatch() {
        telephonyWatcher.stopWatch()
    }

    override fun onStart() {
        super.onStart()
        mViewModel?.onAppStart()
        stopDaemonService()
    }

    override fun onResume() {
        super.onResume()
        mViewModel?.setEnableSpeakerphone(true)
    }

    override fun onStop() {
        super.onStop()
        mViewModel?.onAppStop()
        startDaemonService()
    }

    private fun stopDaemonService() {
        DaemonService.stopDaemonService(this)
    }

    override fun onUserLeaveHint() {
        super.onUserLeaveHint()
        val currentFragmentId = getCurrentFragmentId()
        if (currentFragmentId == R.id.meeting) {
            qualityReportDialog?.dismiss()
            uploadingDialog?.dismiss()
            val currentFragment = getCurrentFragment()
            if (currentFragment is MeetingFragment) {
                currentFragment.onUserLeaveHint()
            }
        }
    }

    private fun startDaemonService() {
        if (roomInfo != null) {
            DaemonService.startDaemonService(this, roomInfo)
        }
    }

    private fun showVideoRatingDialog(videoScore: VideoScore) {
        videoRatingDialog = VideoRatingDialog(this, videoScore = videoScore)
            .setOnRatingClickListener(object : VideoRatingDialog.OnRatingClickListener {
                override fun onRatingClick(videoScore: VideoScore) {
                    videoRatingDialog?.dismiss()
                    mViewModel?.recordVideoScore(videoScore)
                }
            }).show()
    }

    private fun getCurrentFragment(): Fragment? {
        val navHostFragment: Fragment? = supportFragmentManager.findFragmentById(R.id.nav_fragment)
        navHostFragment?.childFragmentManager?.fragments?.let {
            if (it.size > 0) {
                return it[0]
            }
        }
        return null
    }

    fun getCurrentFragmentId(): Int? {
        return getNavController().currentDestination?.id
    }

    private fun findFragment(clazz: KClass<*>): Fragment? {
        val navHostFragment: Fragment? = supportFragmentManager.findFragmentById(R.id.nav_fragment)
        navHostFragment?.childFragmentManager?.fragments?.let {
            it.forEach { _fragment ->
                if (_fragment::class == clazz) {
                    return _fragment
                }
            }
        }
        return null
    }

    private fun getNavController() = Navigation
        .findNavController(this, R.id.nav_fragment)

    private fun showShareLinkConfirmDialog() {
        roomInfo?.name?.let {
            shareLinkConfirmDialog =
                MaterialAlertDialogBuilder(context, R.style.CustomMaterialAlertDialog)
                    .setTitle(R.string.notice_share_link_title)
                    .setCancelable(true)
                    .setMessage(getString(R.string.notice_share_link_message))
                    .setNegativeButton(R.string.notice_share_link_cancel) { _, _ ->
                        mViewModel?.createAgoraLink2Clipboard(0)
                    }
                    .setPositiveButton(R.string.notice_share_link_confirm) { _, _ ->
                        mViewModel?.createShareLinkFromServer()
                    }
                    .show()
        }
    }

    private fun showUploadResult(issue: Issue) {
        if (uploadingDialog == null || uploadingDialog?.isShowing() == false) {
            uploadingDialog = UploadingDialog.build {
                this.context = this@AgoraActivity.context
                this.checkAgain = true
                this.cancelClickListener = object : UploadingDialog.OnCancelClickListener {
                    override fun onCancelClick() {
                        mViewModel?.cancelReportIssue()
                    }
                }
                this.retryClickListener = object : UploadingDialog.OnRetryClickListener {
                    override fun onRetryClick() {
                        mViewModel.bugReportRetryEvent.postValue(true)
                    }
                }
            }
        }
        uploadingDialog?.show(issue.status, getBugReportUploadDialogContent(issue))
    }

    private fun showAudioScoringUploadResult(status: Int) {
        if (uploadingDialog == null || uploadingDialog?.isShowing() == false) {
            uploadingDialog = UploadingDialog.build {
                this.context = this@AgoraActivity.context
                this.cancelClickListener = object : UploadingDialog.OnCancelClickListener {
                    override fun onCancelClick() {
                        mViewModel?.cancelAudioRating()
                    }
                }
                this.retryClickListener = object : UploadingDialog.OnRetryClickListener {
                    override fun onRetryClick() {
                        mViewModel.audioRatingRetryEvent.postValue(true)
                    }
                }
            }
        }
        uploadingDialog?.show(status)
    }

    fun isUploadDialogShowing(): Boolean {
        return uploadingDialog?.isShowing() == true
    }

    fun isErrorAlertDialogShowing(): Boolean {
        return errorAlertDialog?.isShowing == true
    }

    fun isPermissionSettingDialogShowing(): Boolean {
        return permissionSettingDialog?.isShowing == true
    }

    fun isShareLinkConfirmDialogShowing() = shareLinkConfirmDialog?.isShowing == true

    override fun onPictureInPictureModeChanged(
        isInPictureInPictureMode: Boolean, newConfig: Configuration?
    ) {
        super.onPictureInPictureModeChanged(isInPictureInPictureMode, newConfig)
        if (needWakeBugReport) {
            wakeBugReport()
            needWakeBugReport = false
        }
        volumeBootHelper.isInPictureInPictureMode = isInPictureInPictureMode
    }

    override fun onKeyDown(keyCode: Int, event: KeyEvent?): Boolean {
        if (keyCode == KeyEvent.KEYCODE_BACK) {
            onBackPressed()
            return true
        }
        if (volumeBootHelper.onKeyDown(mViewModel.isInRoom(), keyCode)) {
            return true
        }
        return super.onKeyDown(keyCode, event)
    }

    override fun onKeyUp(keyCode: Int, event: KeyEvent?): Boolean {
        if (volumeBootHelper.onKeyUp(mViewModel.isInRoom(), keyCode)) {
            return true
        }
        return super.onKeyDown(keyCode, event)
    }

    private fun showJoinRoomConfirm(room: Room) {
        if (room.name != null && context != null) {
            val message = if (room.isInternal()) {
                getString(R.string.join_room_invite_agora, room.name)
            } else {
                getString(R.string.join_room_invite_public, room.name)
            }

            MaterialAlertDialogBuilder(this, R.style.CustomMaterialAlertDialog)
                .setMessage(message)
                .setNegativeButton(R.string.cancel_button) { _, _ ->
                    Logger.i(TAG, "my cancel enter room:${room.name}")
                }
                .setPositiveButton(R.string.ok_button) { _, _ ->
                    exitRoom(true)
                    mViewModel?.joinRoom(room)
                }
                .create()
                .show()
        }
    }

    private fun getBugReportUploadDialogContent(issue: Issue) = when (issue.status) {
        UploadingStatus.COMPLETED -> {
            SpanUtils()
                .append(context.getString(R.string.upload_status_success_order))
                .append(issue.issueId.toString())
                .setForegroundColor(ContextCompat.getColor(context, R.color.red_400_3))
                .append("\n")
                .append(context.getString(R.string.dump_uploaded_success))
                .create()
        }
        UploadingStatus.STARTING -> {
            context.getString(R.string.dump_uploading_wait)
        }
        UploadingStatus.ERROR -> {
            if (!StringUtils.isEmpty(issue.path)) {
                context.getString(
                    R.string.upload_status_error_with_content,
                    roomInfo?.name ?: "",
                    issue.path
                )
            } else {
                context.getString(
                    R.string.upload_status_error_content,
                    roomInfo?.name
                )
            }
        }
        else -> {
            null
        }
    }

    companion object {
        private const val TAG = "[UI][Main]"
    }
}
