package io.agora.avc.app.master

import android.Manifest
import android.app.Application
import android.content.Context
import android.content.Intent
import android.media.AudioManager
import android.net.Uri
import androidx.lifecycle.MutableLiveData
import com.agora.valoran.Constants
import com.agora.valoran.bean.IncomingData
import io.agora.avc.MyApplication
import io.agora.avc.R
import io.agora.avc.auth.signin.SignInManager
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.*
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.dao.PermissionDao
import io.agora.avc.extensions.getConferenceNickname
import io.agora.avc.functions.RetryForever
import io.agora.avc.manager.audiodump.AudioDumper
import io.agora.avc.manager.bi.BIManager
import io.agora.avc.manager.bugreport.BugReporter
import io.agora.avc.manager.link.LinkManager
import io.agora.avc.manager.logupload.LogUploader
import io.agora.avc.manager.notice.NoticeEvent
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.manager.notice.NoticeType
import io.agora.avc.manager.quality.QualityManager
import io.agora.avc.manager.rating.call.CallRatingManager
import io.agora.avc.manager.rating.video.VideoRatingManager
import io.agora.avc.manager.rating.video.VideoScore
import io.agora.avc.manager.upgrade.AppUpgradeManager
import io.agora.avc.net.BaseObserver
import io.agora.avc.net.download.Downloader
import io.agora.avc.push.*
import io.agora.avc.receiver.MediaRequestReceiver
import io.agora.avc.repository.AppConfigRepository
import io.agora.avc.repository.EquipmentRepository
import io.agora.avc.repository.RoomRepository
import io.agora.avc.repository.UserRepository
import io.agora.avc.utils.*
import io.agora.frame.base.livedata.EventLiveData
import io.agora.frame.base.livedata.SingleLiveEvent
import io.agora.logger.LogConverter
import io.agora.logger.Logger
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import java.io.File
import java.util.concurrent.atomic.AtomicBoolean
import javax.inject.Inject

class MainViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val userRepository: UserRepository,
    private val roomRepository: RoomRepository,
    private val appConfigRepository: AppConfigRepository,
    private val equipmentRepository: EquipmentRepository,
    private val linkManager: LinkManager,
    private val logUploader: LogUploader,
    private val qualityManager: QualityManager,
    private val signInManager: SignInManager,
    private val videoRatingManager: VideoRatingManager,
    private val bugReporter: BugReporter,
    private val biManager: BIManager,
    private val appEventBus: AppEventBus,
    private val appUpgradeManager: AppUpgradeManager,
    private val audioDumper: AudioDumper,
    private val callRatingManager: CallRatingManager,
    private val noticeManager: NoticeManager,
    private val pushManager: PushManager,
) : AppViewModel(application, appController) {
    val softInputEvent = EventLiveData<Int>()
    val appUpgradeEvent: EventLiveData<AppVersion?> = EventLiveData()
    val appNoticeEvent: EventLiveData<String> = EventLiveData()
    val roomChangedConfirmLiveData: EventLiveData<Room> = EventLiveData()
    val localUserChangedLiveData: EventLiveData<LocalUser> = EventLiveData()
    val exitRoomLiveData: SingleLiveEvent<Boolean> = SingleLiveEvent()
    val roomInfoChangedLiveData: EventLiveData<Room> = EventLiveData()
    val qualityDialogVisible: EventLiveData<Boolean> = EventLiveData()
    val loginRequestEvent: EventLiveData<Room> = EventLiveData()
    val loginWeWorkEvent: EventLiveData<Int> = EventLiveData()

    val videoRatingEvent = EventLiveData<VideoScore>()
    val toastLiveData = EventLiveData<NoticeEvent?>()
    val dialogLiveData = EventLiveData<NoticeEvent?>()
    val userVideoSelectEvent = EventLiveData<ArrayList<ARoomUser>>()
    val permissionsDeniedLiveData = EventLiveData<String>()
    val collectReportEvent = EventLiveData<ConversationQuality>()
    val shareLinkConfirmLiveData = EventLiveData<Unit>()
    val audioDumpStatusChangedEvent = EventLiveData<Int>()
    val reportResultEvent = EventLiveData<Issue>()
    val roomModeChangedLiveData = EventLiveData<Int>()
    val syncRoomModeEvent = EventLiveData<Int>()
    val weworkLoginEvent: EventLiveData<Boolean> = EventLiveData()
    val joinRoomConfirmEvent = EventLiveData<Room>()
    val joinRoomEvent = EventLiveData<Room>()
    val audioRatingEvent = EventLiveData<MosScore>()
    val audioRatingUploadStatusChangedEvent = EventLiveData<Int>()
    val audioRatingRetryEvent = EventLiveData<Boolean>()
    val callRatingRetryEvent = EventLiveData<Boolean>()
    val bugReportRetryEvent = EventLiveData<Boolean>()

    val systemVolumeChangeEvent = EventLiveData<Intent?>()
    val installApkEvent = EventLiveData<String>()
    val privacyTermsLiveData = EventLiveData<Boolean>()
    val incomingCallEvent = EventLiveData<IncomingData?>()

    val appConfigLiveData = MutableLiveData<AppConfig?>()
    val apkDownloadProgress = MutableLiveData<Int>()

    private var isVideoHang: Boolean = false

    private val privacyShowing: AtomicBoolean = AtomicBoolean(false)

    override fun onCreate() {
        super.onCreate()
        checkIncomingCallStatus()
    }

    fun initialize() {
        queryPrivacy()
        signInManager.initialize(getApplication())
        initAppController()
        initLocalUser(DeviceUtils.getPlatform(), DeviceUtils.getModel())
        if (!privacyShowing.get()) {
            queryAppVer(false)
        }
        preloadMediaEffect()
    }

    /**
     * When the interface is created, you need to recheck the call status
     */
    private fun checkIncomingCallStatus() {
        pushManager.triggerIncomingCall()
    }

    private fun queryPrivacy() {
        if (!appConfigRepository.privacyShowed()) {
            privacyShowing.compareAndSet(false, true)
            privacyTermsLiveData.postValue(false)
        }
    }

    fun agreePrivacy() {
        if (privacyShowing.compareAndSet(true, false)) {
            queryAppVer(false)
        }
        appConfigRepository.saveShowPrivacy(true)
    }

    fun queryAppVer(filterRecord: Boolean) {
        val appVer = AppUtils.getAppVersionName()
        appConfigRepository.queryAppVer(
            appVer,
            UPGRADE_TYPE,
            LanguageUtils.getCurrentLocale().language
        )
            .retryWhen(RetryForever())
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<AppVersion>() {
                override fun onSuccess(config: AppVersion?) {
                    val notice = ""
                    val forceUpgrade = appUpgradeManager.isForceUpgrade(config, appVer)
                    config?.forceUpdate = forceUpgrade
                    config?.latest?.let { latest ->
                        config?.upgradeUrl = appUpgradeManager.getUpgradeUrl(latest)
                    }
                    val showNotice = notice != null && appUpgradeManager.showNoticeIfNeed(notice)
                    config?.showNotice = showNotice
//                    if (appUpgradeManager.hasNewVersion(config, appVer, filterRecord)) {
//                        doAppUpgrade(config)
//                    } else if (!filterRecord && showNotice) {
//                        appNoticeEvent.postValue(notice)
//                    } else if (filterRecord) {
//                        noticeManager.notice(
//                            NoticeEvent(
//                                NoticeCode.CODE_SETTING_VERSION_LATEST,
//                                NoticeType.TOAST
//                            )
//                        )
//                    }
                }

                private fun doAppUpgrade(config: AppVersion?) {
                    val process = apkDownloadProgress.value ?: -1
                    if (process in 0 until 100) {
                        noticeManager.notice(
                            NoticeEvent(
                                NoticeCode.CODE_UPGRADE_DOWNLOADING,
                                NoticeType.TOAST
                            )
                        )
                        return
                    }
                    appUpgradeEvent.postValue(config)
                }

                override fun onFail(t: AppVersion?) {
                }
            })

    }

    private fun uploadLogIfNeed(config: AppConfig?) {
        if (config?.app?.needUploadLog == true) {
            Logger.i(TAG, "Upload log, when receiving needUploadLog from config")
            getLocalUser()?.streamId?.apply {
                logUploader.doUploadLog(this)
            }
        }
    }

    private fun queryAppConfig() {
        appConfigRepository.queryAppConfig(
            AppUtils.getAppVersionName(),
            UPGRADE_TYPE,
            "all"
        )
            .retryWhen(RetryForever())
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<AppConfig>() {
                override fun onSuccess(config: AppConfig?) {
                    appConfigLiveData.postValue(config)
                    uploadLogIfNeed(config)
                }

                override fun onFail(t: AppConfig?) {
                }
            })
    }

    fun softInputChanged(height: Int) {
        softInputEvent.value = height
    }

    fun analysisScheme(intent: Intent?) {
        //action analysis
        if (intent?.action == ACTION_PUSH) {
            Logger.i(TAG, "action push requestId: ${intent.extras?.getString(KEY_REQUEST_ID)}")
            intent.extras?.apply {
                val rid = getString(KEY_RID)
                val pwd = getString(KEY_PWD)
                val requestId = getString(KEY_REQUEST_ID)
                val ticket = getString(KEY_TICKET)
                val sourceName = getString(KEY_INVITER)
                val avatar = getString(KEY_AVATAR)
                pushManager.onIncomingNew(IncomingData().apply {
                    this.rid = rid
                    this.pwd = pwd
                    this.ticket = ticket
                    this.inviterName = sourceName
                    this.inviterAvatar = avatar
                    this.requestId = requestId
                })
            }
            return
        }
        //uri analysis
        val data: Uri = intent?.data ?: return
        Logger.i(TAG, "scheme: $data received from app launch")
        val localUser = getLocalUser()
        if (localUser == null) {
            Logger.i(TAG, "can't handle scheme link, local user is empty")
            return
        }
        linkManager.handleSchemeLink(data, getRoom(), localUser)
    }

    fun changeRoom(newRoom: Room) {
        appController.changeRoom(newRoom)
    }

    fun notifyQualityDialogVisibleChanged(visible: Boolean) {
        qualityDialogVisible.postValue(visible)
    }

    private fun verifyInnerSession(localUser: LocalUser) {
        if (localUser.isThirdPartyLoggedIn) {
            signInManager.verifyAccount(localUser.thirdPartyInnerSession, localUser.source)
        }
    }

    private fun preloadMediaEffect() {
        val effects = arrayListOf<android.util.Pair<Constants.EffectType, String>>()
        copyAssetFile(getEffectFilePath(JOIN_MEDIA_FILE_NAME), JOIN_MEDIA_FILE_NAME)
        copyAssetFile(getEffectFilePath(DISCONNECT_MEDIA_FILE_NAME), DISCONNECT_MEDIA_FILE_NAME)
        copyAssetFile(getEffectFilePath(RECONNECT_MEDIA_FILE_NAME), RECONNECT_MEDIA_FILE_NAME)
        copyAssetFile(getEffectFilePath(MUTE_AUDIO_EN_FILE_NAME), MUTE_AUDIO_EN_FILE_NAME)
        copyAssetFile(getEffectFilePath(MUTE_AUDIO_ZH_FILE_NAME), MUTE_AUDIO_ZH_FILE_NAME)
        copyAssetFile(getEffectFilePath(MUTE_VIDEO_FILE_NAME), MUTE_VIDEO_FILE_NAME)
        copyAssetFile(getEffectFilePath(UNMUTE_MEDIA_FILE_NAME), UNMUTE_MEDIA_FILE_NAME)
        copyAssetFile(getEffectFilePath(CRS_FILE_NAME), CRS_FILE_NAME)
        copyAssetFile(getEffectFilePath(ADS_FILE_NAME), ADS_FILE_NAME)

        effects.add(
            android.util.Pair(
                Constants.EffectType.EFFECT_JOIN_SUCCESS,
                getEffectFilePath(JOIN_MEDIA_FILE_NAME)
            )
        )
        effects.add(
            android.util.Pair(
                Constants.EffectType.EFFECT_MEDIA_DISCONNECTED,
                getEffectFilePath(DISCONNECT_MEDIA_FILE_NAME)
            )
        )
        effects.add(
            android.util.Pair(
                Constants.EffectType.EFFECT_MEDIA_RECONNECTED,
                getEffectFilePath(RECONNECT_MEDIA_FILE_NAME)
            )
        )
        effects.add(
            android.util.Pair(
                Constants.EffectType.EFFECT_MUTE_VIDEO,
                getEffectFilePath(MUTE_VIDEO_FILE_NAME)
            )
        )
        effects.add(
            android.util.Pair(
                Constants.EffectType.EFFECT_MUTE_AUDIO,
                if (LanguageUtils.isChinese()) {
                    getEffectFilePath(MUTE_AUDIO_ZH_FILE_NAME)
                } else {
                    getEffectFilePath(MUTE_AUDIO_EN_FILE_NAME)
                }
            )
        )
        effects.add(
            android.util.Pair(
                Constants.EffectType.EFFECT_UNMUTE,
                getEffectFilePath(UNMUTE_MEDIA_FILE_NAME)
            )
        )
        effects.add(
            android.util.Pair(
                Constants.EffectType.EFFECT_RECORDING,
                getEffectFilePath(CRS_FILE_NAME)
            )
        )
        effects.add(
            android.util.Pair(
                Constants.EffectType.EFFECT_DUMPING,
                getEffectFilePath(ADS_FILE_NAME)
            )
        )
        appController.configAudioEffects(effects)
    }

    private fun copyAssetFile(filePath: String, fileName: String) {
        if (!File(filePath).exists()) {
            FileUtils.copyAssetFile(filePath, fileName)
        }
    }

    private fun getEffectFilePath(fileName: String): String {
        return FileUtils.getFileDir(null).absolutePath + File.separator + fileName
    }

    fun forbidQualityReport() {
        qualityManager.forbidQualityReport()
    }

    fun quitRoom() {
        appController.leaveRoom()
    }

    fun startVideoRating(streamId: Int) {
        videoRatingManager.startVideoRating(streamId)
    }

    fun loginViaWeWork(type: Int) {
        signInManager.signInWework(type)
    }

    fun enterPureAudioMode() {
        appController.changeToAudioMode()
    }

    fun exitPureAudioMode() {
        appController.changeToVideoMode()
    }

    fun onScreenStatusChanged(on: Boolean) {

    }

    fun agreeOpenLocalVideoByRequest() {
        appController.acceptMediaRequest(Constants.RequestType.getValue(Constants.RequestType.VIDEO))
    }

    fun agreeOpenLocalAudio() {
        appController.acceptMediaRequest(Constants.RequestType.getValue(Constants.RequestType.AUDIO))
    }

    fun disagreeOpenLocalAudio() {
        appController.refuseMediaRequest(Constants.RequestType.getValue(Constants.RequestType.AUDIO))
    }

    fun disagreeOpenLocalVideo() {
        appController.refuseMediaRequest(Constants.RequestType.getValue(Constants.RequestType.VIDEO))
    }

    fun onAppStart() {
        //if user is interrupted then un_mute video
        if (getRoom() != null && isVideoHang) {
            Logger.i(TAG, "${LogConverter.muted(false)} local video,when App started")
            isVideoHang = false
            appController.setLocalVideo(true)
        }
        //clear request notification
        MediaRequestReceiver.dismissRequestNotification(getApplication())
    }

    fun onAppStop() {
        if (getRoom() != null && getLocalUser()?.videoState == true) {
            Logger.i(TAG, "${LogConverter.muted(true)} local video,when App stopped")
            isVideoHang = true
            appController.setLocalVideo(false)
        }
    }

    private fun queryRoom() {
        roomInfoChangedLiveData.postValue(roomRepository.getRoom())
    }

    fun toggleVideoRating() {
        videoRatingManager.toggleVideoRating()
    }

    fun recordVideoScore(videoScore: VideoScore) {
        videoRatingManager.recordVideoScore(videoScore)
    }

    fun createAgoraLink2Clipboard(type: Int) {
        val userName = getLocalUser()?.getConferenceNickname()
        val roomId = roomRepository.getRoom()?.name
        val roomPwd = roomRepository.getRoom()?.pwd
        if (roomId != null) {
            linkManager.createAgoraLink2Clipboard(roomId, roomPwd, userName, type, null)
        }
    }

    fun createShareLinkFromServer() {
        val userName = getLocalUser()?.getConferenceNickname()
        val roomId = roomRepository.getRoom()?.name
        val roomPwd = roomRepository.getRoom()?.pwd
        if (roomId != null) {
            linkManager.createShareLinkFromServer(roomId, roomPwd, userName)
        }
    }

    fun createShareLink() {
        val userName = getLocalUser()?.name
        val roomId = roomRepository.getRoom()?.name
        val roomPwd = roomRepository.getRoom()?.pwd
        if (roomRepository.getRoom()?.isInternal() == true) {
            shareLinkConfirmLiveData.postValue(Unit)
        } else if (userName != null && roomId != null) {
            linkManager.createShareLink2Clipboard(userName, roomId, roomPwd)
        }
    }

    fun cancelReportIssue() {
        bugReporter.cancelReportIssue()
    }

    fun onPermissionDenied(
        permissionsDeniedForever: MutableList<String>,
        permissionsDenied: MutableList<String>
    ) {
        val deniedDetails = PermissionDao.getDeniedDetails(permissionsDenied)
        if (permissionsDenied.contains(Manifest.permission.RECORD_AUDIO) ||
            permissionsDeniedForever.contains(Manifest.permission.RECORD_AUDIO)
        ) {
            appController.setAudioEnable(false)
        }
        permissionsDeniedLiveData.postValue(deniedDetails)
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.LEAVE_ROOM_EVENT.ordinal -> {
                audioDumper.destroy()
                queryRoom()
                exitRoomLiveData.postValue(true)
            }
            AppEvent.TOAST_EVENT.ordinal -> {
                toastLiveData.postValue(arg.obj as? NoticeEvent)
            }
            AppEvent.DIALOG_EVENT.ordinal -> {
                dialogLiveData.postValue(arg.obj as? NoticeEvent)
            }
            AppEvent.AUDIO_DUMPING_STATUS_CHANGE.ordinal -> {
                val data = arg.obj
                if (data is Issue) {
                    audioDumpStatusChangedEvent.postValue(data.dumpStatus)
                }
            }
            AppEvent.BUG_REPORT_STATUS_CHANGE.ordinal -> {
                arg.obj?.let { data ->
                    if (data is Issue) {
                        reportResultEvent.postValue(data)
                    }
                }
            }
            AppEvent.VIDEO_RATING_START.ordinal -> {
                val data = arg.obj as? ArrayList<ARoomUser>
                data?.apply {
                    userVideoSelectEvent.postValue(this)
                }
            }
            AppEvent.VIDEO_RATING_FINISH.ordinal -> {
                arg.obj?.let { data ->
                    if (data is VideoScore) {
                        videoRatingEvent.postValue(data)
                    }
                }
            }
            AppEvent.ROOM_INFO_CHANGED.ordinal -> {
                queryRoom()
            }
            AppEvent.LOCAL_USER_CHANGED.ordinal -> {
                val isLoginStatusChanged = arg.obj
                if (isLoginStatusChanged is Boolean && isLoginStatusChanged) {
                    calibrateRoomMode()
                }
            }
            AppEvent.LINK_ENTER_NEW_ROOM_EVENT.ordinal -> {
                val data = arg.obj
                if (data is Room) {
                    roomChangedConfirmLiveData.postValue(data)
                }
            }
            AppEvent.LINK_LOGIN_REQUEST_EVENT.ordinal -> {
                val data = arg.obj
                if (data is Room) {
                    loginRequestEvent.postValue(data)
                }
            }
            AppEvent.WEWORK_LOGIN.ordinal -> {
                val succeed = arg.obj
                if (succeed is Boolean) {
                    Logger.i(TAG, "received wework login success event")
                    weworkLoginEvent.postValue(succeed)
                }
            }
            AppEvent.ROOM_MODE_CHANGED.ordinal -> {
                changeRoomMode(getRoomMode())
            }
            AppEvent.LINK_ENTER_ROOM_EVENT.ordinal -> {
                arg.obj?.let { data ->
                    if (data is Room) {
                        joinRoomConfirmEvent.postValue(data)
                    }
                }
            }
            AppEvent.AUDIO_RATING_EVENT.ordinal -> {
                arg.obj?.let { data ->
                    if (data is MosScore) {
                        audioRatingEvent.postValue(data)
                    }
                }
            }
            AppEvent.AUDIO_RATING_UPLOAD_STATUS_CHANGE_EVENT.ordinal -> {
                arg.obj?.let { data ->
                    if (data is Int) {
                        audioRatingUploadStatusChangedEvent.postValue(data)
                    }
                }
            }
            AppEvent.WAIT_ASSISTANT_RESULT.ordinal -> {
                userRepository.isLocalUserApplyingAssistant = true
            }
            AppEvent.APPLY_ASSISTANT_OPERATION_RESULT.ordinal -> {
                userRepository.isLocalUserApplyingAssistant = false
            }
            AppEvent.INCOMING_CALL_EVENT.ordinal -> {
                val data = arg.obj
                if (data is IncomingData) {
                    pushManager.receiveIncomingCall(data)
                    incomingCallEvent.postValue(data)
                }
            }
            AppEvent.INCOMING_CALL_TIME_OUT.ordinal,
            AppEvent.INCOMING_CANCEL_EVENT.ordinal -> {
                incomingCallEvent.postValue(null)
            }
            AppEvent.CHANGE_ROOM_EVENT.ordinal -> {
                val data = arg.obj
                if (data is Room) {
                    changeRoomMode(data.mode)
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.ENTER_ROOM_EVENT.ordinal,
                            data
                        )
                    )
                }
            }
        }
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.ROOM_INFO_CHANGED,
            AppEvent.USER_INFO_CHANGED,
            AppEvent.LOCAL_USER_CHANGED,
            AppEvent.LEAVE_ROOM_EVENT,
            AppEvent.TOAST_EVENT,
            AppEvent.AUDIO_DUMPING_STATUS_CHANGE,
            AppEvent.BUG_REPORT_STATUS_CHANGE,
            AppEvent.VIDEO_RATING_START,
            AppEvent.VIDEO_RATING_FINISH,
            AppEvent.LINK_ENTER_NEW_ROOM_EVENT,
            AppEvent.LINK_LOGIN_REQUEST_EVENT,
            AppEvent.DIALOG_EVENT,
            AppEvent.WEWORK_LOGIN,
            AppEvent.ROOM_MODE_CHANGED,
            AppEvent.LINK_ENTER_ROOM_EVENT,
            AppEvent.AUDIO_RATING_EVENT,
            AppEvent.AUDIO_RATING_UPLOAD_STATUS_CHANGE_EVENT,
            AppEvent.WAIT_ASSISTANT_RESULT,
            AppEvent.APPLY_ASSISTANT_OPERATION_RESULT,
            AppEvent.INCOMING_CALL_EVENT,
            AppEvent.INCOMING_CALL_TIME_OUT,
            AppEvent.INCOMING_CANCEL_EVENT,
            AppEvent.CHANGE_ROOM_EVENT,
        )
    }

    private fun calibrateRoomMode() {
        val user = getLocalUser()
        val mode = getRoomMode()
        if ((user == null || !user.isThirdPartyLoggedIn) && mode == RoomMode.AGORA.value) {
            Logger.i(TAG, "calibrate room mode, user:${user?.isThirdPartyLoggedIn}, mode:$mode")
            changeRoomMode(RoomMode.NORMAL.value)
        }
    }

    private fun initAppController() {
        appController.initialize(
            getApplication(),
            appConfigRepository.isTestServer().not(),
            logUploader.getValoranLogPath(),
            UPGRADE_TYPE
        )
        val application = getApplication<MyApplication>()
        application.appContainer.appController = appController
    }

    private fun initLocalUser(platform: String, deviceName: String) {
        userRepository.getLocalUser(platform, deviceName)
            .retryWhen(RetryForever())
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : io.reactivex.Observer<LocalUser> {
                override fun onSubscribe(d: Disposable) {

                }

                override fun onNext(t: LocalUser) {
                    appController.setLocalUser(t)
                    appController.configClientInfo(
                        t?.uid ?: "",
                        equipmentRepository.getLangType(),
                        equipmentRepository.getFeatureType(),
                        equipmentRepository.getAppVersionName(),
                        equipmentRepository.getEquipmentInfo(),
                        equipmentRepository.getPlatform()
                    )
                    appController.configUser(t.streamId, t.uid, t.token)
                    biManager.biApp()
                    verifyInnerSession(t)
                    queryAppConfig()
                }

                override fun onError(e: Throwable) {
                    Logger.e(TAG, "failed to init local user", e)
                }

                override fun onComplete() {

                }
            })
    }

    fun collectQualityInfo(peerUid: Int?) {
        val channelId = getRoom()?.getChannelId()
        if (channelId == null) {
            Logger.e(TAG, "can not collect quality info, channelId is empty")
            return
        }
        val name =
            getLocalUser()?.getConferenceNickname() ?: StringUtils.getString(R.string.unknown_user)
        val reportId = getLocalUser()?.uid ?: ""
        val reportStreamId = getLocalUser()?.streamId ?: 0
        val callId = appController.getCallId()
        val conversation = qualityManager.createConversation(
            channelId,
            name,
            reportId,
            reportStreamId,
            callId,
            peerUid ?: 0,
            true
        )
        collectReportEvent.postValue(conversation)
    }

    fun reportConversationQuality(conversationQuality: ConversationQuality) {
        qualityManager.reportConversationQuality(conversationQuality)
    }

    fun isWWAppInstalled(): Boolean {
        return signInManager.isAppSupportWework()
    }

    override fun onDestroy() {
        signInManager.deInitialize()
        super.onDestroy()
    }

    fun getRoomMode(): Int {
        return appConfigRepository.getRoomMode()
    }

    fun changeRoomMode(roomMode: Int) {
        Logger.i(TAG, "change room mode request, mode:$roomMode")
        appConfigRepository.saveRoomMode(roomMode)
        roomModeChangedLiveData.postValue(roomMode)
    }

    fun joinRoom(room: Room) {
        joinRoomEvent.postValue(room)
    }

    fun adjustRemoteAudio(volume: Int) {
        Logger.i(TAG, "adjust remote audio: volume = $volume")
        appController.adjustPlaybackSignalVolume(volume)
    }

    fun isInRoom() = roomRepository.getRoom() != null

    fun systemVolumeChange(intent: Intent?) {
        systemVolumeChangeEvent.postValue(intent)
    }

    fun upgradeApp(appVersion: AppVersion?) {
        val upgradeUrl = appVersion?.upgradeUrl
        val version = appVersion?.latest?.v
        val forceUpdate = appVersion?.forceUpdate ?: false
        if (upgradeUrl == null || version == null) {
            Logger.e(TAG, "apk url or version is empty")
            return
        }
        appUpgradeManager.downloadApk(
            upgradeUrl,
            version,
            forceUpdate,
            if (forceUpdate) DOWNLOAD_APK_DELAY else 0L,
            progressListener = { currentBytes, contentLength, done ->
                val progress = (currentBytes.toFloat() / contentLength).times(100).toInt()
                apkDownloadProgress.postValue(progress)
            },
            completeListener = object : Downloader.CompleteListener {
                override fun onComplete(path: String) {
                    //Note: The user will be prompted to install when it is visible
                    apkDownloadProgress.postValue(PROCESS_APK_DOWNLOAD_SUCCEED)
                    installApkEvent.postValue(path)
                }
            },
            errorListener = object : Downloader.ErrorListener {
                override fun onError(e: Throwable) {
                    apkDownloadProgress.postValue(PROCESS_APK_DOWNLOAD_FAILED)
                    noticeManager.notice(
                        NoticeEvent(
                            NoticeCode.CODE_STRID_UPDATER_ERROR_TITLE,
                            NoticeType.TOAST
                        )
                    )
                }
            }
        )

    }

    fun cancelAudioRating() {
        callRatingManager.cancelAudioRating()
    }

    fun cancelCallRating() {
        callRatingManager.cancelCallRating()
    }

    fun installApp(path: String?) {
        if (path == null || !FileUtils2.isFileExists(path)) {
            Logger.e(TAG, "failed to install apk, please check apk path:$path")
            noticeManager.notice(
                NoticeEvent(
                    NoticeCode.CODE_UPGRADE_ERROR_FILE_PATH,
                    NoticeType.TOAST
                )
            )
            return
        }
        AppUtils.installApp(path)
    }

    fun setEnableSpeakerphone(enable: Boolean) {
        (getApplication<MyApplication>().getSystemService(Context.AUDIO_SERVICE) as? AudioManager)?.let {
            if (it.isSpeakerphoneOn) {
                Logger.i(TAG, "AUDIO_ROUTE_SPEAKERPHONE")
            } else if (it.isBluetoothScoOn || it.isBluetoothA2dpOn) {
                Logger.i(TAG, "AUDIO_ROUTE_HEADSETBLUETOOTH")
            } else if (it.isWiredHeadsetOn) {
                Logger.i(TAG, "AUDIO_ROUTE_HEADSET")
            } else {
                Logger.i(TAG, "AUDIO_ROUTE_EARPIECE")
                appController.setEnableSpeakerphone(enable)
            }
        }
    }

    companion object {
        private const val TAG = "[VM][Main]"
        private const val UPGRADE_TYPE = 0

        //In order to show the loading status, forcibly delayed 2 seconds
        private const val DOWNLOAD_APK_DELAY = 2000L
        private const val PROCESS_APK_DOWNLOAD_FAILED = -1
        private const val PROCESS_APK_DOWNLOAD_SUCCEED = 100

        private const val JOIN_MEDIA_FILE_NAME = "join_success.mp3"
        private const val RECONNECT_MEDIA_FILE_NAME = "rtc_reconnected.mp3"
        private const val DISCONNECT_MEDIA_FILE_NAME = "rtc_disconnected.mp3"
        private const val MUTE_AUDIO_EN_FILE_NAME = "be_muted_en.mp3"
        private const val MUTE_AUDIO_ZH_FILE_NAME = "be_muted_cn.mp3"
        private const val MUTE_VIDEO_FILE_NAME = "ctr_mute.mp3"
        private const val UNMUTE_MEDIA_FILE_NAME = "ctr_request_unmute.mp3"
        private const val CRS_FILE_NAME = "cloud_recording.mp3"
        private const val ADS_FILE_NAME = "audio_dumping.mp3"
    }
}