package io.agora.avc.app.meeting

import android.Manifest
import android.annotation.SuppressLint
import android.content.ActivityNotFoundException
import android.content.Context
import android.content.Intent
import android.content.res.Configuration
import android.os.Build
import android.os.Bundle
import android.util.Log
import android.view.*
import androidx.appcompat.content.res.AppCompatResources
import androidx.core.content.ContextCompat
import androidx.core.view.*
import androidx.lifecycle.Observer
import androidx.navigation.NavController
import androidx.navigation.fragment.NavHostFragment
import androidx.recyclerview.widget.DividerItemDecoration
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.bumptech.glide.Glide
import com.google.android.material.bottomsheet.BottomSheetBehavior
import io.agora.avc.R
import io.agora.avc.app.audio.AudioFragment
import io.agora.avc.app.developer.DeveloperOptions
import io.agora.avc.app.master.AgoraActivity
import io.agora.avc.app.master.MainViewModel
import io.agora.avc.app.operation.OperationFragment
import io.agora.avc.bo.*
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.config.KEY_REQUEST_ID
import io.agora.avc.config.RESOLUTION_MEDIUM
import io.agora.avc.dao.PermissionDao
import io.agora.avc.databinding.FragmentMeetingBinding
import io.agora.avc.extensions.*
import io.agora.avc.manager.bugreport.DumpStatus
import io.agora.avc.receiver.MediaRequestReceiver
import io.agora.avc.screenshare.ScreenShareManager
import io.agora.avc.utils.DeviceUtils
import io.agora.avc.utils.PermissionConstants
import io.agora.avc.utils.PermissionUtils
import io.agora.avc.utils.ToastUtils
import io.agora.avc.widget.ConnectingView
import io.agora.avc.widget.MeetingRootLayout
import io.agora.avc.widget.RoomMenuPopupWindow
import io.agora.avc.widget.ScaleLayout
import io.agora.avc.widget.helper.PictureInPictureHelper
import io.agora.avc.widget.helper.SmallStreamCoordinatorLayout
import io.agora.frame.base.HolderFragment
import io.agora.logger.LogConverter
import io.agora.logger.Logger

/**
 * When the Activity changed, you need to set the wantToBeInPipMode value,
 * according to whether you want to start picture-in-picture
 * Otherwise it will cause problems with the recent task
 */
class MeetingFragment : HolderFragment<MeetingViewModel, FragmentMeetingBinding>(),
    View.OnClickListener {

    private var systemWindowInsetTop: Int = 0
    private var systemWindowInsetLeft: Int = 0
    private var systemWindowInsetRight: Int = 0
    private var systemWindowInsetBottom: Int = 0

    private val mainViewModel: MainViewModel? by lazy {
        getViewModel(requireActivity().viewModelStore, MainViewModel::class.java)
    }
    private var mainScreenUser: ARoomUser? = null
    private var room: Room? = null
    private var isCloudRecording = false
    private var isScreenSharing = false
    private var hasUnreadMessage = false
    private var isMediaListNotEmpty = false
    private var openAudioConfirm = false
        set(value) {
            field = value
            audioFragment.openAudioConfirm = value
        }

    private var localUser: LocalUser? = null

    private var developerOptions: DeveloperOptions? = null

    private var requestingScreenShare = false
    private var requestingPermissions = false
    private var bizConnected = false

    private var pageAdapter = MeetingAdapter()

    private val dividerItemDecoration by lazy {
        DividerItemDecoration(context, DividerItemDecoration.HORIZONTAL).also { decoration ->
            AppCompatResources.getDrawable(requireContext(), R.drawable.shape_meeting_divider)
                ?.let {
                    decoration.setDrawable(it)
                }
        }
    }

    private val pictureHelper by lazy {
        PictureInPictureHelper(requireActivity(), object : PictureInPictureHelper.Callback {
            override fun getResolutionOption(): Int {
                return room?.resolution ?: RESOLUTION_MEDIUM
            }

            override fun getMainScreenUser(): ARoomUser? {
                return mainScreenUser
            }
        })
    }
    private val audioFragment by lazy {
        AudioFragment().apply {
            onBackPressListener = { audioModeOnBackPress() }
        }
    }

    private var mainScreenVideo: View? = null

    private val screenShareManager by lazy {
        ScreenShareManager(this, mViewModel)
    }

    private val screenShareObserver: Observer<ShareInfo> = Observer { shareInfo ->
        screenShareManager.start(shareInfo)
    }

    private val kickOutObserver: Observer<Boolean> = Observer<Boolean> {
        screenShareManager.stop()
        if (isInPictureInPictureMode()) {
            (activity as? AgoraActivity)?.exitPictureInPictureMode()
        }
    }

    private val appInBackgroundNotificationChangedObserver = Observer<MeetingNotification> {
        context?.let { _context ->
            if (it.show) {
                Logger.i(
                    TAG,
                    "show system notification : remote request code is ${it.code.ordinal}"
                )
                MediaRequestReceiver.notifyRemoteRequest(_context, it)
            }
        }
    }
    private var isQualityReportShowing: Boolean = false

    private val configChangedHelper by lazy {
        MeetingConfigChangedHelper(requireContext())
    }

    private val systemWindowHelper by lazy {
        MeetingSystemWindowHelper(activity)
    }
    private val fullScreenHelper by lazy {
        MeetingFullScreenHelper(activity, this)
    }

    private var connectingView = ConnectingView()

    /**
     * ScreenUtil#isPortrait is not used because after the background is switched to the foreground,
     * the screenUtil#isPortrait gets the old value after the screen is quickly rotated.
     */
    private var isScreenPortrait = true

    private fun isInPictureInPictureMode() =
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            activity?.isInPictureInPictureMode == true
        } else {
            false
        }

    private fun isFullScreen() = fullScreenHelper.isFullScreen

    fun getSystemWindowInsetTop() = systemWindowInsetTop
    fun getSystemWindowInsetLeft() = systemWindowInsetLeft
    fun getSystemWindowInsetRight() = systemWindowInsetRight
    fun getSystemWindowInsetBottom() = systemWindowInsetBottom

    override fun getLayoutId() = R.layout.fragment_meeting

    private var dumpStatus: Int = DumpStatus.NONE

    override fun onAttach(context: Context) {
        super.onAttach(context)
        isScreenPortrait =
            context.resources.configuration.orientation == Configuration.ORIENTATION_PORTRAIT
    }

    private val destinationChangedListener =
        NavController.OnDestinationChangedListener { _, destination, _ ->
            Logger.i(TAG, "destination changed, destination:$destination")
            mBinding.flRoot.screenFullEnable(destination.id == R.id.meeting)
        }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        requireActivity().window.addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {
        return super.onCreateView(inflater, container, savedInstanceState).apply {

            activity?.window?.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_NOTHING)

            NavHostFragment.findNavController(this@MeetingFragment)
                .addOnDestinationChangedListener(destinationChangedListener)

            pageAdapter.setItemClickListener(object : MeetingAdapter.OnItemListener {
                override fun onClicked(user: ARoomUser) {
                    mViewModel?.switchUserSequence(user)
                }

                override fun onLongClick(user: ARoomUser) {
                    if (mViewModel?.hasPermission(user) == true) {
                        OperationFragment.navigateTo(
                            this@MeetingFragment,
                            user,
                        )
                    } else {
                        Logger.e(TAG, "can not open operation fragment,user has not permission")
                    }
                }

                override fun onCameraClick(user: ARoomUser) {
                    clickCameraSwitch()
                }
            })

            mBinding.flRoot.timerCallBack = object :
                MeetingRootLayout.OnTimerCallBack {
                override fun onDismiss() {
                    if (!isFullScreen()
                        && !audioFragment.isAudioVisible
                        && !isQualityReportShowing
                        && !systemWindowHelper.hasPopupWindowShowing()
                        && !screenShareManager.isRequesting()
                    ) {
                        fullScreenHelper.isFullScreen = true
                    }
                }
            }

            mBinding.flContainer.apply {
                setRenderAdapter(object : ScaleLayout.RenderAdapter {
                    override fun getVideoView() = if (mainScreenUser?.isShareStream() == true) {
                        mainScreenVideo
                    } else {
                        null
                    }

                    override fun getWatermarkView() = if (mainScreenUser?.hasWatermark == true) {
                        mBinding.watermark
                    } else {
                        null
                    }
                })

                if (isInPictureInPictureMode()) {
                    onPictureInPictureModeChanged(true)
                }
            }
            initNotificationPop()
        }
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        setOnApplyWindowInsetsListener()
        activity?.window?.transparentNavBar()
        if (isScreenPortrait != (resources.configuration.orientation == Configuration.ORIENTATION_PORTRAIT)) {
            onConfigurationChanged(resources.configuration)
        }
    }

    @SuppressLint("ClickableViewAccessibility")
    override fun initialize(savedInstanceState: Bundle?) {
        systemWindowHelper.bind(mBinding)
        configChangedHelper.bind(mBinding, isScreenPortrait)
        fullScreenHelper.bind(mBinding, isScreenPortrait = { isScreenPortrait })

        mBinding.llTitle.setOnClickListener(this)
        mBinding.flInvite.setOnClickListener(this)
        mBinding.ivSignal.setOnClickListener(this)
        mBinding.flBugReport.setOnClickListener(this)
        mBinding.flSetting.setOnClickListener(this)
        mBinding.ivMediaListArrow.setOnClickListener(this)
        mBinding.bottomBar.setOnClickListener {
            bottomBarClick(it)
        }

        childFragmentManager.beginTransaction().apply {
            replace(R.id.flAudioContainer, audioFragment)
            commitNowAllowingStateLoss()
        }

        initSmallStream()

        mBinding.dragDownLayout.priorityView = mBinding.callLayout
        mBinding.dragDownLayout.onTargetViewVisibilityChanged = {
            onAudioShowOrHide(it)
        }

        View.OnLongClickListener {
            Logger.i(TAG, "main screen onLongPress, user:${mainScreenUser?.streamId}")
            mainScreenUser?.let { _mainScreenUser ->
                if (mViewModel?.hasPermission(_mainScreenUser) == true) {
                    OperationFragment.navigateTo(
                        this@MeetingFragment,
                        _mainScreenUser,
                    )
                }
            }
            true
        }.apply {
            mBinding.flRoot.setOnLongClickListener(this)
            mBinding.flContainer.setOnLongClickListener(this)
        }

        View.OnClickListener {
            Logger.i(
                TAG,
                "click to toggle full screen, is in audio mode:${audioFragment.isAudioVisible}"
            )
            if (!audioFragment.isAudioVisible) {
                fullScreenHelper.toggleFullScreen()
            }
        }.apply {
            mBinding.flRoot.setOnClickListener(this)
            mBinding.flContainer.setOnClickListener(this)
        }
    }

    private fun initSmallStream() {
        Logger.i(TAG, "init media list")
        mBinding.cdlSmallStream.bottomSheetCallback = object :
            BottomSheetBehavior.BottomSheetCallback() {
            override fun onStateChanged(bottomSheet: View, newState: Int) {
                when (newState) {
                    BottomSheetBehavior.STATE_COLLAPSED -> {
                        mBinding.ivMediaListArrow.setImageResource(R.drawable.ic_small_stream_arrow_2)
                    }
                    BottomSheetBehavior.STATE_EXPANDED -> {
                        mBinding.ivMediaListArrow.setImageResource(R.drawable.ic_small_stream_arrow_3)
                    }
                    else -> {
                        mBinding.ivMediaListArrow.setImageResource(R.drawable.ic_small_stream_arrow_1)
                    }
                }
            }

            override fun onSlide(bottomSheet: View, slideOffset: Float) {
                if (!mBinding.cdlSmallStream.isMediaListVisible) {
                    mBinding.rvMediaList.alpha = slideOffset * 10f
                    if (isScreenPortrait) {
                        setMainScreenInfoVisible(mainScreenUser != null)
                    }
                } else {
                    mBinding.rvMediaList.alpha = 1f
                    if (isScreenPortrait) {
                        setMainScreenInfoVisible(false)
                    }
                }
                configChangedHelper.translationTitleIfNeed(slideOffset)
            }
        }

        mBinding.rvMediaList.apply {
            layoutManager = LinearLayoutManager(context, LinearLayoutManager.HORIZONTAL, false)
            adapter = pageAdapter
            isNestedScrollingEnabled = false
            addItemDecoration(dividerItemDecoration.apply { setOrientation(RecyclerView.HORIZONTAL) })
            val insetMedium = resources.getDimension(R.dimen.inset_medium).toInt()
            updatePaddingRelative(start = insetMedium)
        }
    }

    private fun setMainScreenInfoVisible(visible: Boolean) {
        if (mBinding.clMainScreenInfo.isInvisible != !visible) {
            Logger.i(TAG, "set main screen info visible:${LogConverter.visible(visible)}")
            mBinding.clMainScreenInfo.isInvisible = !visible
        }
    }

    private fun initNotificationPop() {
        mBinding.notificationPop.setOnClickListener { view, item, position ->
            when (view.id) {
                R.id.btnPositive -> {
                    Logger.i(TAG, "click positive of notification, code:${item.code.ordinal}")
                    when (item.code) {
                        NoticeCode.CODE_PEER_INVITE_MICROPHONE_RECEIVE -> {
                            checkPermissionOnLocalAudioClicked(true)
                            mViewModel.removeNotification(position)
                        }
                        NoticeCode.CODE_PEER_INVITE_CAMERA_RECEIVE -> {
                            onLocalVideoClicked(true)
                            mViewModel.removeNotification(position)
                        }
                        NoticeCode.CODE_ROOM_RTM_DISCONNECTED_PREVIEW -> {
                            mViewModel.changeLocalAudioStatus(true)
                            mViewModel.changeLocalVideoStatus(true)
                            mViewModel.removeNotification(position)
                        }
                        NoticeCode.CODE_AS_ROOM_ASK_HOST_EN,
                        NoticeCode.CODE_AS_ROOM_ASK_HOST_CN -> {
                            item.data?.getString(KEY_REQUEST_ID)?.let {
                                mViewModel.acceptAssistantRequest(it)
                            }
                            mViewModel.removeNotification(position)
                        }
                        NoticeCode.CODE_AS_OPEN_TITLE_EN,
                        NoticeCode.CODE_AS_OPEN_TITLE_CN -> {
                            mViewModel.enableAssist(false)
                        }
                    }
                }
                R.id.btnNegative -> {
                    Logger.i(TAG, "click negative of notification, code:${item.code.ordinal}")
                    when (item.code) {
                        NoticeCode.CODE_PEER_INVITE_MICROPHONE_RECEIVE -> {
                            mainViewModel?.disagreeOpenLocalAudio()
                        }
                        NoticeCode.CODE_PEER_INVITE_CAMERA_RECEIVE -> {
                            mainViewModel?.disagreeOpenLocalVideo()
                        }
                        NoticeCode.CODE_ROOM_RTM_DISCONNECTED_PREVIEW -> {
                            mViewModel.changeLocalAudioStatus(false)
                            mViewModel.changeLocalVideoStatus(false)
                        }
                        NoticeCode.CODE_AS_ROOM_ASK_HOST_EN,
                        NoticeCode.CODE_AS_ROOM_ASK_HOST_CN -> {
                            item.data?.getString(KEY_REQUEST_ID)?.let {
                                mViewModel.refuseAssistantRequest(it)
                            }
                        }
                        NoticeCode.CODE_AS_OPEN_TITLE_EN,
                        NoticeCode.CODE_AS_OPEN_TITLE_CN -> {
                            mViewModel.disableAssist()
                        }
                    }
                    mViewModel.removeNotification(position)
                }
                R.id.ivClose -> {
                    mViewModel.removeNotification(position)
                    Logger.i(
                        TAG,
                        "click close of notification, notificationType:${item?.notificationType}"
                    )
                    if (item.notificationType == MeetingNotificationType.NO_1) {
                        mViewModel.closeNetworkPrompt()
                    }
                }
                R.id.tvMessage -> {
                    Logger.i(TAG, "click link of notification, code:${item.code.ordinal}")
                    mViewModel.removeNotification(position)
                    mViewModel.tryNetworkPromptAction()
                }
                else -> {
                }
            }

        }
        mBinding.notificationPop.setOnItemSwipedDismissListener { item, position ->
            mViewModel.removeNotification(position)
            Logger.i(
                TAG,
                "click close of notification, notificationType:${item.notificationType}"
            )
            if (item.notificationType == MeetingNotificationType.NO_1) {
                mViewModel.closeNetworkPrompt()
            }
        }
    }

    override fun onBackPressed() {
        if (localUser?.isCloudRecording == true) {
            showKeepRecordingConfirmDialog()
        } else {
            showExitConfirmDialog()
        }
    }

    override fun onClick(v: View) {
        if (audioFragment.isAudioVisible) {
            return
        }
        when (v.id) {
            R.id.llTitle -> {
                if (room?.isInternal() == true && localUser?.isThirdPartyLoggedIn == false) {
                    return
                }
                showRoomInfoPop()
            }
            R.id.ivSignal -> {
                showSignalPop()
            }
            R.id.flBugReport -> {
                clickBugReport()
            }
            R.id.flSetting -> {
                safeNavigate(R.id.action_meeting_to_room_settings)
            }
            R.id.ivMediaListArrow -> {
                mBinding.cdlSmallStream.turnToHalfExpanded()
            }
            R.id.flInvite -> {
                safeNavigate(R.id.action_meeting_to_address_book_container)
            }
        }
    }

    private fun clickBugReport() {
        Logger.i(TAG, "click bug report, button status:${DumpStatus.convertToLog(dumpStatus)}")
        when (dumpStatus) {
            DumpStatus.PREPARE -> {
                mViewModel?.startAudioDumping()
            }
            DumpStatus.DUMPING -> {
                mViewModel?.stopAudioDumping()
            }
            else -> {
                if ((activity as? AgoraActivity)?.getCurrentFragmentId() != R.id.bugReport) {
                    safeNavigate(R.id.action_global_bug_report)
                }
            }
        }
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)
        Logger.i(
            TAG,
            "receive a requestCode:$requestCode,resultCode:$resultCode from activityResult"
        )
        if (requestCode == ScreenShareManager.CODE_SCREEN_CAPTURE_NO_WATERMARK_REQUEST ||
            requestCode == ScreenShareManager.CODE_SCREEN_CAPTURE_WATERMARK_REQUEST
        ) {
            requestingScreenShare = false
        }
        screenShareManager.onActivityResult(requestCode, resultCode, data)
    }

    override fun allocObserver() {
        mViewModel?.attendeesLiveData?.observe(this, { it ->
            it.isNotEmpty().let { _isNotEmpty ->
                if (isMediaListNotEmpty != _isNotEmpty) {
                    isMediaListNotEmpty = _isNotEmpty
                    mBinding.cdlSmallStream.setVisible2(_isNotEmpty)
                }
            }
            pageAdapter.submitList(it)
        })

        mViewModel?.mainUserUpdateEvent?.observe(this, {
            updateMainScreen(it)
        })

        mViewModel?.meetingDurationEvent?.observe(this, {
            mBinding.tvTime.text = it
        })

        mViewModel?.userAmountChangedLiveData?.observe(this, {
            mBinding.bottomBar.setAttendeesNumber(it)
            openAudioConfirm = it >= OPEN_AUDIO_NEED_CONFIRM_ATTENDEES_NUMBER
        })

        mViewModel?.unReadMessageChangedLiveData?.observe(this, {
            hasUnreadMessage = it > 0
            mBinding.bottomBar.setHasUnreadMessage(hasUnreadMessage)
        })

        systemWindowHelper.observeUnReadMessageChanged(
            this,
            mViewModel?.unReadMessageChangedLiveData
        )

        systemWindowHelper.observeVideoRating(
            this,
            mViewModel?.videoRatingChangedLiveData,
            mViewModel?.videoRatingTimingLiveData
        )

        systemWindowHelper.observePopup(this, mViewModel?.popupLiveData) {
            isScreenPortrait
        }

        mViewModel?.audioDumpingChangedLiveData?.observe(this, {
            Logger.i(TAG, "audio dumping changed event, status:$it")
            if (!it) {
                removeStatus(MeetingStatus.ITEM_TYPE_DUMP)
            } else {
                showStatus(
                    MeetingStatus(
                        itemType = MeetingStatus.ITEM_TYPE_DUMP,
                        description = getString(R.string.dump_reporting_status)
                    )
                )
            }
        })

        mViewModel?.cloudRecordInfoChangedLiveData?.observe(this, { it ->
            if (it == null) {
                if (isCloudRecording) {
                    Logger.i(TAG, "stop cloud recording")
                    isCloudRecording = false
                    removeStatus(MeetingStatus.ITEM_TYPE_CLOUD_RECORDING)
                }
            } else {
                if (!isCloudRecording) {
                    Logger.i(TAG, "start cloud recording, streamId:${it.user.streamId}")
                    isCloudRecording = true
                    showStatus(
                        MeetingStatus(
                            itemType = MeetingStatus.ITEM_TYPE_CLOUD_RECORDING,
                            description = getString(R.string.cloud_recording_status),
                            duration = it.recordTime
                        )
                    )
                } else {
                    mBinding.statusPop.updateDuration(it.recordTime)
                }
            }
        })

        mViewModel?.screenShareChangedLiveData?.observe(this) {
            Logger.i(TAG, "screen share changed event, isSharing:$it")
            isScreenSharing = it
            setupScreenShare(it)
        }

        mViewModel?.localUserStatusChanged?.observe(this, { user ->
            localUser = user
            mBinding.watermark.setMarkText(user.getConferenceNicknameMaybeAssistant())
            setupLocalMediaStatus(user)
            setupMenuPopupWindow(isScreenSharing)
            setupInvitationStatus()
        })

        mViewModel?.screenShareLiveData?.observeForever(screenShareObserver)

        mViewModel?.bizConnectChangedLiveData?.observe(this, {
            Logger.i(TAG, "biz connect changed event, connected:$it")
            bizConnected = it
            mBinding.ivSignal.bizConnected = it
        })

        mViewModel?.kickOutEvent?.observeForever(kickOutObserver)

        mViewModel?.audioDumpTimer?.observe(this, {
            Log.d(TAG, "audioDumpTimer event, progress = $it")
            mBinding.flDumpProgressBar.progress = it
        })

        mViewModel?.roomInfoChanged?.observe(this, { room ->
            this.room = room
            showConnectingIfNeed(room)
            mBinding.tvTitle.text = room.name
            mBinding.bottomBar.refreshData(
                user = localUser,
                isAgoraRoom = room.isInternal(),
                isSharingScreen = screenShareManager.isSharingScreen()
            )
            setupInvitationStatus()
        })

        mViewModel?.audioDumpStatusChangedEvent?.observe(this, {
            this.dumpStatus = it
            setAudioDumpStatus(it)
        })

        mViewModel?.networkSignalChangedLiveData?.observe(this) {
            setupNetQuality(it.quality, it.bizConnected)
        }

        systemWindowHelper.observeNetworkSignalChanged(
            this,
            mViewModel?.networkSignalChangedLiveData
        )

        mViewModel?.appInBackgroundNotificationChangedEvent?.observeForever(
            appInBackgroundNotificationChangedObserver
        )

        mViewModel?.developerChangedLiveData?.observe(this) {
            Logger.i(TAG, "developer changed event, developerOptions:$it")
            developerOptions = it
            pageAdapter.showVideoData.postValue(it.showVideoData)
            mainScreenUser?.let { _mainScreenUser ->
                setupMediaStatics(_mainScreenUser)
            }
        }

        mainViewModel?.qualityDialogVisible?.observe(this, {
            Logger.i(TAG, "quality dialog visible event:${LogConverter.visible(it)}")
            isQualityReportShowing = it
            if (it && isFullScreen()) {
                fullScreenHelper.isFullScreen = false
            }
        })

        mViewModel?.notificationLiveData?.observe(this) {
            mBinding.notificationPop.submitList(it)
        }
    }

    private fun showConnectingIfNeed(room: Room?) {
        room?.let {
            if (it.rtmJoinedBefore) {
                connectingView.hide {
                    if (!isInPictureInPictureMode()) {
                        mBinding.callLayout.isVisible = true
                    }
                }
            } else if (it.isInternal() && !connectingView.isShowing()) {
                connectingView.show(mBinding.flRoot)
                mBinding.callLayout.isVisible = false
            }
        }
    }

    private fun setupInvitationStatus() {
        mBinding.flInvite.isVisible =
            room?.isInternal() == true
                    && localUser?.isThirdPartyLoggedIn == true
                    && (localUser?.source == UserSource.AGORAN.value || localUser?.source == UserSource.EMAIL.value)
    }

    private fun setupScreenShare(it: Boolean) {
        Logger.i(TAG, "setup screen share: $it")
        mBinding.bottomBar.setScreenShare(it)
        setupMenuPopupWindow(isScreenSharing)
    }

    private fun setupMenuPopupWindow(isScreenSharing: Boolean) {
        systemWindowHelper.setupMenuPopupWindow(
            isScreenSharing = isScreenSharing,
            isThirdPartyLoggedIn = localUser?.isThirdPartyLoggedIn == true,
            isCloudRecording = localUser?.isCloudRecording == true,
            isInternal = room?.isInternal() == true
        )
    }

    private fun setAudioDumpStatus(@DumpStatus dumpStatus: Int) {
        Logger.i(TAG, "set audio dump status, status:${DumpStatus.convertToLog(dumpStatus)}")
        if (dumpStatus == DumpStatus.PREPARE) {
            mBinding.flDumpProgressBar.progress = 0
        }
        if (dumpStatus == DumpStatus.PREPARE || dumpStatus == DumpStatus.DUMPING) {
            mBinding.ivBugReport.isVisible = false
            mBinding.flDumpProgressBar.isVisible = true
            if (dumpStatus == DumpStatus.PREPARE) {
                mBinding.flDumpProgressBar.doOnPreDraw { showDumpTipPop() }
            }
        } else {
            mBinding.ivBugReport.isVisible = true
            mBinding.flDumpProgressBar.isVisible = false
        }
    }

    override fun onStart() {
        super.onStart()
        activity?.window?.let {
            WindowCompat.setDecorFitsSystemWindows(it, false)
            it.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_ALWAYS_HIDDEN)
        }
        if (!isInPictureInPictureMode()) {
            fullScreenHelper.isFullScreen = false
        }
        mViewModel?.startCountTime()
    }

    override fun onPause() {
        super.onPause()
        systemWindowHelper.dismissPops()
        screenShareManager.dismissDialog()
    }

    override fun onDestroyView() {
        mBinding.callLayout.setOnApplyWindowInsetsListener(null)
        NavHostFragment.findNavController(this@MeetingFragment)
            .removeOnDestinationChangedListener(destinationChangedListener)
        super.onDestroyView()
    }

    override fun onDestroy() {
        pageAdapter.destroy()
        requireActivity().window.clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)
        mViewModel?.cleanMainScreenUser()
        mViewModel?.kickOutEvent?.removeObserver(kickOutObserver)
        mViewModel?.screenShareLiveData?.removeObserver(screenShareObserver)
        mViewModel?.appInBackgroundNotificationChangedEvent?.removeObserver(
            appInBackgroundNotificationChangedObserver
        )
        screenShareManager.unbindShareScreenService()
        super.onDestroy()
    }

    override fun onDetach() {
        super.onDetach()
        pictureHelper.onDestroy()
        systemWindowHelper.onDestroy()
        configChangedHelper.onDestroy()
        fullScreenHelper.onDestroy()
        activity?.let {
            it.window?.decorView?.setOnSystemUiVisibilityChangeListener(null)
        }
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        Logger.i(
            TAG,
            "onConfigurationChanged: orientation = ${if (newConfig.orientation == Configuration.ORIENTATION_PORTRAIT) "portrait" else "land"}," +
                    " isInPictureInPictureMode = ${isInPictureInPictureMode()}"
        )

        isScreenPortrait = newConfig.orientation == Configuration.ORIENTATION_PORTRAIT

        if (isInPictureInPictureMode()) {
            return
        }

        dismissPops()

        configChangedHelper.onConfigurationChanged(newConfig)

        audioFragment.onConfigurationChanged(newConfig)

        if (isScreenPortrait) {
            setMainScreenInfoVisible(
                !mBinding.cdlSmallStream.isMediaListVisible
                        && mainScreenUser != null
            )
        }

        mBinding.callLayout.doOnLayout {
            fullScreenHelper.translationViews(
                1f,
                isScreenPortrait = { isScreenPortrait }
            )
        }

        if (mainScreenUser?.hasWatermark == true) {
            mBinding.flRoot.doOnLayout {
                mainScreenUser?.let { _mainScreenUser ->
                    resizeWatermark(_mainScreenUser)
                }
            }
        }
    }

    override fun onPictureInPictureModeChanged(isInPictureInPictureMode: Boolean) {
        Logger.i(
            TAG,
            "onPictureInPictureModeChanged: isInPictureInPictureMode = $isInPictureInPictureMode"
        )
        pictureHelper.onPictureInPictureModeChanged(isInPictureInPictureMode)

        if (connectingView.isShowing()) {
            connectingView.isVisible = !isInPictureInPictureMode
            return
        }

        if (isInPictureInPictureMode) {
            mBinding.callLayout.isInvisible = true
            systemWindowHelper.dismissPops()
            screenShareManager.dismissDialog()
            mBinding.flContainer.scaleToOrigin()
        } else {
            mBinding.callLayout.isInvisible = false
        }

        mBinding.flRoot.screenFullEnable(!isInPictureInPictureMode)
    }

    fun onUserLeaveHint() {
        val supportPip = DeviceUtils.isSupportPicInPic()
        val wantToBePip = iWantToBeInPipModeNow()
        val versionFit = Build.VERSION.SDK_INT >= Build.VERSION_CODES.O
        Logger.i(TAG, "on user leave hint, support:$supportPip, want:$wantToBePip, fit:$versionFit")
        if (wantToBePip && versionFit && supportPip) {
            mBinding.callLayout.isInvisible = true
            if (!pictureHelper.enterPictureInPicture()) {
                Logger.e(TAG, "failed to start pip mode")
                mBinding.callLayout.isInvisible = false
            }
        }
    }

    private fun setupMainScreen(user: ARoomUser) {
        Logger.i(TAG, "setup main screen, user:${user.logString()}")
        setupMainVideo(user)
        setupMainAudio(user)
        setupMainWatermark(user)
        setupMainShareIcon(user)
        setupMediaStatics(user)
        setupMainUserName(user)
        setMainScreenInfoVisible(!mBinding.cdlSmallStream.isMediaListVisible)
        pictureHelper.changeAspectRatioInPicMode(user)
    }

    private fun setupMainVideo(user: ARoomUser) {
        Logger.i(
            TAG,
            "setup main video, streamId:${user.streamId}, video:${LogConverter.muted(!user.videoState)}"
        )
        if (user.videoState) {
            openMainScreenVideo(user)
        } else {
            muteMainScreenVideo(user)
        }
    }

    @Synchronized
    private fun updateMainScreen(newUser: ARoomUser) {
        if (mainScreenUser == null) {
            Logger.i(TAG, "initialize mainScreenUser")
            this.mainScreenUser = newUser
            setupMainScreen(newUser)
            return
        }

        if (mainScreenUser === newUser) {
            Logger.i(TAG, "do not refresh because the old and new users are the same")
            return
        }

        if (mainScreenUser?.areTheSame(newUser) == false) {
            Logger.i(TAG, "reset mainScreenUser")
            this.mainScreenUser = newUser
            setupMainScreen(newUser)
            return
        }

        if (mainScreenUser?.audioState != newUser.audioState) {
            Logger.i(TAG, "main user:${mainScreenUser?.streamId} audio status changed")
            setupMainAudio(newUser)
        }

        if (mainScreenUser?.volume != newUser.volume) {
            setupMainVolume(newUser)
        }

        if (mainScreenUser?.videoState != newUser.videoState) {
            setupMainVideo(newUser)
        }

        if (mainScreenUser?.avatar != newUser.avatar) {
            setupMainScreenAvatar(newUser)
        }

        if (mainScreenUser?.hasWatermark != newUser.hasWatermark
            || mainScreenUser?.width != newUser.width
            || mainScreenUser?.height != newUser.height
        ) {
            Logger.i(
                TAG,
                "main user:${mainScreenUser?.streamId} watermark changed,watermark:${newUser.hasWatermark}" +
                        ", oldW=${mainScreenUser?.width}, oldH=${mainScreenUser?.height}" +
                        ", newW=${newUser.width}, newH=${newUser.height}"
            )
            setupMainWatermark(newUser)
        }

        if (mainScreenUser?.volume != newUser.volume ||
            mainScreenUser?.width != newUser.width ||
            mainScreenUser?.height != newUser.height ||
            mainScreenUser?.fps != newUser.fps ||
            mainScreenUser?.bitrate != newUser.bitrate ||
            mainScreenUser?.streamType != newUser.streamType ||
            mainScreenUser?.rotation != newUser.rotation
        ) {
            setupMediaStatics(newUser)
        }

        if (mainScreenUser?.quality != newUser.quality) {
            setupMainLastMile(newUser)
        }

        if (mainScreenUser?.isSpeaking != newUser.isSpeaking) {
            setupMainAudio(newUser)
        }

        if (mainScreenUser?.parentStreamId != newUser.parentStreamId) {
            Logger.i(TAG, "main user:${mainScreenUser?.streamId} shareId changed")
            setupMainShareIcon(newUser)
        }

        if (mainScreenUser?.name != newUser.name) {
            Logger.i(TAG, "main user:${mainScreenUser?.streamId} name changed")
            setupMainUserName(newUser)
        }

        if (mainScreenUser?.thirdPartyName != newUser.thirdPartyName) {
            Logger.i(TAG, "main user:${mainScreenUser?.streamId} third party name changed")
            setupMainUserName(newUser)
        }

        if (mainScreenUser?.isAssistant != newUser.isAssistant) {
            Logger.i(TAG, "main user:${mainScreenUser?.streamId} assistant identity changed")
            setupMainUserName(newUser)
        }

        this.mainScreenUser = newUser
    }

    private fun setupMainWatermark(user: ARoomUser) {
        mBinding.watermark.isVisible = user.hasWatermark
        if (user.hasWatermark) {
            resizeWatermark(user)
        }
    }

    private fun setupMainLastMile(user: ARoomUser) {
        mBinding.signalIcon.isVisible = (
                !user.isMySelf()
                        && user.quality != NetworkQuality.EXCELLENT
                        && user.quality != NetworkQuality.GOOD
                        && user.quality != NetworkQuality.POOR
                )
    }

    private fun setupMainUserName(user: ARoomUser) {
        user.getConferenceNicknameMaybeAssistant().let {
            Logger.i(TAG, "setup main username, username = $it")
            mBinding.tvName.text = it
        }
    }

    private fun setupMainShareIcon(user: ARoomUser) {
        Logger.i(TAG, "setup main share icon, isShareStream:${user.isShareStream()}")
        mBinding.ivLabelScreenShare.isVisible = user.isShareStream()
        mBinding.ivSpeak.isInvisible = user.isShareStream()
    }

    private fun setupMainAudio(user: ARoomUser) {
        if (mBinding.ivSpeak.mute != !user.audioState) {
            Logger.i(TAG, "setup main audio, audio:${LogConverter.muted(!user.audioState)}")
        }
        mBinding.ivSpeak.mute = !user.audioState
        mBinding.ivSpeak.isMySelf = user.isMySelf()
        if (user.isMySelf()) {
            setupMainVolume(user)
        } else {
            mBinding.ivSpeak.speaking = user.isSpeaking
        }
    }

    private fun setupMainVolume(user: ARoomUser) {
        mBinding.ivSpeak.volume = user.volume
    }

    private fun setupMediaStatics(user: ARoomUser) {
        if (developerOptions?.showVideoData == true) {
            mBinding.mediaStatics.setVideoStatics(user.getVideoStats())
            mBinding.mediaStatics.setVolume(user.volume)
            mBinding.mediaStatics.visibility = View.VISIBLE
        } else {
            mBinding.mediaStatics.visibility = View.GONE
        }
    }

    private fun setupLocalMediaStatus(user: LocalUser) {
        mBinding.bottomBar.refreshData(
            user = user,
            isAgoraRoom = room?.isInternal() == true,
            isSharingScreen = screenShareManager.isSharingScreen()
        )
        setupNetQuality(user.quality)
    }

    private fun setupNetQuality(quality: Int, bizConnected: Boolean? = null) {
        mBinding.ivSignal.setQuality(quality)
        if (bizConnected != null) {
            mBinding.ivSignal.bizConnected = bizConnected
        }
    }

    private fun openMainScreenVideo(user: ARoomUser) {
        Logger.i(TAG, "unMute user:${user.streamId} video in main-screen")
        mBinding.flContainer.removeAllViews()
        user.streamId.let { streamId ->
            mainScreenVideo = if (user !is LocalUser) {
                mViewModel?.createRemoteVideo(streamId)
            } else {
                mViewModel?.createLocalVideo(0)
            }
        }
        mainScreenVideo?.let {
            mBinding.flContainer.addView(it)
        }
        mBinding.flContainer.isVisible = true
        mBinding.bigStreamDefault.isVisible = false

        context?.let {
            mViewModel?.setBackgroundColor(
                user.streamId,
                ContextCompat.getColor(it, R.color.background_gray)
            )
        }
    }

    private fun muteMainScreenVideo(user: ARoomUser) {
        Logger.i(TAG, "mute user:${user.streamId} video in main-screen, video:${user.videoState}")
        mBinding.flContainer.removeAllViews()
        user.streamId.let { streamId ->
            mViewModel?.clearRemoteVideo(streamId)
            mainScreenVideo = null
        }
        mBinding.flContainer.isVisible = false
        mBinding.bigStreamDefault.isVisible = true

        setupMainScreenAvatar(user)
    }

    private fun setupMainScreenAvatar(user: ARoomUser) {
        context?.let { _context ->
            mBinding.ivBigStreamDefault?.let { _imageView ->
                Glide.with(_context)
                    .load(user.avatar)
                    .centerCrop()
                    .placeholder(R.drawable.ic_head_portrait)
                    .into(_imageView)
            }
        }
    }

    private fun setOnApplyWindowInsetsListener() {
        mBinding.callLayout.setOnApplyWindowInsetsListener { _, it ->
            val windowInsetsCompat = WindowInsetsCompat.toWindowInsetsCompat(it)

            val insets =
                windowInsetsCompat.systemWindowInsets
            systemWindowInsetTop = insets.top
            systemWindowInsetLeft = insets.left
            systemWindowInsetRight = insets.right
            systemWindowInsetBottom = insets.bottom

            Logger.i(TAG, "on apply window insets:$insets")

            connectingView.onApplyWindowInsets(
                systemWindowInsetLeft,
                systemWindowInsetTop
            )

            val insetAction = {
                if (mBinding != null) {
                    mBinding.insetTopView.setGuidelineBegin(systemWindowInsetTop)
                    mBinding.insetLeftView.setGuidelineBegin(systemWindowInsetLeft)
                    mBinding.insetRightView.setGuidelineEnd(systemWindowInsetRight)
                    mBinding.insetBottomView.setGuidelineEnd(systemWindowInsetBottom)
                }
            }

            if (mBinding.insetTopView.isSafeToRequestDirectly()) {
                insetAction.invoke()
            } else {
                mBinding.insetTopView.post {
                    insetAction.invoke()
                }
            }

            audioFragment.dispatchApplyWindowInsets(insets)

            mBinding.bottomBar.updatePaddingRelative(bottom = systemWindowInsetBottom)

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                WindowInsets.CONSUMED
            } else {
                it.consumeSystemWindowInsets()
            }
        }
    }

    private fun onLocalVideoClicked(isNotification: Boolean) {
        Logger.i(TAG, "Trigger local video click")
        requestingPermissions = true
        PermissionUtils
            .permission(PermissionConstants.CAMERA)
            .callback(object : PermissionUtils.FullCallback {
                override fun onGranted(permissionsGranted: MutableList<String>) {
                    if (PermissionDao.isAllGranted(permissionsGranted, getRequestPermission())) {
                        if (isNotification) {
                            mainViewModel?.agreeOpenLocalVideoByRequest()
                        } else {
                            mViewModel?.changeLocalVideoStatus(!mBinding.bottomBar.isLocalVideoOn())
                            mViewModel?.removeNotification(NoticeCode.CODE_PEER_INVITE_CAMERA_RECEIVE)
                        }
                    }
                    requestingPermissions = false
                }

                private fun getRequestPermission(): Array<String> {
                    return arrayOf(Manifest.permission.CAMERA)
                }

                override fun onDenied(
                    permissionsDeniedForever: MutableList<String>,
                    permissionsDenied: MutableList<String>
                ) {
                    mainViewModel?.onPermissionDenied(permissionsDeniedForever, permissionsDenied)
                    requestingPermissions = false
                }
            })
            .request()
    }

    private fun onLocalAudioClicked() {
        Logger.i(TAG, "Trigger local audio click")
        if (openAudioConfirm && !mBinding.bottomBar.isLocalAudioOn()) {
            systemWindowHelper.showOpenAudioConfirmDialog {
                checkPermissionOnLocalAudioClicked(false)
            }
        } else {
            checkPermissionOnLocalAudioClicked(false)
        }
    }

    private fun checkPermissionOnLocalAudioClicked(isNotification: Boolean) {
        requestingPermissions = true
        PermissionUtils
            .permission(PermissionConstants.MICROPHONE)
            .callback(object : PermissionUtils.FullCallback {
                override fun onGranted(permissionsGranted: MutableList<String>) {
                    if (PermissionDao.isAllGranted(permissionsGranted, getRequestPermission())) {
                        if (isNotification) {
                            mainViewModel?.agreeOpenLocalAudio()
                        } else {
                            mViewModel?.changeLocalAudioStatus(!mBinding.bottomBar.isLocalAudioOn())
                            mViewModel?.removeNotification(NoticeCode.CODE_PEER_INVITE_MICROPHONE_RECEIVE)
                        }
                    }
                    requestingPermissions = false
                }

                private fun getRequestPermission(): Array<String> {
                    return arrayOf(Manifest.permission.RECORD_AUDIO)
                }

                override fun onDenied(
                    permissionsDeniedForever: MutableList<String>,
                    permissionsDenied: MutableList<String>
                ) {
                    mainViewModel?.onPermissionDenied(permissionsDeniedForever, permissionsDenied)
                    requestingPermissions = false
                }
            })
            .request()
    }

    private fun clickCameraSwitch() {
        Logger.i(TAG, "Trigger camera switch click")
        mViewModel?.switchCamera()
    }

    private fun resizeWatermark(user: ARoomUser) {
        Logger.i(TAG, "resizeWatermark, videoWidth = ${user.width}, videoHeight = ${user.height}")
        mBinding.watermark.updateLayoutParams {
            val updateWatermarkSize = {
                val srcScale = user.width.toFloat() / user.height.toFloat()
                val dstScale =
                    mBinding.flRoot.width.toFloat() / mBinding.flRoot.height.toFloat()
                if (srcScale > dstScale) {
                    width = mBinding.flRoot.width
                    height = (width / srcScale).toInt()
                } else {
                    height = mBinding.flRoot.height
                    width = (height * srcScale).toInt()
                }
            }
            if (mBinding.flRoot.width == 0) {
                Logger.d(TAG, "resizeWatermark, update watermark size, doOnLayout")
                mBinding.flRoot.doOnLayout {
                    updateWatermarkSize()
                }
            } else {
                Logger.d(TAG, "resizeWatermark, update watermark size direct")
                updateWatermarkSize()
            }
        }
    }

    private fun iWantToBeInPipModeNow(): Boolean {
        if (mViewModel?.pipModeEnable() != true
            || requestingPermissions
            || requestingScreenShare
            || (activity as? AgoraActivity)?.getCurrentFragmentId() != R.id.meeting
            || mainScreenUser?.videoState == false
            || audioFragment.isAudioVisible
        ) {
            Logger.i(TAG, "picture in picture enable:${LogConverter.enable(false)}")
            return false
        }
        Logger.i(TAG, "picture in picture enable:${LogConverter.enable(true)}")
        return true
    }

    private fun clickCloudRecording(open: Boolean) {
        showCloudRecordingConfirmDialog(open)
    }

    private fun audioModeOnBackPress() {
        if (audioFragment.isAudioVisible) {
            mBinding.dragDownLayout.reset()
            onAudioShowOrHide(false)
        } else {
            onBackPressed()
        }
    }

    private fun onAudioShowOrHide(show: Boolean) {
        Logger.i(TAG, "on audio show or hide, audio page is {${LogConverter.visible(show)}}")
        audioFragment.isAudioVisible = show
        if (show) {
            screenShareManager.stop()
        }
        fullScreenHelper.isFullScreen = false

        mBinding.cdlSmallStream.setVisible2(pageAdapter.itemCount > 0)
        mBinding.bottomBar.touchEnable = !show
    }

    private fun bottomBarClick(it: View) {
        if (audioFragment.isAudioVisible) {
            return
        }
        when (it.id) {
            R.id.flCameraSwitch -> {
                clickCameraSwitch()
            }
            R.id.ivLocalAudio,
            R.id.flLocalAudio -> {
                onLocalAudioClicked()
            }
            R.id.ivLocalVideo -> {
                if (mBinding.bottomBar.isLocalVideoOn()) {
                    showMediaVideoPop()
                } else {
                    onLocalVideoClicked(false)
                }
            }
            R.id.flLocalVideo -> {
                onLocalVideoClicked(false)
            }
            R.id.flScreenShare -> {
                doScreenShare(!mBinding.bottomBar.isScreenSharing())
            }
            R.id.flCloudRecording -> {
                clickCloudRecording(!mBinding.bottomBar.isCloudRecording())
            }
            R.id.ivAttendees,
            R.id.flAttendees -> {
                safeNavigate(R.id.action_meeting_to_attendees)
            }
            R.id.flChat -> {
                safeNavigate(R.id.action_meeting_to_chat)
            }
            R.id.flHangup -> {
                userHangup()
            }
            R.id.flMenu -> {
                showMenuPop()
            }
        }
    }

    private fun userHangup() {
        Logger.i(TAG, "user hang up the meeting")
        if (localUser?.isCloudRecording == true) {
            showKeepRecordingConfirmDialog()
        } else {
            mViewModel?.leaveChannel()
        }
    }

    private fun doScreenShare(open: Boolean) {
        if (mViewModel?.hasScreenSharePermission() == true) {
            try {
                requestingScreenShare = true
                screenShareManager.applyScreenShare(
                    open,
                    room?.isInternal() ?: false,
                    mViewModel?.getEnableWaterMark() ?: false
                )
            } catch (e: ActivityNotFoundException) {
                requestingScreenShare = false
                ToastUtils.showShort(getString(R.string.screen_share_not_support))
                Logger.e(TAG, "Didn't find MediaProjectionPermissionActivity")
            }
        } else {
            mViewModel?.notifyNotScreenSharePermission()
            Logger.e(TAG, "clickCloudRecording: not screen share permission!")
        }
    }

    private fun showStatus(meetingStatus: MeetingStatus) {
        mBinding.statusPop.showStatus(meetingStatus)
    }

    private fun removeStatus(itemType: Int) {
        mBinding.statusPop.removeStatus(itemType)
    }

    private fun showSignalPop() {
        systemWindowHelper.showSignalPop()
    }

    private fun dismissPops() {
        systemWindowHelper.dismissPops()
    }

    private fun showMenuPop() {
        systemWindowHelper.showMenuPop(
            isSharingScreen = screenShareManager.isSharingScreen(),
            localUser = localUser,
            room = room,
            developerOptions = developerOptions,
            hasUnreadMessage = hasUnreadMessage,
            callback = object : RoomMenuPopupWindow.MenuCallback {

                override fun onCloudRecordingChange(view: View, open: Boolean) {
                    clickCloudRecording(open)
                }

                override fun onShareScreenStateChange(view: View, open: Boolean) {
                    doScreenShare(open)
                }

                override fun onClickChat() {
                    safeNavigate(R.id.action_meeting_to_chat)
                }

                override fun onVideoRatingClicked() {
                    mainViewModel?.toggleVideoRating()
                }

                override fun onQualityReportClicked() {
                    mainViewModel?.collectQualityInfo(mainScreenUser?.streamId)
                }
            }
        )
    }

    private fun showCloudRecordingConfirmDialog(start: Boolean) {
        systemWindowHelper.showCloudRecordingConfirmDialog(start) {
            if (start) {
                Logger.i(TAG, "The user clicked start cloud recording button")
                mViewModel?.startCloudRecording()
            } else {
                Logger.i(TAG, "The user clicked stop cloud recording button")
                mViewModel?.stopCloudRecording()
            }
        }
    }

    private fun showKeepRecordingConfirmDialog() {
        systemWindowHelper.showKeepRecordingConfirmDialog { mViewModel?.leaveChannel() }
    }

    private fun showMediaVideoPop() {
        systemWindowHelper.showMediaVideoPop(
            onCloseVideo = { mViewModel?.changeLocalVideoStatus(false) },
            onConvertCamera = { mViewModel?.switchCamera() }
        )
    }

    private fun showRoomInfoPop() {
        systemWindowHelper.showRoomInfoPop(room) {
            mainViewModel?.createShareLink()
        }
    }

    private fun showDumpTipPop() {
        systemWindowHelper.showDumpTipPop()
    }

    private fun showExitConfirmDialog() {
        systemWindowHelper.showExitConfirmDialog(
            room,
            onNegativeClick = { Logger.i(TAG, "user cancel leave room") },
            onPositiveClick = {
                Logger.i(TAG, "user confirm leave room")
                mViewModel?.leaveChannel()
            })
    }

    private fun SmallStreamCoordinatorLayout.setVisible2(visible: Boolean) {
        Logger.i(
            TAG,
            "set small stream visible, visible:${LogConverter.visible(visible)}, audioFragment#isAudioVisible:${audioFragment.isAudioVisible}, isCollapsed:${isCollapsed()}"
        )
        isVisible = visible && !audioFragment.isAudioVisible
        if (isVisible
            && isScreenPortrait
            && (isHalfExpanded() || isExpanded())
        ) {
            setMainScreenInfoVisible(false)
        } else {
            setMainScreenInfoVisible(true)
        }
    }

    companion object {
        private const val TAG = "[UI][Meeting]"

        //The minimum number of participants that open audio needs to confirm
        const val OPEN_AUDIO_NEED_CONFIRM_ATTENDEES_NUMBER = 20
    }
}

