package io.agora.avc.app.meeting

import android.animation.ValueAnimator
import android.app.Activity
import androidx.core.animation.addListener
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.marginBottom
import io.agora.avc.databinding.FragmentMeetingBinding
import io.agora.avc.extensions.setNavBarVisibility
import io.agora.avc.extensions.setSystemBarVisibility
import io.agora.logger.Logger

/**
 * helper of meeting full screen switch
 */
class MeetingFullScreenHelper(
    var activity: Activity?,
    var fragment: MeetingFragment?
) {

    lateinit var mBinding: FragmentMeetingBinding

    var isFullScreen: Boolean = false
        set(value) {
            field = value
            if (value) {
                turnFullScreen()
            } else {
                turnNormalScreen()
            }
        }

    private val toggleScreenAnimation: ValueAnimator by lazy {
        ValueAnimator.ofFloat(0f, 1f)
    }

    fun bind(
        binding: FragmentMeetingBinding,
        isScreenPortrait: () -> Boolean
    ) {
        mBinding = binding
        initToggleScreenAnimatorListener(isScreenPortrait)
    }

    fun toggleFullScreen() {
        isFullScreen = !isFullScreen
    }

    private fun turnNormalScreen() {
        Logger.i(TAG, "turn normal screen")

        toggleScreenAnimation.cancel()
        toggleScreenAnimation.start()
    }

    private fun turnFullScreen() {
        Logger.i(TAG, "turn full screen")

        toggleScreenAnimation.cancel()
        toggleScreenAnimation.start()
    }

    private fun initToggleScreenAnimatorListener(
        isScreenPortrait: () -> Boolean
    ) {
        toggleScreenAnimation.apply {
            duration = 150L
            addUpdateListener {
                translationViews(
                    it.animatedValue as Float,
                    isScreenPortrait = isScreenPortrait
                )
            }
            addListener(
                onStart = {
                    activity?.let {
                        if (!isFullScreen) {
                            if (isScreenPortrait.invoke()) {
                                it.window.setSystemBarVisibility(WindowInsetsCompat.Type.systemBars(), true)
                            } else {
                                it.setNavBarVisibility(true)
                            }
                        }
                    }
                },
                onEnd = {
                    activity?.let {
                        if (isFullScreen) {
                            if (isScreenPortrait.invoke()) {
                                it.window.setSystemBarVisibility(WindowInsetsCompat.Type.systemBars(), false)
                            } else {
                                it.setNavBarVisibility(false)
                            }
                        }
                    }
                }
            )
        }
    }

    fun translationViews(
        animatedValue: Float,
        isScreenPortrait: () -> Boolean
    ) {
        (if (isFullScreen) animatedValue else (1 - animatedValue)).let { _animatorValue ->
            (mBinding.llTitle.height + (fragment?.getSystemWindowInsetTop() ?: 0)).inv()
                .times(_animatorValue)
                .let { _offset ->
                    mBinding.llTitle.translationY = _offset
                    mBinding.flSignal.translationY = _offset
                    mBinding.flInvite.translationY = _offset
                    mBinding.flBugReport.translationY = _offset
                    mBinding.flDumpProgressBar.translationY = _offset
                    mBinding.flSetting.translationY = _offset
                }
            mBinding.llTitle.height.inv().times(_animatorValue)
                .let { _offset ->
                    mBinding.statusPop.translationY = _offset
                }
            mBinding.ivTopShadow.height.inv().times(_animatorValue)
                .let { _offset ->
                    mBinding.ivTopShadow.translationY = _offset
                }
            (mBinding.bottomBar.height + mBinding.bottomBar.marginBottom)
                .times(_animatorValue)
                .let { _offset ->
                    mBinding.bottomBar.translationY = _offset
                    mBinding.clMainScreenInfo.translationY = _offset
                    if (isScreenPortrait.invoke()) {
                        mBinding.cdlSmallStream.translationY = _offset
                    }
                }
        }
    }

    fun onDestroy() {
        activity = null
        fragment = null
    }

    companion object {
        private const val TAG = "[UI][MeetingFullScreenHelper]"
    }
}