package io.agora.avc.app.meeting

import android.app.Application
import android.view.View
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.viewModelScope
import io.agora.avc.app.developer.DeveloperOptions
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.*
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.extensions.copy
import io.agora.avc.manager.audiodump.AudioDumper
import io.agora.avc.manager.notice.NoticeEvent
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.manager.notice.NoticeType
import io.agora.avc.manager.notice.notification.NotificationManager
import io.agora.avc.manager.rating.video.VideoRatingManager
import io.agora.avc.repository.*
import io.agora.avc.screenshare.ScreenShareAction
import io.agora.avc.utils.TimeUtils
import io.agora.avc.video.ShareStatus
import io.agora.frame.base.livedata.EventLiveData
import io.agora.frame.base.livedata.SingleLiveEvent
import io.agora.logger.Logger
import io.agora.rtc.video.VideoCanvas
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.concurrent.TimeUnit
import javax.inject.Inject

class MeetingViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val roomRepository: RoomRepository,
    private val chatRepository: ChatRepository,
    private val videoRatingManager: VideoRatingManager,
    private val audioDumper: AudioDumper,
    private val developerRepository: DeveloperRepository,
    private val noticeManager: NoticeManager,
    private val appConfigRepository: AppConfigRepository,
    private val notificationRepository: NotificationRepository,
    private val notificationManager: NotificationManager,
) : AppViewModel(application, appController), ScreenShareAction {
    val attendeesLiveData = MutableLiveData<List<ARoomUser>>()
    val localUserStatusChanged = MutableLiveData<LocalUser>()
    val roomInfoChanged = EventLiveData<Room>()
    val meetingDurationEvent: SingleLiveEvent<String> = SingleLiveEvent()
    val mainUserUpdateEvent = EventLiveData<ARoomUser>()
    val unReadMessageChangedLiveData = MutableLiveData<Int>()
    val userAmountChangedLiveData = MutableLiveData<Int>()
    val audioDumpingChangedLiveData = MutableLiveData<Boolean>()
    val cloudRecordInfoChangedLiveData = MutableLiveData<CloudRecordInfo?>()

    val videoRatingTimingLiveData = EventLiveData<Long>()
    val screenShareLiveData = EventLiveData<ShareInfo>()
    val bizConnectChangedLiveData = EventLiveData<Boolean>()
    val roomMediaAlignEvent = EventLiveData<Int>()
    val kickOutEvent = EventLiveData<Boolean>()
    val audioDumpTimer = MutableLiveData<Int>()
    val audioDumpStatusChangedEvent = EventLiveData<Int>()
    val networkSignalChangedLiveData = MutableLiveData<NetworkSignal>()
    val appInBackgroundNotificationChangedEvent = EventLiveData<MeetingNotification>()
    val popupLiveData = EventLiveData<MessageEvent>()
    val videoRatingChangedLiveData = MutableLiveData<Boolean>()
    val developerChangedLiveData = EventLiveData<DeveloperOptions>()
    val screenShareChangedLiveData = MutableLiveData<Boolean>()
    val notificationLiveData = MutableLiveData<List<MeetingNotification>>()

    private var countTimer: Disposable? = null
    private var mainScreenUser: ARoomUser? = null
    private var cloudRecordInfo: CloudRecordInfo? = null
    private var shareScreenSeq: Int = 0
    private var useWatermark: Boolean = false

    override fun onCreate() {
        super.onCreate()
        notifyBizConnectChanged()
        queryMajorMediaUser()
        queryAttendees()
        queryLocalUser()
        queryRoomInfo()
        queryIssueDumping()
        queryCloudRecordingInfo()
        queryUserAmount()
        queryChatMessage()
        queryDevelopOption()
        queryNotification()
    }

    private fun queryRoomInfo() {
        getRoom()?.let { roomInfo ->
            roomInfoChanged.postValue(roomInfo)
        }
    }

    private fun notifyBizConnectChanged() {
        bizConnectChangedLiveData.postValue(appController.bizConnected())
    }

    private fun queryAttendees() {
        viewModelScope.launch(Dispatchers.IO) {
            attendeesLiveData.postValue(roomRepository.getMediaList().copy())
        }
    }

    private fun queryMajorMediaUser() {
        roomRepository.getMajorMedia()?.let { user ->
            mainUserUpdateEvent.postValue(user)
        }
    }

    private fun queryChatMessage() {
        Observable
            .create<Int> {
                val messages = chatRepository.getMessages()
                var index = messages.size
                var unreadCount = 0
                while (index > 0) {
                    if (messages.get(index - 1)?.isRead == true) {
                        break
                    }
                    ++unreadCount
                    index--
                }

                it.onNext(unreadCount)
            }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : io.reactivex.Observer<Int> {
                override fun onComplete() {

                }

                override fun onSubscribe(d: Disposable) {

                }

                override fun onNext(t: Int) {
                    unReadMessageChangedLiveData.value = t
                }

                override fun onError(e: Throwable) {

                }
            })
    }

    private fun queryDevelopOption() {
        developerChangedLiveData.postValue(developerRepository.getDeveloperOptions())
    }

    private fun queryNotification() {
        notificationLiveData.postValue(notificationRepository.getNotificationList())
    }

    fun switchUserSequence(user: ARoomUser) {
        Logger.i(TAG, "switch user:${user.streamId} from recycler to main screen")
        appController.setMajorUser(user.streamId)
    }

    private fun queryUserAmount() {
        Observable
            .create<Int> {
                var count = 0
                roomRepository.getRoomUserList().forEach { user ->
                    if (!user.isShareStream() && user.isAttendee) {
                        count++
                    }
                }
                it.onNext(count)
            }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : io.reactivex.Observer<Int> {
                override fun onComplete() {

                }

                override fun onSubscribe(d: Disposable) {

                }

                override fun onNext(t: Int) {
                    userAmountChangedLiveData.value = t
                }

                override fun onError(e: Throwable) {

                }
            })
    }

    fun startCountTime() {
        Logger.i(TAG, "start counting room elapse")
        Observable
            .interval(0, 1, TimeUnit.SECONDS)
            .subscribeOn(Schedulers.io())
            .subscribe(object : io.reactivex.Observer<Long> {
                override fun onComplete() {

                }

                override fun onSubscribe(d: Disposable) {
                    countTimer = d
                }

                override fun onNext(t: Long) {
                    val elapse = roomRepository.getRoom()?.elapse
                    val elapsedRealTime = TimeUtils.elapsedRealTime()
                    if (elapse != null && elapse > 0) {
                        val time = TimeUtils.getRoomTime(elapsedRealTime - elapse)
                        meetingDurationEvent.postValue(time)
                    }
                    cloudRecordInfo?.apply {
                        this.recordTime = TimeUtils.getRoomTime(elapsedRealTime - this.elapsedTime)
                        cloudRecordInfoChangedLiveData.postValue(this)
                    }
                }

                override fun onError(e: Throwable) {
                }
            })
    }

    fun stopCountTime() {
        Logger.w(TAG, "stop counting room elapse")
        countTimer?.dispose()
    }

    fun cleanMainScreenUser() {
        this.mainScreenUser = null
    }

    private fun queryIssueDumping() {
        viewModelScope.launch(Dispatchers.Main) {
            withContext(Dispatchers.IO) {
                val size = roomRepository.getAudioDumpingUser()?.size ?: 0
                audioDumpingChangedLiveData.postValue(size > 0)
            }
        }
    }

    private fun queryCloudRecordingInfo() {
        viewModelScope.launch(Dispatchers.Main) {
            withContext(Dispatchers.IO) {
                cloudRecordInfo = roomRepository.getCloudRecordInfo()
                cloudRecordInfo?.let {
                    val elapsedRealTime = TimeUtils.elapsedRealTime()
                    cloudRecordInfo?.recordTime =
                        TimeUtils.getRoomTime(elapsedRealTime - it.elapsedTime)
                }
                cloudRecordInfoChangedLiveData.postValue(cloudRecordInfo)
            }
        }
    }

    fun stopAudioDumping() {
        audioDumper.stopDumping()
    }

    fun clearRemoteVideo(streamId: Int) {
        appController.stopRenderVideo(streamId)
    }

    fun startAudioDumping() {
        audioDumper.startDumping()
    }

    fun clearLocalVideo() {
        getLocalUser()?.streamId?.let {
            appController.stopRenderVideo(it)
        }
    }

    override fun onCleared() {
        super.onCleared()
    }

    fun createRemoteVideo(uid: Int, renderMode: Int = VideoCanvas.RENDER_MODE_FIT): View? {
        val view = appController.createTextureView(getApplication())
        appController.startRenderVideo(uid, view, renderMode)
        return view
    }

    fun createLocalVideo(streamId: Int): View? {
        val view = appController.createTextureView(getApplication())
        appController.startRenderVideo(streamId, view, VideoCanvas.RENDER_MODE_HIDDEN)
        return view
    }

    fun setBackgroundColor(streamId: Int, color: Int) {
        if (developerRepository.isChangeMainScreenVideoBG()) {
            appController.setBackgroundColor(streamId, color)
        }
    }

    fun changeLocalVideoStatus(on: Boolean) {
        appController.setLocalVideo(on)
    }

    fun changeLocalAudioStatus(on: Boolean) {
        appController.setLocalAudio(on)
    }

    fun switchCamera() {
        appController.switchCamera()
    }

    fun leaveChannel() {
        appController.leaveRoom()
    }

    fun startCloudRecording() {
        appController.startCloudRecording()
    }

    fun stopCloudRecording() {
        appController.stopCloudRecording()
    }

    fun toggleVideoRating() {
        videoRatingManager.toggleVideoRating()
    }

    fun tryNetworkPromptAction() {
        appController.tryNetworkPromptAction()
    }

    fun closeNetworkPrompt() {
        appController.closeNetworkPrompt()
    }

    override fun stopScreenShare() {
        appController.cancelScreenShare()
    }

    override fun requestScreenShare(useWatermark: Boolean) {
        this.useWatermark = useWatermark
        appController.applyScreenShare(++shareScreenSeq, useWatermark)
    }

    override fun notifyServiceDisConnection() {
        noticeManager.notice(
            NoticeEvent(
                NoticeCode.CODE_SCREEN_SHARING_INTERRUPTED,
                NoticeType.DIALOG
            )
        )
    }

    override fun notifyScreenStatusChanged(status: Int) {
        screenShareChangedLiveData.postValue(status == ShareStatus.RUNNING.value)
    }

    fun acceptAssistantRequest(tag: String) {
        appController.acceptAssistantRequest(tag)
    }

    fun refuseAssistantRequest(tag: String) {
        appController.refuseAssistantRequest(tag)
    }

    fun enableAssist(originSoundEnable: Boolean) {
        appController.enableAssist(originSoundEnable)
        noticeManager.notice(
            NoticeEvent(
                NoticeCode.CODE_AS_ROOM_ANSWER_USERS_YES,
                NoticeType.NOTIFICATION_A,
                MeetingNotificationType.NO_8,
            )
        )
    }

    fun disableAssist() {
        appController.disableAssist()
        noticeManager.notice(
            NoticeEvent(
                NoticeCode.CODE_AS_ROOM_ANSWER_USERS_NO,
                NoticeType.NOTIFICATION_A,
                MeetingNotificationType.NO_8,
            )
        )
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.USER_INFO_CHANGED.ordinal -> {
                queryAttendees()
                queryLocalUser()
            }
            AppEvent.USER_LIST_CHANGED.ordinal -> {
                queryAttendees()
                queryUserAmount()
            }
            AppEvent.BIZ_STATUS_CHANGED.ordinal -> {
                notifyBizConnectChanged()
            }
            AppEvent.SCREEN_SHARE_EVENT.ordinal -> {
                (arg.obj as? ShareInfo)?.let {
                    if (it.seq == shareScreenSeq) {
                        it.hasWatermark = useWatermark
                        screenShareLiveData.postValue(it)
                    } else {
                        Logger.e(TAG, "discard invalid screen sharing requests")
                    }
                }
            }
            AppEvent.CHAT_MESSAGE_ADD.ordinal -> {
                queryChatMessage()
            }
            AppEvent.CHAT_MESSAGE_UPDATE.ordinal -> {
                queryChatMessage()
            }
            AppEvent.CHAT_MESSAGE_READ.ordinal -> {
                queryChatMessage()
            }
            AppEvent.MAJOR_USER_CHANGED.ordinal -> {
                queryMajorMediaUser()
                queryAttendees()
                queryLocalUser()
            }
            AppEvent.LEAVE_ROOM_EVENT.ordinal -> {
                kickOutEvent.postValue(true)
            }
            AppEvent.AUDIO_DUMPING_STATUS_CHANGE.ordinal -> {
                val data = arg.obj
                if (data is Issue) {
                    audioDumpStatusChangedEvent.postValue(data.dumpStatus)
                }
            }
            AppEvent.AUDIO_DUMPING_TIME.ordinal -> {
                (arg.obj as? Int)?.let {
                    audioDumpTimer.postValue(it)
                }
            }
            AppEvent.VIDEO_RATING_START.ordinal -> {
                videoRatingChangedLiveData.postValue(videoRatingManager.videoScore != null)
            }
            AppEvent.VIDEO_RATING_FINISH.ordinal -> {
                videoRatingChangedLiveData.postValue(false)
            }
            AppEvent.VIDEO_RATING_TIME.ordinal -> {
                (arg.obj as? Long)?.let {
                    videoRatingTimingLiveData.postValue(it)
                }
            }
            AppEvent.ROOM_NETWORK_STATUS_CHANGED.ordinal -> {
                (arg.obj as? NetworkSignal)?.let {
                    networkSignalChangedLiveData.postValue(it)
                }
            }
            AppEvent.CLOUD_RECORDING_STATUS_CHANGED.ordinal -> {
                queryCloudRecordingInfo()
            }
            AppEvent.ROOM_DUMP_STATUS_CHANGED.ordinal -> {
                queryIssueDumping()
            }
            AppEvent.NOTIFY_EVENT_APP_IN_BACKGROUND.ordinal -> {
                (arg.obj as? MeetingNotification)?.let {
                    appInBackgroundNotificationChangedEvent.postValue(it)
                }
            }
            AppEvent.LOCAL_USER_CHANGED.ordinal -> {
                queryLocalUser()
            }
            AppEvent.POPUP_EVENT.ordinal -> {
                popupLiveData.postValue(arg)
            }
            AppEvent.DEVELOPER_OPTIONS_EVENT.ordinal -> {
                arg.obj.let {
                    if (it is DeveloperOptions) {
                        developerChangedLiveData.postValue(it)
                    }
                }
            }
            AppEvent.ROOM_INFO_CHANGED.ordinal -> {
                queryRoomInfo()
            }
            AppEvent.APPLY_ASSISTANT_OPERATION_RESULT.ordinal,
            AppEvent.ROOM_INFO_CHANGED.ordinal,
            AppEvent.NOTIFY_EVENT.ordinal -> {
                notificationManager.onEventReceived(arg)
            }
            AppEvent.NOTIFICATION_CHANGED_EVENT.ordinal -> {
                notificationLiveData.postValue(notificationRepository.getNotificationList())
            }
        }
    }

    private fun queryLocalUser() {
        getLocalUser()?.let { user ->
            localUserStatusChanged.postValue(user)
        }
    }

    fun pipModeEnable(): Boolean {
        return appConfigRepository.getPictureMode()
    }

    fun getDeveloperOptions(): DeveloperOptions {
        return developerRepository.getDeveloperOptions()
    }

    fun hasPermission(operationUser: ARoomUser): Boolean {
        val hostUid = getRoom()?.hostUid
        val uid = getLocalUser()?.uid
        return !operationUser.isShareStream()
                && operationUser.online
                && (operationUser.uid == uid
                || hostUid == null
                || hostUid?.isEmpty()
                || hostUid == uid
                || operationUser.isAssistant)
    }

    fun getEnableWaterMark(): Boolean? {
        return appConfigRepository.getAppConfig()?.app?.enableWatermark ?: false
    }

    fun removeNotification(position: Int) {
        notificationManager.removeAt(position)
    }

    fun removeNotification(code: NoticeCode) {
        notificationManager.remove(code)
    }

    fun hasScreenSharePermission() =
        appConfigRepository.getAppConfig()?.app?.enableSS ?: false ||
                getLocalUser()?.isThirdPartyLoggedIn == true || getRoom()?.mode == RoomMode.AGORA.value

    fun notifyNotScreenSharePermission() {
        noticeManager.notice(
            NoticeEvent(
                code = NoticeCode.CODE_FRAUD_SCREEN_SHARING,
                NoticeType.NOTIFICATION_C,
                MeetingNotificationType.NO_3
            )
        )
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.USER_LIST_CHANGED,
            AppEvent.USER_INFO_CHANGED,
            AppEvent.BIZ_STATUS_CHANGED,
            AppEvent.SCREEN_SHARE_EVENT,
            AppEvent.CHAT_MESSAGE_ADD,
            AppEvent.CHAT_MESSAGE_UPDATE,
            AppEvent.CHAT_MESSAGE_READ,
            AppEvent.MAJOR_USER_CHANGED,
            AppEvent.NOTIFY_EVENT_APP_IN_BACKGROUND,
            AppEvent.LEAVE_ROOM_EVENT,
            AppEvent.AUDIO_DUMPING_STATUS_CHANGE,
            AppEvent.AUDIO_DUMPING_TIME,
            AppEvent.VIDEO_RATING_FINISH,
            AppEvent.VIDEO_RATING_TIME,
            AppEvent.ROOM_NETWORK_STATUS_CHANGED,
            AppEvent.CLOUD_RECORDING_STATUS_CHANGED,
            AppEvent.ROOM_DUMP_STATUS_CHANGED,
            AppEvent.LOCAL_USER_CHANGED,
            AppEvent.POPUP_EVENT,
            AppEvent.DEVELOPER_OPTIONS_EVENT,
            AppEvent.ROOM_INFO_CHANGED,
            AppEvent.NOTIFY_EVENT,
            AppEvent.APPLY_ASSISTANT_OPERATION_RESULT,
            AppEvent.ROOM_INFO_CHANGED,
            AppEvent.NOTIFICATION_CHANGED_EVENT,
        )
    }

    companion object {
        private const val TAG = "[VM][Meeting]"
    }
}