package io.agora.avc.app.mine

import android.content.Intent
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.view.View
import android.view.ViewGroup
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatDelegate
import androidx.lifecycle.Observer
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.R
import io.agora.avc.app.master.AgoraActivity
import io.agora.avc.app.master.MainViewModel
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.Version
import io.agora.avc.databinding.AgoraFragmentMineBinding
import io.agora.avc.extensions.getThirdPartyNickName
import io.agora.avc.extensions.safeNavigate
import io.agora.avc.listener.DeveloperClickedListener
import io.agora.avc.utils.DeviceUtils
import io.agora.avc.utils.ThemeUtils
import io.agora.avc.utils.ToastUtils
import io.agora.avc.widget.COMPANY_AGORA
import io.agora.avc.widget.LoadingStatusTextButton
import io.agora.avc.widget.PrivacyTermsDialog
import io.agora.avc.widget.ResolutionSetting
import io.agora.frame.base.NovelFragment
import io.agora.frame.base.livedata.StatusEvent
import io.agora.logger.LogConverter
import io.agora.logger.Logger

class MineFragment : NovelFragment<MineViewModel, AgoraFragmentMineBinding>() {

    private var localUser: LocalUser? = null

    private var exitConfirmDialog: AlertDialog? = null

    private val privacyTermsDialog by lazy { PrivacyTermsDialog(requireContext()) }

    private val mainViewModel: MainViewModel by lazy {
        getViewModel(requireActivity().viewModelStore, MainViewModel::class.java)
    }

    private val developerClickedListener = object : DeveloperClickedListener(false) {
        override fun onDeveloperClicked(view: View?) {
            mViewModel?.saveDeveloperVisible(true)
        }
    }

    override fun getLayoutId(): Int {
        return R.layout.agora_fragment_mine
    }

    override fun allocObserver() {
        mainViewModel.roomModeChangedLiveData.observe(this) {
            val act = activity
            if (act is AgoraActivity) {
                act.setRoomTheme(it)
            }
            val viewGroup = view as ViewGroup
            viewGroup.removeAllViews()
            val layout = onCreateView(layoutInflater, viewGroup, null)
            viewGroup.addView(layout)
            mViewModel?.onResume()
        }

        mainViewModel.loginWeWorkEvent?.observe(this, {
            safeNavigate(R.id.action_mine_to_loginOperation)
        })

        mViewModel?.userNameLiveData?.observe(this, {
            if (it != null && it.not()) {
                ToastUtils.showShort(getString(R.string.nick_modify_fail))
            }
        })

        mViewModel?.uploadLogLiveData?.observe(this, {
            Logger.i(TAG, "upload log event, success:$it")
            if (it) {
                mBinding.btnUploadLog.state = LoadingStatusTextButton.State.SUCCESS
            } else {
                mBinding.btnUploadLog.state = LoadingStatusTextButton.State.FAIL
            }
        })

        mViewModel?.statusEvent?.observe(this, Observer {
            when (it) {
                StatusEvent.Status.LOADING -> {
                    mBinding.btnUploadLog.state = LoadingStatusTextButton.State.LOADING
                }
            }
        })

        mViewModel?.renderingEvent?.observe(this) { model ->
            this.localUser = model.user
            renderUI(model)
        }

        mViewModel?.versionTagEvent?.observe(this, Observer { version ->
            setVersionText(version)
        })

    }

    private fun renderUI(model: MineVo) {
        setAccount(model.user)

        when (model.theme) {
            -1 -> {
                mBinding.darkTheme.setSwitchChecked(ThemeUtils.isNightMode(context))
            }
            AppCompatDelegate.MODE_NIGHT_YES -> {
                mBinding.darkTheme.setSwitchChecked(true)
            }
            else -> {
                mBinding.darkTheme.setSwitchChecked(false)
            }
        }

        mBinding.resolution.setResolution(model.resolution)

        developerClickedListener.showDeveloper = model.developerOptions.showDeveloper

        refreshDeveloperMenu(model.developerOptions.showDeveloper)

        if (DeviceUtils.isSupportPicInPic()) {
            mBinding.pictureInPicture.visibility = View.VISIBLE
            mBinding.pictureInPicture.setSwitchChecked(model.picMode)
            mBinding.pictureInPicture.callback = {
                it?.apply {
                    Logger.i(TAG, "set picture in picture mode ${LogConverter.enable(it)}")
                    mViewModel?.setPictureMode(it)
                }
            }
        }

    }

    override fun initialize(savedInstanceState: Bundle?) {

        mBinding.resolution.setOnResolutionChangedListener(object :
            ResolutionSetting.OnResolutionChangedListener {
            override fun onResolutionChanged(position: Int) {
                Logger.i(TAG, "I chose the resolution:$position")
                mViewModel?.onResolutionChanged(position)
            }
        })

        mBinding.darkTheme.callback = {
            it?.apply {
                mViewModel?.changeTheme(it)
                // Android P and later can automatically change theme
                if (Build.VERSION.SDK_INT < Build.VERSION_CODES.P) {
                    activity?.recreate()
                }
            }
        }

        mBinding.uploadLog.callback = {
            Logger.i(TAG, "I clicked the upload log button, state:${mBinding.btnUploadLog.state}")
            if (mBinding.btnUploadLog.canUpload()) {
                mViewModel?.uploadLog(DeviceUtils.getPlatform(), DeviceUtils.getModel())
            }
        }

        /******** Developer Menu *******/
        mBinding.tvAboutUs.setOnClickListener(developerClickedListener)

        privacyTermsDialog.setPrivacyTermsDialogListener(object :
            PrivacyTermsDialog.OnPrivacyTermsDialogListener {
            override fun onConfirmClick() {

            }

            override fun onCancelClick() {
                activity?.finish()
            }
        })

        mBinding.terms.callback = {
            jumpWebView(getString(R.string.terms_url))
        }

        mBinding.policy.callback = {
            jumpWebView(getString(R.string.policy_url))
        }

        mBinding.version.callback = {
            if (mBinding.version.tag is Int) {
                mViewModel?.versionTagClicked(mBinding.version.tag as Int)
            }
        }

        mBinding.itemAccount.callback = {
            if (localUser?.isThirdPartyLoggedIn != true) {
                mainViewModel.loginWeWorkEvent.postValue(COMPANY_AGORA)
            } else {
                showExitConfirmDialog()
            }
        }

        mBinding.upgrade.callback = {
            mainViewModel.queryAppVer(true)
        }
    }

    private fun jumpWebView(url: String) {
        val uri = Uri.parse(url)
        val intent = Intent(Intent.ACTION_VIEW, uri)
        context?.startActivity(intent)
    }

    private fun showExitConfirmDialog() {
        exitConfirmDialog =
            MaterialAlertDialogBuilder(requireContext(), R.style.CustomMaterialAlertDialog)
                .setTitle(R.string.notice_log_out_title)
                .setCancelable(false)
                .setMessage(getString(R.string.notice_log_out_message))
                .setNegativeButton(R.string.notice_log_out_cancel) { _, _ ->
                    Logger.i(TAG, "user cancel log out")
                }
                .setPositiveButton(R.string.notice_log_out_confirm) { _, _ ->
                    Logger.i(TAG, "user confirm log out")
                    mViewModel?.logoutViaWeWork()
                }
                .show()
    }

    private fun setAccount(user: LocalUser?) {
        if (user?.isThirdPartyLoggedIn == true) {
            mBinding.itemAccount.setSecondaryText(
                user.getThirdPartyNickName() ?: getString(R.string.unknown_user)
            )
            mBinding.itemAccount.setSecondaryTextVisible(true)
            mBinding.itemAccount.setActionButtonText(getString(R.string.sign_out_butoon))
        } else {
            mBinding.itemAccount.setSecondaryText("")
            mBinding.itemAccount.setSecondaryTextVisible(false)
            mBinding.itemAccount.setActionButtonText(getString(R.string.go_login_button))
        }
    }

    private fun setVersionText(version: Version) {
        val text = when (version.tag) {
            1 -> {
                String.format("V%s", version.des)
            }
            2 -> {
                String.format("RTC %s", version.des)
            }
            else -> {
                String.format("RTM %s", version.des)
            }
        }
        mBinding.version.setActionButtonText(text)
        mBinding.version.tag = version.tag
    }

    private fun refreshDeveloperMenu(show: Boolean) {
        if (show) {
            mBinding.tvDeveloper.visibility = View.VISIBLE
            mBinding.mcvDeveloper.visibility = View.VISIBLE
            mBinding.developerMenu.visibility = View.VISIBLE
            mBinding.developerMenu.setOnClickListener {
                safeNavigate(R.id.action_mine_to_developer)
            }
        }
    }

    override fun onDestroyView() {
        super.onDestroyView()
        exitConfirmDialog?.let {
            if (it.isShowing) {
                it.dismiss()
            }
        }
    }

    companion object {
        private const val TAG = "[UI][Mine]"
    }
}
