package io.agora.avc.app.operation

import android.app.Application
import android.text.TextUtils
import androidx.lifecycle.MutableLiveData
import com.agora.valoran.Constants
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.AssistantInfo
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.Room
import io.agora.avc.bo.RoomMode
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.extensions.isMySelf
import io.agora.avc.repository.RoomRepository
import io.agora.avc.utils.GsonUtils
import io.agora.frame.base.livedata.EventLiveData
import io.agora.logger.Logger
import javax.inject.Inject

class OperationViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val appEvent: AppEventBus,
    private val roomRepository: RoomRepository
) : AppViewModel(application, appController) {

    val permissionChangedLiveData = MutableLiveData<Boolean>()
    val remoteUserChangedLiveData = MutableLiveData<ARoomUser?>()
    val roomInfoChangedLiveData = MutableLiveData<Room?>()
    val localAssistStateChangedLiveData = MutableLiveData<Int>()
    val assistantChangedLiveData = MutableLiveData<AssistantInfo?>()
    val closePageEvent = EventLiveData<Any?>()
    val attendeesCountChanged = MutableLiveData<Int>()

    private var operationUser: ARoomUser? = null
    private var hasHost: Boolean? = null

    override fun onCreate() {
        super.onCreate()
    }

    override fun onResume() {
        super.onResume()
        queryUser()
        queryRoom()
        queryPermission()
        queryLocalAssistState()
        queryAssistant()
        queryAttendeesCount()
    }

    private fun queryLocalAssistState() {
        localAssistStateChangedLiveData.postValue(roomRepository.queryAssistState())
    }

    private fun queryAssistant() {
        assistantChangedLiveData.postValue(roomRepository.getAssistantInfo())
    }

    private fun queryRoom() {
        hasHost = getRoom()?.hasHost()
        roomInfoChangedLiveData.postValue(getRoom())
    }

    private fun queryAttendeesCount() {
        var count = 0
        roomRepository.getRoomUserList().forEach { user ->
            if (!user.isShareStream() && user.isAttendee) {
                count++
            }
        }
        attendeesCountChanged.postValue(count)
    }

    private fun queryPermission() {
        val hostUid = getRoom()?.hostUid
        val uid = getLocalUser()?.uid
        val hasPermission =
            operationUser?.uid == uid //myself
                    || hostUid == uid //I'm host
                    || (TextUtils.isEmpty(hostUid) && operationUser?.isAssistant != true) //not host and user not assistant

        permissionChangedLiveData.postValue(hasPermission)
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.USER_INFO_CHANGED.ordinal -> {
                Logger.i(TAG, "receive user info changed event")
                queryUser()
            }
            AppEvent.USER_LIST_CHANGED.ordinal -> {
                Logger.i(TAG, "receive user list changed event")
                queryUser()
            }
            AppEvent.ROOM_INFO_CHANGED.ordinal -> {
                notifyClosePageIfNeed(
                    operationUser,
                    operationUser,
                    hasHost != getRoom()?.hasHost()
                )
                hasHost = getRoom()?.hasHost()
                queryPermission()
            }
            AppEvent.LOCAL_ASSISTANT_STATE_CHANGED.ordinal -> {
                queryLocalAssistState()
            }
            AppEvent.ROOM_ASSISTANT_CHANGED.ordinal -> {
                queryAssistant()
            }
            AppEvent.USER_LIST_CHANGED.ordinal -> {
                queryAttendeesCount()
            }
        }
    }

    private fun queryUser() {
        val userList = roomRepository.getRoomUserList()
        var findUser: ARoomUser? = null
        for (user in userList) {
            if (user.streamId == operationUser?.streamId) {
                findUser = user
                break
            }
        }

        notifyClosePageIfNeed(operationUser, findUser)

        operationUser = findUser
        remoteUserChangedLiveData.postValue(operationUser)
    }

    /**
     * notify close page when operation user become assistant or become not assistant,
     * or operation is assistant and he leaving
     */
    private fun notifyClosePageIfNeed(
        oldUser: ARoomUser?,
        newUser: ARoomUser?,
        hostChanged: Boolean = false
    ) {
        if (oldUser == null || newUser == null) {
            closePageEvent.postValue(null)
            return
        }

        if (newUser.isMySelf()) {
            return
        }

        if (oldUser.isAssistant != newUser.isAssistant
            || oldUser.online != newUser.online
            || (hostChanged && !newUser.isAssistant)
        ) {
            closePageEvent.postValue(null)
        }
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.USER_LIST_CHANGED,
            AppEvent.USER_INFO_CHANGED,
            AppEvent.ROOM_INFO_CHANGED,
            AppEvent.LOCAL_ASSISTANT_STATE_CHANGED,
            AppEvent.ROOM_ASSISTANT_CHANGED,
            AppEvent.USER_LIST_COUNT_CHANGED,
        )
    }

    private fun getRoomMode(): RoomMode {
        return if (getRoom()?.isInternal() == true) {
            RoomMode.AGORA
        } else {
            RoomMode.NORMAL
        }
    }

    fun saveData(data: String) {
        val user = GsonUtils.fromJson(data, ARoomUser::class.java)
        val localUser = getLocalUser()
        if (user != null && user.streamId == localUser?.streamId) {
            this.operationUser = localUser
        } else {
            this.operationUser = user
        }
        remoteUserChangedLiveData.postValue(operationUser)
    }

    fun setUserAudio(user: ARoomUser, on: Boolean) {
        if (user.streamId == 0) {
            Logger.e(TAG, "Illegal audio operation, remote user streamId is null")
            return
        }
        if (user is LocalUser) {
            appController.setLocalAudio(on)
        } else {
            appController.setRemoteAudio(user.streamId, on)
        }
    }

    fun setUserVideo(user: ARoomUser, on: Boolean) {
        if (user.streamId == 0) {
            Logger.e(TAG, "Illegal video operation, remote user streamId is null")
            return
        }
        if (user is LocalUser) {
            appController.setLocalVideo(on)
        } else {
            appController.setRemoteVideo(user.streamId, on)
        }
    }

    fun kickOut(user: ARoomUser) {
        appController.kickRemote(user.streamId)
    }

    fun applyAssistant(type: Constants.TransLangType) {
        Logger.i(TAG, "applyAssistant request, type:${type.value}")
        appEvent.notifyObservers(MessageEvent(AppEvent.WAIT_ASSISTANT_RESULT.ordinal))
        appController.applyAssistant(type.value)
    }

    fun enableAssist(originSoundEnable: Boolean) {
        Logger.i(TAG, "enableAssist request, listenOriginSound${originSoundEnable}")
        appController.enableAssist(originSoundEnable)
    }

    fun disableAssist() {
        Logger.i(TAG, "disableAssist request")
        appController.disableAssist()
    }

    companion object {
        private const val TAG = "[VM][Operation]"
    }

}