package io.agora.avc.app.rating.audio

import android.os.Bundle
import android.widget.AdapterView
import android.widget.SimpleAdapter
import androidx.core.view.updateLayoutParams
import androidx.navigation.fragment.NavHostFragment
import com.google.android.material.bottomsheet.BottomSheetBehavior
import io.agora.avc.R
import io.agora.avc.app.master.MainViewModel
import io.agora.avc.bo.Accuracy
import io.agora.avc.bo.MosScore
import io.agora.avc.databinding.FragmentAudioRatingBinding
import io.agora.avc.utils.GsonUtils
import io.agora.avc.utils.ScreenUtils
import io.agora.avc.utils.StringUtils
import io.agora.avc.utils.ToastUtils
import io.agora.avc.widget.UploadingStatus
import io.agora.frame.base.BaseSheetDialogFragment
import io.agora.frame.base.livedata.StatusEvent
import io.agora.logger.Logger

class AudioRatingFragment :
    BaseSheetDialogFragment<AudioRatingViewModel, FragmentAudioRatingBinding>() {

    private var userListSelectedPosition = AdapterView.INVALID_POSITION

    private val mainViewModel: MainViewModel? by lazy {
        getViewModel(requireActivity().viewModelStore, MainViewModel::class.java)
    }

    private lateinit var mosScore: MosScore

    override fun getLayoutId() = R.layout.fragment_audio_rating

    override fun onStart() {
        super.onStart()
        mDialog.setCanceledOnTouchOutside(false)
        mDialog.setCancelable(true)
        mBinding.container.updateLayoutParams {
            height = (ScreenUtils.getAppScreenHeight() * 0.84f).toInt()
        }
        mBehavior.isHideable = true
        mBehavior.skipCollapsed = true
        mBehavior.state = BottomSheetBehavior.STATE_EXPANDED
    }

    override fun initialize(savedInstanceState: Bundle?) {
        if (arguments == null) {
            Logger.e(TAG, "arguments is null, close the fragment ")
            NavHostFragment.findNavController(this).navigateUp()
            return
        }

        arguments?.getString("data")?.let {
            try {
                mosScore = GsonUtils.fromJson(it, MosScore::class.java)
                mViewModel.mosScore = mosScore
            } catch (e: Exception) {
                Logger.e(TAG, "arguments error, error message:${e.stackTraceToString()}")
                NavHostFragment.findNavController(this).navigateUp()
            }
        }

        mBinding.btnSubmit.setOnClickListener {
            submit()
        }

        mBinding.lvUserList.setOnItemClickListener { _, _, position, _ ->
            Logger.i(TAG, "selected position of user list: position=$position")
            userListSelectedPosition = position
        }
    }

    override fun allocObserver() {
        mViewModel.reportResultEvent.observe(this) {
            if (it == UploadingStatus.COMPLETED) {
                NavHostFragment.findNavController(this).navigateUp()
            }
        }
        mViewModel.localScoreEvent.observe(this) {
            mBinding.tvScore.text = it
        }
        mViewModel.userListEvent.observe(this) {
            Logger.i(TAG, "user list event: size=${it.size}")
            mBinding.lvUserList.adapter = SimpleAdapter(
                requireContext(),
                it,
                R.layout.item_audio_scoring_user_list,
                arrayOf("serialNumber", "uid", "score"),
                intArrayOf(R.id.tvSerialNumber, R.id.tvNickName, R.id.tvScore),
            )
        }
        mViewModel.statusEvent.observe(this, StatusEvent.StatusObserver {
            if (it == StatusEvent.Status.ERROR) {
                ToastUtils.showShort(R.string.request_failed)
            }
        })
        mainViewModel?.bugReportRetryEvent?.observe(this) {
            submit()
        }
    }

    private fun submit() {
        mViewModel.submit(
            feedback = getFeedback(),
            selectedIndex = if (userListSelectedPosition == AdapterView.INVALID_POSITION) null else userListSelectedPosition,
            description = mBinding.edtDescription.text.toString().let {
                if (StringUtils.isTrimEmpty(it)) null else it
            },
        )
    }

    private fun getFeedback() =
        when (mBinding.cgAccuracy.checkedChipId) {
            R.id.chipAccuracyHigh -> {
                Accuracy.HIGH.value
            }
            R.id.chipAccuracyAccurate -> {
                Accuracy.ACCURACY.value
            }
            R.id.chipAccuracyLow -> {
                Accuracy.LOW.value
            }
            else -> {
                null
            }
        }

    companion object {
        private const val TAG = "[UI][AudioRating]"
    }
}